﻿using System;
using System.Collections.Concurrent;
using Curse.Friends.Data;
using Curse.Friends.ServerHosting;
using Curse.Logging;
using System.Threading;

namespace Curse.Friends.GiveawayService
{
    public class GiveawayServer : ServerHost<GroupGiveawayHost,GroupGiveaway>
    {
        private static readonly GiveawayServer _instance = new GiveawayServer();

        public static void StartServer()
        {
            _instance.Start();
        }

        public static void StopServer()
        {
            _instance.Stop();
        }

        private readonly ConcurrentDictionary<string, GiveawaySession> Sessions = new ConcurrentDictionary<string, GiveawaySession>();

        private GiveawayServer()
        {
        }

        protected override void CustomStartup()
        {
            // Queues and timers
            GroupGiveawayCoordinator.StartProcessor(ProcessGroupGiveawayCoordinator);
            MessageLock<RoleSyncCoordinator>.Initialize(r => r.IsEligible);
            RoleSyncCoordinator.StartProcessor(MessageLock<RoleSyncCoordinator>.ProcessRoleSyncCheck);

            AddTimer("UpdateEntries", TimeSpan.FromSeconds(5), UpdateEntries);
        }

        protected override void CustomStop()
        {
            try
            {
                foreach (var session in Sessions.Values)
                {
                    session.Shutdown();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error while shutting down Giveaway Sessions.");
            }
        }


        private readonly ConcurrentDictionary<string, object> _createSessionLocks = new ConcurrentDictionary<string, object>();

        private GiveawaySession GetOrAddSession(Guid groupID, int giveawayID, out bool retrievedFromCache)
        {
            var key = GetKey(groupID, giveawayID);

            GiveawaySession session;
            if (Sessions.TryGetValue(key, out session))
            {                
                retrievedFromCache = true;
                return session;
            }

            var createSession = _createSessionLocks.GetOrAdd(key, new object());
            var lockAcquired = false;
            Monitor.TryEnter(createSession, TimeSpan.FromSeconds(20), ref lockAcquired);

            try
            {
                if (!lockAcquired)
                {
                    throw new InvalidOperationException("Failed to acquire create session lock for giveaway: " + key);
                }

                lock (createSession)
                {
                    if (Sessions.TryGetValue(key, out session))
                    {                        
                        retrievedFromCache = true;
                        return session;
                    }

                    retrievedFromCache = false;
                    return Sessions.GetOrAdd(key, guid => new GiveawaySession(groupID, giveawayID));
                }
            }
            finally
            {
                if (lockAcquired)
                {
                    Monitor.Exit(createSession);
                }
            }

        }

        private void UpdateEntries()
        {
            foreach (var session in Sessions.Values)
            {
                session.SaveAndNotify();
            }
        }


        private void ProcessGroupGiveawayCoordinator(GroupGiveawayCoordinator coordinator)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(coordinator.GroupID, coordinator.GiveawayID, out retrievedFromCache);

                switch (coordinator.ChangeType)
                {
                    case GroupGiveawayCoordinatorType.Start:
                        session.Start(coordinator.Roles);
                        break;
                    case GroupGiveawayCoordinatorType.End:
                        session.End(coordinator.Requestor, false);
                        break;
                    case GroupGiveawayCoordinatorType.AddParticipant:
                        session.ParticipantAdded(coordinator.AffectedParticipant);
                        break;
                    case GroupGiveawayCoordinatorType.RemoveParticipant:
                        session.ParticipantRemoved(coordinator.AffectedParticipant);
                        break;
                    case GroupGiveawayCoordinatorType.Roll:
                        session.Roll(coordinator.Requestor);
                        break;
                    case GroupGiveawayCoordinatorType.Claim:
                        session.Claimed(coordinator.AffectedParticipant);
                        break;
                    case GroupGiveawayCoordinatorType.Continue:
                        session.Continue(coordinator.Requestor);
                        break;
                    case GroupGiveawayCoordinatorType.Deactivate:
                        session.End(coordinator.Requestor, true);
                        session.Shutdown();
                        Sessions.TryRemove(GetKey(coordinator.GroupID, coordinator.GiveawayID), out session);
                        break;
                    default:
                        throw new NotImplementedException("Unexpected GroupGiveawayChangeType");
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process group giveaway coordinator.", coordinator);
                throw;
            }
        }

        private static string GetKey(Guid groupID, int giveawayID)
        {
            return groupID.ToString() + giveawayID;
        }
    }
}
