﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Logging;

namespace Curse.Friends.GiveawayService
{
    public class GiveawaySession
    {
        private readonly LogCategory Logger;
        private readonly GroupGiveaway _giveaway;
        private readonly Group _group;
        private readonly GroupMember _creator;
        private volatile bool _notify;
        private readonly ConcurrentDictionary<int, GroupGiveawayParticipant> _participants = new ConcurrentDictionary<int, GroupGiveawayParticipant>();
        private readonly CancellationTokenSource _cts = new CancellationTokenSource();
        private readonly List<GiveawayRoll> _rolls = new List<GiveawayRoll>();
        private int _totalEntries;
        private GiveawayRoll _currentRoll;

        public Group Group
        {
            get { return _group; }
        }

        public GroupGiveaway Giveaway
        {
            get { return _giveaway; }
        }

        public GiveawaySession(Guid groupID, int giveawayID)
        {
            var sw = Stopwatch.StartNew();

            _group = Group.GetByID(groupID);

            if (_group == null)
            {
                throw new DataNotFoundException();
            }            

            _giveaway = _group.GetGiveaway(giveawayID);
            if (_giveaway == null)
            {
                throw new DataNotFoundException();
            }

            if (_giveaway.Status == GroupGiveawayStatus.Inactive)
            {
                throw new InvalidOperationException("Cannot start giveaway session for one that is inactive");
            }

            Logger = new LogCategory(_group.Title + " - " + _giveaway.Title) { ReleaseLevel = LogLevel.Debug }; 

            _creator = _group.GetMember(_giveaway.CreatorID);
            if (_creator == null)
            {
                throw new DataNotFoundException();
            }

            var participants = _giveaway.GetParticipants().Where(p => p.Status == GroupGiveawayParticipantStatus.Entered || p.Status == GroupGiveawayParticipantStatus.Winner);
            foreach (var participant in participants)
            {
                _participants.TryAdd(participant.UserID, participant);
            }

            var rolls = GroupEventManager.Search(new GroupEventSearch
            {
                GroupID = groupID,
                GiveawayID = giveawayID,
                PageSize = 50,
                PageNumber = 1,
                IncludedEventTypes = new[]{GroupEventType.GiveawayRoll},
            });

            _totalEntries = _giveaway.TotalEntries;

            _rolls.AddRange(rolls.Reverse().Select(r => new GiveawayRoll
            {
                UserID = r.GiveawayDetails.WinnerUserID,
                Username = r.GiveawayDetails.WinnerUsername,
                RollNumber = r.GiveawayDetails.RollNumber,
                Timestamp = r.Timestamp,
                RollStatus = r.GiveawayDetails.RollStatus ?? GroupGiveawayRollStatus.Invalid,
                BestRoleName = r.GiveawayDetails.WinnerBestRoleName,
                BestRoleID = r.GiveawayDetails.WinnerBestRoleID
            }));

            if (_giveaway.PendingWinnerUserID > 0)
            {
                var member = GroupMember.GetLocal(_group.GroupID, _giveaway.PendingWinnerUserID);
                _rolls.Add(new GiveawayRoll
                {
                    UserID = _giveaway.PendingWinnerUserID,
                    Username = member.GetTitleName(),
                    RollNumber = _giveaway.CurrentRoll,
                    RollStatus = GroupGiveawayRollStatus.Pending,
                    Timestamp = _giveaway.DateStatusChanged.ToEpochMilliseconds(),
                    BestRoleID = member.BestRole,
                    BestRoleName = _group.GetRole(member.BestRole, true).Name
                });
                ResponseWindowTimer.Start(_giveaway.GetLookupIndex(), DateTime.UtcNow - _giveaway.DateStatusChanged + _giveaway.ResponseWindow,
                    () => ClaimExpiredAction(member));
            }

            sw.Stop();

            Logger.Info("Started session in " + sw.Elapsed.TotalSeconds.ToString("F2") + " seconds");
        }

        public void Shutdown()
        {
            Logger.Info("Shutting down session");

            _cts.Cancel();
            _cts.Dispose();
            ResponseWindowTimer.Stop(_giveaway.GetLookupIndex());
            _giveaway.MachineName = string.Empty;
            _giveaway.Update(g => g.MachineName);
        }

        public void Start(HashSet<int> autoEnterRoles)
        {
            Logger.Info("Giveaway started!", new { autoEnterRoles });

            GroupEventManager.LogCreateGiveawayEvent(_creator, _giveaway);

            NotifyPublicMessage(GroupGiveawayChangeType.Started, _creator);

            if (_giveaway.AutoEnterActiveUsers || _giveaway.AutoEnterRoles.Count > 0)
            {
                var totalEntries = 0;

                Task.Delay(TimeSpan.FromSeconds(5), _cts.Token).ContinueWith(t =>
                {
                    var sw = Stopwatch.StartNew();

                    Logger.Info("Starting process to auto enter users...");

                    GroupMember.BatchOperateOnIndexLocal(g => g.GroupID, _giveaway.GroupID, 1000, membersEnumerable =>
                    {
                        if (_cts.IsCancellationRequested)
                        {
                            return;
                        }

                        var members = membersEnumerable.ToArray();

                        Logger.Info("Auto entering " + members.Length + " users...");
                        totalEntries += members.Length;

                        var userStats = UserStatistics.MultiGetLocal(members.Select(m => new KeyInfo(m.UserID))).ToDictionary(m => m.UserID);
                        var participants = GroupGiveawayParticipant.MultiGetLocal(members.Select(m => new KeyInfo(m.UserID, m.GroupID, _giveaway.GiveawayID))).ToDictionary(p => p.UserID);
                        foreach (var member in members)
                        {
                            if (_cts.IsCancellationRequested)
                            {
                                break;
                            }

                            try
                            {
                                if (_giveaway.IgnoredUsers.Contains(member.UserID))
                                {
                                    continue;
                                }

                                UserStatistics stats;
                                var effectiveRoles = new HashSet<int>(member.Roles) {_group.DefaultRoleID};
                                var isActive = userStats.TryGetValue(member.UserID, out stats) && stats.ConnectionStatus != UserConnectionStatus.Offline;
                                var isEnteredBasedOnRole = (isActive || _giveaway.IncludeOfflineMembers) && effectiveRoles.Overlaps(_giveaway.AutoEnterRoles);

                                // Legacy
                                var isEnteredBasedOnActive = _giveaway.AutoEnterActiveUsers && isActive;

                                if (isEnteredBasedOnActive || isEnteredBasedOnRole)
                                {
                                    GroupGiveawayParticipant participant;
                                    if (!participants.TryGetValue(member.UserID, out participant))
                                    {
                                        participant = new GroupGiveawayParticipant
                                        {
                                            GroupID = _giveaway.GroupID,
                                            UserID = member.UserID,
                                            Username = member.GetTitleName(),
                                            DateEntered = DateTime.UtcNow,
                                            Status = GroupGiveawayParticipantStatus.Entered,
                                            DateStatusChanged = DateTime.UtcNow,
                                            GiveawayID = _giveaway.GiveawayID,
                                            GroupAndGiveawayIdIndex = _giveaway.GetLookupIndex(),
                                            ReferrerUserID = 0
                                        };
                                        participant.InsertLocal();
                                        ParticipantAdded(participant);
                                    }
                                }
                            }
                            catch (Exception ex)
                            {
                                Logger.Warn(ex, "Error auto-adding user into giveaway.", new {Member = member, Giveaway = _giveaway});
                            }
                        }
                    });

                    sw.Stop();
                    Logger.Info("Completed auto entering " + totalEntries.ToString("F0") + " users in " + sw.Elapsed.TotalSeconds.ToString("F2") + " seconds");
                }, _cts.Token);
            }
        }

        public void End(GroupMember requestor, bool deactivate)
        {
            Logger.Info("Giveaway ended by " + requestor.GetTitleName(), new { deactivate });

            _cts.Cancel();
            ResponseWindowTimer.Stop(_giveaway.GetLookupIndex());

            Save();

            GroupEventManager.LogEndGiveawayEvent(requestor, _giveaway);

            if (deactivate)
            {
                _giveaway.Status = GroupGiveawayStatus.Inactive;
                _giveaway.DateStatusChanged = DateTime.UtcNow;
                _giveaway.Update(g => g.Status, g => g.DateStatusChanged);
                NotifyPublicMessage(GroupGiveawayChangeType.Removed, requestor);
            }
            else
            {
                _giveaway.Status = GroupGiveawayStatus.Ended;
                _giveaway.DateStatusChanged = DateTime.UtcNow;
                _giveaway.Update(g => g.Status, g => g.DateStatusChanged);
                NotifyPublicMessage(GroupGiveawayChangeType.Ended, requestor);
            }
        }

        public void Continue(GroupMember requestor)
        {
            if (_giveaway.Status != GroupGiveawayStatus.Claimed)
            {
                return;
            }

            Logger.Info("Giveaway continued by " + requestor.GetTitleName());

            _giveaway.Status = GroupGiveawayStatus.Active;
            _giveaway.DateStatusChanged = DateTime.UtcNow;
            _giveaway.FakeRollsLeft = _giveaway.FakeRollsBeforeWinner;
            _giveaway.Update(g => g.Status, g => g.DateStatusChanged, g=>g.FakeRollsLeft);
            NotifyPublicMessage(GroupGiveawayChangeType.Started, requestor);
        }

        public void SaveAndNotify()
        {
            if (_notify)
            {
                _notify = false;

                _giveaway.Refresh();
                
                if (_giveaway.Status == GroupGiveawayStatus.Inactive)
                {
                    Logger.Warn("Giveaway has become inactive, but did not get cleared via the coordinator. It will no longer dispatch change events.");
                    return;
                }

                Save();                
                NotifyPublicMessage(GroupGiveawayChangeType.EntriesUpdated, _creator);
            }
        }

        private void Save()
        {
            _giveaway.EligibleEntries = _participants.Count;
            _giveaway.TotalEntries = _totalEntries;
            _giveaway.TopShares = GetShares(GroupGiveawayParticipant.GetAllLocal(g => g.GroupAndGiveawayIdIndex, _giveaway.GetLookupIndex()))
                .OrderBy(kvp => kvp.Value)
                .Take(3)
                .ToDictionary(kvp => kvp.Key, kvp => kvp.Value);
            _giveaway.Update(g => g.EligibleEntries, g=>g.TotalEntries, g => g.TopShares);
        }

        public void ParticipantAdded(GroupGiveawayParticipant participant)
        {
            if (_giveaway.Status == GroupGiveawayStatus.Ended)
            {
                return;
            }

            if (_participants.TryAdd(participant.UserID, participant))
            {
                _notify = true;

                _giveaway.EligibleEntries = _participants.Count;
                Interlocked.Increment(ref _totalEntries);

                var member = GroupMember.GetLocal(_group.GroupID, participant.UserID);
                NotifyTargetedMessage(GroupGiveawayChangeType.ParticipantAdded, member, member, participant.DateEntered);
            }
        }

        public void ParticipantRemoved(GroupGiveawayParticipant participant)
        {
            if (_giveaway.Status == GroupGiveawayStatus.Ended)
            {
                return;
            }

            GroupGiveawayParticipant localParticipant;
            if (_participants.TryRemove(participant.UserID, out localParticipant))
            {
                _notify = true;
                _giveaway.EligibleEntries = _participants.Count;
                Interlocked.Decrement(ref _totalEntries);

                var member = GroupMember.GetLocal(_group.GroupID, participant.UserID);
                NotifyTargetedMessage(GroupGiveawayChangeType.ParticipantRemoved, member, member, participant.DateStatusChanged);
            }
        }

        public void Claimed(GroupGiveawayParticipant participant)
        {
            if(_giveaway.PendingWinnerUserID<1 || participant.UserID!=_giveaway.PendingWinnerUserID)
            {
                Logger.Warn("Claim called without a winner or by the wrong participant.");
                return;
            }

            Logger.Info("Prize being claimed by " + participant.Username);

            var success = ResponseWindowTimer.Stop(_giveaway.GetLookupIndex());

            if(!success)
            {
                // Get a fresh participant to see if the claim is actually valid
                participant = GroupGiveawayParticipant.GetLocal(participant.UserID, participant.GroupID, participant.GiveawayID);
                success = participant.Status == GroupGiveawayParticipantStatus.Claimed;
            }

            var member = GroupMember.GetLocal(_group.GroupID, participant.UserID);

            GroupGiveawayParticipant throwaway;
            if (!success)
            {
                _participants.TryRemove(participant.UserID, out throwaway);

                UpdateParticipantStatus(participant.UserID, GroupGiveawayParticipantStatus.Ineligible, participant);

                _giveaway.Status = GroupGiveawayStatus.Active;
                _giveaway.PendingWinnerUserID = 0;
                _giveaway.DateStatusChanged = DateTime.UtcNow;
                _giveaway.EligibleEntries = _participants.Count;
                _giveaway.Update(g => g.Status, g => g.DateStatusChanged, g => g.PendingWinnerUserID, g => g.EligibleEntries);
                _rolls.Last().RollStatus = GroupGiveawayRollStatus.ClaimExpired;
                GroupEventManager.LogRollGiveawayEvent(_creator, _giveaway, member, GroupGiveawayRollStatus.ClaimExpired, _group.GetRole(member.BestRole, true));
                NotifyPublicMessage(GroupGiveawayChangeType.ClaimExpired, member, member);
                return;
            }

            if (_giveaway.AllowRepeatWinners)
            {
                UpdateParticipantStatus(participant.UserID, GroupGiveawayParticipantStatus.Entered, participant);
            }
            else
            {
                _participants.TryRemove(participant.UserID, out throwaway);
            }

            GroupEventManager.LogRollGiveawayEvent(_creator, _giveaway, member, GroupGiveawayRollStatus.Claimed, _group.GetRole(member.BestRole, true));

            _rolls.Last().RollStatus = GroupGiveawayRollStatus.Claimed;

            _giveaway.PendingWinnerUserID = 0;
            _giveaway.Status = GroupGiveawayStatus.Claimed;
            _giveaway.DateStatusChanged = DateTime.UtcNow;
            _giveaway.EligibleEntries = _participants.Count;
            _giveaway.Update(g => g.Status, g => g.DateStatusChanged, g => g.PendingWinnerUserID, g => g.EligibleEntries);

            NotifyPublicMessage(GroupGiveawayChangeType.PrizeClaimed, _creator, member, participant.DateStatusChanged);
        }

        private void UpdateParticipantStatus(int userID, GroupGiveawayParticipantStatus status, GroupGiveawayParticipant backup)
        {
            GroupGiveawayParticipant participant;
            if (!_participants.TryGetValue(userID, out participant))
            {
                participant = backup;
            }

            participant.Status = status;
            participant.DateStatusChanged = DateTime.UtcNow;
            participant.Update(p => p.Status, p => p.DateStatusChanged);
        }

        private Dictionary<int, int> GetShares(IEnumerable<GroupGiveawayParticipant> participants)
        {
            return participants.GroupBy(p => p.ReferrerUserID).Where(g => g.Key > 0).ToDictionary(g => g.Key, g => g.Count());
        }

        private GiveawayWinner SelectWinner(Dictionary<int,GroupGiveawayParticipant> participants, Dictionary<int,GroupMember> groupMembers, bool isFakeRoll)
        {
            GiveawayWinner winner = null;

            var attempt = 1;
            while (attempt++ < 20)
            {
                var winnerID = RollForWinner(participants, groupMembers);
                if (winnerID <= 0)
                {
                    continue;
                }

                var member = groupMembers[winnerID];
                var participant = participants[winnerID];

                winner = new GiveawayWinner
                {
                    UserID = winnerID,
                    Member = member,
                    Participant = participant,
                    ValidStatus = GroupGiveawayWinnerValidStatus.Valid
                };

                if (isFakeRoll)
                {
                    winner.ValidStatus = GroupGiveawayWinnerValidStatus.FakeRoll;
                    break;
                }

                if (member.IsDeleted)
                {
                    winner.ValidStatus = GroupGiveawayWinnerValidStatus.NoMembership;
                    break;
                }

                var stats = UserStatistics.GetLocal(winnerID);
                var isOnline = stats != null && stats.ConnectionStatus != UserConnectionStatus.Offline;
                if (_giveaway.IncludeOfflineMembers || isOnline)
                {
                    break;
                }

                winner.ValidStatus = GroupGiveawayWinnerValidStatus.Offline;
            }
            if (winner == null)
            {
                return null;
            }

            if (winner.ValidStatus == GroupGiveawayWinnerValidStatus.Valid && !ValidateWinner(winner.Member))
            {
                winner.ValidStatus = GroupGiveawayWinnerValidStatus.MissingRole;
            }
            return winner;
        }

        private int RollForWinner(Dictionary<int, GroupGiveawayParticipant> participants, Dictionary<int, GroupMember> groupMembers)
        {
            // Precalculate important information outside of loop
            var sortedRoleBonuses = _giveaway.RoleBonuses.OrderByDescending(rbKvp => rbKvp.Value);
            var shares = GetShares(participants.Values);

            var cumulative = 0.0;
            var cumulativeDictionary = new Dictionary<int, double>();
            foreach (var participant in participants)
            {
                GroupMember member;
                if (!groupMembers.TryGetValue(participant.Key, out member))
                {
                    Logger.Warn("Giveaway participant was not a member of the group!", new { _giveaway, participant.Value });
                    continue;
                }

                var entryValue = 1.0;

                if (sortedRoleBonuses.Any() && member.Roles.Overlaps(_giveaway.RoleBonuses.Keys))
                {
                    foreach (var rbKvp in sortedRoleBonuses)
                    {
                        if (member.Roles.Contains(rbKvp.Key))
                        {
                            entryValue += rbKvp.Value;
                            break;
                        }
                    }
                }

                int shareCount;
                if (_giveaway.SharingBonus > 0 && shares.TryGetValue(participant.Key, out shareCount))
                {
                    entryValue += Math.Min(1.0, _giveaway.SharingBonus / 100.0 * shareCount);
                }

                cumulativeDictionary[participant.Key] = cumulative;
                cumulative += entryValue;
            }

            if (cumulativeDictionary.Count == 0)
            {
                return -1;
            }

            var roll = new Random().NextDouble() * cumulative;
            return cumulativeDictionary.Reverse().First(kvp => kvp.Value <= roll).Key;
        }

        private bool ValidateWinner(GroupMember winnerMembership)
        {
            if (_giveaway.RequiredRoles.Any() && !winnerMembership.Roles.Overlaps(_giveaway.RequiredRoles))
            {
                // Check with external community services to do a forced check to reduce false negatives.
                var rolesToCheck = GroupRole.MultiGetLocal(_giveaway.RequiredRoles.Select(r => new KeyInfo(_giveaway.GroupID, r))).Where(r => r.IsSynced && !string.IsNullOrWhiteSpace(r.SyncID)).ToArray();
                var requestID = Guid.NewGuid();
                using (var lockObject = MessageLock<RoleSyncCoordinator>.Create(requestID, rolesToCheck.Length))
                {
                    lock (lockObject)
                    {
                        var mappings = ExternalCommunityMapping.GetAllLocal(m => m.GroupID, _giveaway.GroupID).Where(m => !m.IsDeleted).ToArray();
                        foreach (var role in rolesToCheck)
                        {
                            var mapping = mappings.FirstOrDefault(c => c.ExternalID == role.SyncID && c.Type == role.Source);
                            if (mapping == null || mapping.Community == null)
                            {
                                // Can't perform a check, so treat it as a false to avoid having to wait the full duration
                                lockObject.IncrementCount();
                                continue;
                            }

                            RoleSyncCoordinator.CreateRequest(mapping.Community.MachineName, mapping.Community.RegionID, winnerMembership.UserID, role.SyncID, role.Tag, requestID);
                        }
                        return lockObject.WaitFor(TimeSpan.FromSeconds(3));
                    }
                }
            }
            return true;
        }

        public void Roll(GroupMember requestor)
        {
            if (_giveaway.Status != GroupGiveawayStatus.Active)
            {
                Logger.Warn("Attempt to roll a giveaway that is not active");
                return;
            }

            Logger.Info("Roll requested by " + requestor.Username);

            _currentRoll = null;
            _giveaway.CurrentRoll++;
            _giveaway.Status = GroupGiveawayStatus.Rolling;
            _giveaway.DateStatusChanged = DateTime.UtcNow;
            _giveaway.Update(g => g.Status, g => g.DateStatusChanged, g => g.CurrentRoll);
            NotifyPublicMessage(GroupGiveawayChangeType.Rolling, requestor);

            // Get a snapshot of participants
            var participants = _participants.Where(kvp => kvp.Value.Status == GroupGiveawayParticipantStatus.Entered).ToDictionary(kvp => kvp.Key, kvp => kvp.Value);
            var groupMembers = GroupMember.MultiGetLocal(participants.Select(p => new KeyInfo(_giveaway.GroupID, p.Key))).ToDictionary(g => g.UserID);

            // Perform roll
            var isFakeRoll = _giveaway.FakeRollsLeft > 0;
            var winner = SelectWinner(participants, groupMembers, isFakeRoll);
            if (winner == null)
            {
                _giveaway.Status = GroupGiveawayStatus.Active;
                _giveaway.DateStatusChanged = DateTime.UtcNow;
                _giveaway.Update(s => s.Status, s => s.DateStatusChanged);

                NotifyPublicMessage(GroupGiveawayChangeType.InvalidRoll, requestor);
                return;
            }

            GroupGiveawayChangeType notificationType;
            var rollStatus = GroupGiveawayRollStatus.Pending;
            if (winner.ValidStatus == GroupGiveawayWinnerValidStatus.Valid)
            {
                GroupGiveawayParticipantStatus newStatus;
                var effectiveRoles = new HashSet<int>(winner.Member.Roles) {_group.DefaultRoleID};
                if (effectiveRoles.Overlaps(_giveaway.AutoClaimRoles))
                {
                    // Auto-claim
                    rollStatus = GroupGiveawayRollStatus.Claimed;
                    if (_giveaway.AllowRepeatWinners)
                    {
                        newStatus = _giveaway.AllowRepeatWinners ? GroupGiveawayParticipantStatus.Entered : GroupGiveawayParticipantStatus.Claimed;
                    }
                    else
                    {
                        newStatus = _giveaway.AllowRepeatWinners ? GroupGiveawayParticipantStatus.Entered : GroupGiveawayParticipantStatus.Claimed;
                        GroupGiveawayParticipant throwaway;
                        _participants.TryRemove(winner.UserID, out throwaway);
                    }
                    _giveaway.Status = GroupGiveawayStatus.Claimed;
                    _giveaway.DateStatusChanged = DateTime.UtcNow;
                    notificationType = GroupGiveawayChangeType.PrizeClaimed;
                    GroupEventManager.LogRollGiveawayEvent(requestor, _giveaway, winner.Member, rollStatus, _group.GetRole(winner.Member.BestRole, true));
                }
                else
                {
                    // Allow winner to claim
                    _giveaway.PendingWinnerUserID = winner.UserID;
                    _giveaway.Update(g => g.PendingWinnerUserID);
                    newStatus = GroupGiveawayParticipantStatus.Winner;

                    StartClaimTimer(winner.Member);

                    _giveaway.Status = GroupGiveawayStatus.WaitingForClaim;
                    _giveaway.DateStatusChanged = DateTime.UtcNow;
                    notificationType = GroupGiveawayChangeType.WinnerSelected;
                }

                UpdateParticipantStatus(winner.UserID, newStatus, winner.Participant);
            }
            else
            {
                _giveaway.Status = GroupGiveawayStatus.Active;
                _giveaway.DateStatusChanged = DateTime.UtcNow;
                if (isFakeRoll)
                {
                    rollStatus = GroupGiveawayRollStatus.Fake;
                    notificationType = GroupGiveawayChangeType.FakeRoll;
                }
                else
                {
                    rollStatus = GroupGiveawayRollStatus.Invalid;
                    notificationType = GroupGiveawayChangeType.InvalidWinnerSelected;
                    GroupEventManager.LogRollGiveawayEvent(requestor, _giveaway, winner.Member, rollStatus, _group.GetRole(winner.Member.BestRole, true));
                }
            }

            _currentRoll = new GiveawayRoll
            {
                UserID = winner.UserID,
                Username = winner.Member.GetTitleName(),
                RollNumber = _giveaway.CurrentRoll,
                Timestamp = _giveaway.DateStatusChanged.ToEpochMilliseconds(),
                RollStatus = rollStatus,
                BestRoleID = winner.Member.BestRole,
                BestRoleName = _group.GetRole(winner.Member.BestRole, true).Name,
                ValidStatus = winner.ValidStatus
            };

            if (!isFakeRoll)
            {
                _rolls.Add(_currentRoll);
            }

            if (isFakeRoll)
            {
                _giveaway.FakeRollsLeft = Math.Max(_giveaway.FakeRollsLeft - 1, 0);
            }
            _giveaway.EligibleEntries = _participants.Count;
            _giveaway.Update(g => g.Status, g => g.DateStatusChanged, g => g.EligibleEntries, g => g.FakeRollsLeft, g => g.CurrentRoll);

            NotifyPublicMessage(notificationType, requestor, winner.Member);
        }

        private void StartClaimTimer(GroupMember winner)
        {
            ResponseWindowTimer.Start(_giveaway.GetLookupIndex(), _giveaway.ResponseWindow, () => ClaimExpiredAction(winner));
        }

        private void ClaimExpiredAction(GroupMember winner)
        {

            Logger.Info("Claim window expired for " + winner.Username);

            // Clean up timer to prevent elapse during this code block
            if (!ResponseWindowTimer.Stop(_giveaway.GetLookupIndex()))
            {
                return;
            }

            var freshParticipant = GroupGiveawayParticipant.GetLocal(winner.UserID, _giveaway.GroupID, _giveaway.GiveawayID);
            if (freshParticipant.Status == GroupGiveawayParticipantStatus.Claimed)
            {
                return;
            }

            UpdateParticipantStatus(winner.UserID, GroupGiveawayParticipantStatus.Ineligible, freshParticipant);

            _giveaway.PendingWinnerUserID = 0;
            _giveaway.Status = GroupGiveawayStatus.Active;
            _giveaway.DateStatusChanged = DateTime.UtcNow;
            _giveaway.Update(g => g.Status, g => g.PendingWinnerUserID);

            GroupGiveawayParticipant throwaway;
            _participants.TryRemove(winner.UserID, out throwaway);
            Save();

            _rolls.Last().RollStatus = GroupGiveawayRollStatus.ClaimExpired;

            NotifyPublicMessage(GroupGiveawayChangeType.ClaimExpired, _creator, winner);

            GroupEventManager.LogRollGiveawayEvent(_creator, _giveaway, winner, GroupGiveawayRollStatus.ClaimExpired, _group.GetRole(winner.BestRole, true));
        }

        private void NotifyPublicMessage(GroupGiveawayChangeType changeType, GroupMember requestor, GroupMember affectedMember = null, DateTime? timestamp=null)
        {
            var notification = new GroupGiveawayChangedNotification
            {
                Giveaway = _giveaway.ToNotification(),
                Rolls = _rolls.Select(r => r.ToContract()).ToArray(),

                ChangeType = changeType,

                Requestor = requestor.ToNotification(null),
                AffectedUser = affectedMember == null ? null : affectedMember.ToNotification(null),

                TimeStamp = (timestamp ?? DateTime.UtcNow).ToEpochMilliseconds()
            };

            if (_currentRoll != null)
            {
                notification.CurrentRoll = _currentRoll.ToContract();
            }

            try
            {
                // Get a fresh hostables in case the hostname has changed
                var group = Group.GetByID(_giveaway.GroupID);
                if (group != null)
                {
                    GroupGiveawayNotificationCoordinator.Create(group, notification);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify group of change type: " + changeType);
            }
        }

        private void NotifyTargetedMessage(GroupGiveawayChangeType changeType, GroupMember requestor, GroupMember affectedMember, DateTime? timestamp = null)
        {
            var notification = new GroupGiveawayChangedNotification
            {
                Giveaway = _giveaway.ToNotification(),
                Rolls = _rolls.Select(r => r.ToContract()).ToArray(),

                ChangeType = changeType,

                Requestor = requestor.ToNotification(null),
                AffectedUser = affectedMember.ToNotification(null),

                TimeStamp = (timestamp ?? DateTime.UtcNow).ToEpochMilliseconds()
            };

            if (_currentRoll != null)
            {
                notification.CurrentRoll = _currentRoll.ToContract();
            }

            var group = Group.GetByID(_giveaway.GroupID);
            if (group != null)
            {
                GroupGiveawayNotificationCoordinator.Create(group, notification, affectedMember.UserID);
            }
        }
    }
}
