﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.GiveawaysWebService.Contracts;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.GiveawaysWebService.Controllers
{
    [RoutePrefix("giveaways")]
    public class GiveawaysController : MicroServiceController
    {
        /// <summary>
        /// Gets all pending claims for the authenticated user.
        /// </summary>
        [HttpGet]
        [Route("pending-claims")]
        [ResponseType(typeof(IEnumerable<ActiveGiveawayResponse>))]
        public IHttpActionResult GetMyActiveClaimWindows()
        {
            var currentUser = GetCurrentUserAndRegion();
            var myEntries = GroupGiveawayParticipant.GetAllLocal(g => g.UserID, Token.UserID).Where(g => g.Status == GroupGiveawayParticipantStatus.Winner);
            var awaitingGiveaways = GroupGiveaway.MultiGetLocal(myEntries.Select(e => new KeyInfo(e.GroupID, e.GiveawayID))).Where(g => g.Status == GroupGiveawayStatus.WaitingForClaim);

            var contracts = new List<ActiveGiveawayResponse>();
            foreach (var giveaway in awaitingGiveaways)
            {
                contracts.Add(new ActiveGiveawayResponse
                {
                    Entered = true,
                    PendingWinner = new PendingWinnerContract
                    {
                        GroupID = giveaway.GroupID,
                        GiveawayID = giveaway.GiveawayID,
                        UserID = currentUser.User.UserID,
                        Username = currentUser.User.GetTitleName(),
                        Timestamp = giveaway.DateStatusChanged.ToEpochMilliseconds(),
                        SecondsUntilExpires = (int)(giveaway.ResponseWindow - (DateTime.UtcNow - giveaway.DateStatusChanged)).TotalSeconds
                    },
                    Giveaway = giveaway.ToNotification(),
                });
            }

            return Ok(contracts);
        }

        /// <summary>
        /// Gets the group's active polls.
        /// </summary>
        /// <param name="groupID">The ID of the group.</param>
        [HttpGet]
        [Route("{groupID}")]
        [ResponseType(typeof(ActiveGiveawayResponse))]
        public IHttpActionResult GetActive(Guid groupID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var giveaway = group.GetGiveaway(group.LatestGiveawayNumber);
            if (giveaway == null || giveaway.Status == GroupGiveawayStatus.Inactive)
            {
                return StatusCode(HttpStatusCode.NoContent);
            }

            var search = new GroupEventSearch
            {
                GroupID = groupID,
                GiveawayID = giveaway.GiveawayID,
                IncludedEventTypes = new[] {GroupEventType.GiveawayRoll},
                PageSize = 25,
                PageNumber = 1
            };

            var events = GroupEventManager.Search(search).Select(e => new GroupGiveawayRollContract
            {
                UserID = e.GiveawayDetails.WinnerUserID,
                Username = e.GiveawayDetails.WinnerUsername,
                Timestamp = e.Timestamp,
                RollNumber = e.GiveawayDetails.RollNumber,
                RollStatus = e.GiveawayDetails.RollStatus ?? GroupGiveawayRollStatus.Invalid,
                BestRoleID = e.GiveawayDetails.WinnerBestRoleID,
                BestRoleName = e.GiveawayDetails.WinnerBestRoleName,
                ValidStatus = e.GiveawayDetails.WinnerValidStatus ?? GroupGiveawayWinnerValidStatus.Unknown,
            });

            PendingWinnerContract pendingWinner = null;
            if (giveaway.Status == GroupGiveawayStatus.WaitingForClaim && giveaway.PendingWinnerUserID > 0)
            {
                var pendingMember = GroupMember.GetLocal(groupID, giveaway.PendingWinnerUserID);
                pendingWinner = new PendingWinnerContract
                {
                    UserID = giveaway.PendingWinnerUserID,
                    Username = pendingMember.GetTitleName(),
                    SecondsUntilExpires = (int)(giveaway.ResponseWindow - (DateTime.UtcNow - giveaway.DateStatusChanged)).TotalSeconds,
                    Timestamp = giveaway.DateStatusChanged.ToEpochMilliseconds(),
                    GroupID = groupID,
                    GiveawayID = giveaway.GiveawayID,
                };

                events = events.Concat(new[]
                {
                    new GroupGiveawayRollContract
                    {
                        UserID = pendingWinner.UserID,
                        Username = pendingWinner.Username,
                        Timestamp = pendingWinner.Timestamp,
                        RollNumber = giveaway.CurrentRoll,
                        RollStatus = GroupGiveawayRollStatus.Pending,
                        BestRoleID = pendingMember.BestRole,
                        BestRoleName = group.GetRole(pendingMember.BestRole, true).Name,
                        ValidStatus = GroupGiveawayWinnerValidStatus.Valid
                    }
                });
            }

            return Ok(new ActiveGiveawayResponse
            {
                Giveaway = giveaway.ToNotification(),
                Rolls = events.ToArray(),
                PendingWinner = pendingWinner,
                Entered = giveaway.GetParticipant(Token.UserID) != null
            });
        }

        /// <summary>
        /// Gets the most recent settings used in this group for giveaways.
        /// </summary>
        /// <param name="groupID">The ID of the group.</param>
        [HttpGet]
        [Route("{groupID}/settings")]
        [ResponseType(typeof(GroupGiveawaySettingsNotification))]
        public IHttpActionResult GetLatestSettings(Guid groupID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var settings = group.GetGiveawaySettingsOrDefault(Token.UserID);

            return Ok(settings.ToNotification());
        }

        /// <summary>
        /// Creates a giveaway.
        /// </summary>
        /// <param name="groupID">The group ID.</param>
        /// <param name="request">The giveaway settings and details.</param>
        [HttpPost]
        [Route("{groupID}")]
        [ResponseType(typeof(GroupGiveawayNotification))]
        [SocialBanFilter]
        public IHttpActionResult Create(Guid groupID, [FromBody] CreateGiveawayRequest request)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var giveaway = group.CreateGiveaway(Token.UserID,
                request.Title,
                request.RequiredRoles,
                request.RoleBonuses,
                request.SharingBonus,
                TimeSpan.FromSeconds(request.ResponseWindowSeconds),
                request.AutoEnterRoles,
                request.AutoClaimRoles,
                request.RollsBeforeWinner,
                request.AutoEnterActiveUsers,
                request.AllowRepeatWinners,
                request.IgnoredUsers,
                request.IncludeOfflineMembers);

            return Ok(giveaway.ToNotification());
        }

        /// <summary>
        /// Gets the specified giveaway.
        /// </summary>
        /// <param name="groupID">The group's ID</param>
        /// <param name="giveawayID">The giveaway number</param>
        [HttpGet]
        [Route("{groupID}/{giveawayID}")]
        [ResponseType(typeof(GroupGiveawayNotification))]
        public IHttpActionResult Get(Guid groupID, int giveawayID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.ManageGiveaways, Token.UserID);

            var giveaway = GroupGiveaway.GetLocal(groupID, giveawayID);
            if (giveaway == null)
            {
                return NotFound();
            }

            return Ok(giveaway.ToNotification());
        }

        /// <summary>
        /// Continue the giveaway after someone has claimed the prize.
        /// </summary>
        /// <param name="groupID">The group's ID.</param>
        /// <param name="giveawayID">The giveaway's ID.</param>
        [HttpPost]
        [Route("{groupID}/{giveawayID}/continues")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Continue(Guid groupID, int giveawayID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var member = group.CheckPermission(GroupPermissions.ManageGiveaways, Token.UserID);

            var giveaway = group.GetGiveaway(giveawayID);
            if (giveaway == null)
            {
                return NotFound();
            }

            if (giveaway.Status != GroupGiveawayStatus.Claimed)
            {
                return Conflict();
            }

            giveaway.Continue(member);
            return StatusCode(HttpStatusCode.Accepted);
        }

        /// <summary>
        /// Ends a giveaway
        /// </summary>
        /// <param name="groupID">the groups's ID</param>
        /// <param name="giveawayID">The giveaway's ID</param>
        /// <param name="keepActive">Whether to keep the giveaway active for members to see.</param>
        [HttpDelete]
        [Route("{groupID}/{giveawayID}")]
        [ResponseType(typeof(void))]
        public IHttpActionResult End(Guid groupID, int giveawayID, [FromUri] bool keepActive = false)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var member = group.CheckPermission(GroupPermissions.ManageGiveaways, Token.UserID);

            var giveaway = group.GetGiveaway(giveawayID);
            if (giveaway == null)
            {
                return NotFound();
            }

            if (keepActive)
            {
                // Just end it, don't deactivate it
                giveaway.End(member);
                return StatusCode(HttpStatusCode.Accepted);
            }

            giveaway.Deactivate(member);
            return StatusCode(HttpStatusCode.NoContent);
        }

        /// <summary>
        /// Performs a roll to see who wins the giveaway.
        /// </summary>
        /// <param name="groupID">The group ID</param>
        /// <param name="giveawayID">The giveaway ID</param>
        [HttpPost]
        [Route("{groupID}/{giveawayID}/rolls")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Roll(Guid groupID, int giveawayID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var member = group.CheckPermission(GroupPermissions.ManageGiveaways, Token.UserID);

            var giveaway = group.GetGiveaway(giveawayID);
            if (giveaway == null)
            {
                return NotFound();
            }


            giveaway.Roll(member);

            return StatusCode(HttpStatusCode.Accepted);
        }

        /// <summary>
        /// Registers the authenticated user in the giveaway and returns whether or not the user meets any of the required role criteria.
        /// </summary>
        /// <param name="groupID">The group ID</param>
        /// <param name="giveawayID">The giveaway ID</param>
        /// <param name="referrerUserID">Who referred the user to the giveaway.</param>
        [HttpPost]
        [Route("{groupID}/{giveawayID}/participation")]
        [ResponseType(typeof(bool))]
        [SocialBanFilter]
        public IHttpActionResult Register(Guid groupID, int giveawayID, [FromBody] int? referrerUserID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var member = group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var giveaway = group.GetGiveaway(giveawayID);
            if (giveaway == null)
            {
                return NotFound();
            }


            return Ok(giveaway.AddParticipant(member, referrerUserID ?? 0));
        }

        /// <summary>
        /// Unregisters the authenticated user from the giveaway.
        /// </summary>
        /// <param name="groupID">The group ID</param>
        /// <param name="giveawayID">The giveaway ID</param>
        [HttpDelete]
        [Route("{groupID}/{giveawayID}/participation")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Unregister(Guid groupID, int giveawayID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var member = group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var giveaway = group.GetGiveaway(giveawayID);
            if (giveaway == null)
            {
                return NotFound();
            }


            giveaway.RemoveParticipant(member);

            return Ok();
        }

        /// <summary>
        /// Claims the giveaway prize if the authenticated user has won and responds within the response window.
        /// </summary>
        /// <param name="groupID">The group ID</param>
        /// <param name="giveawayID">The giveaway ID</param>
        [HttpPost]
        [Route("{groupID}/{giveawayID}/claims")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Claim(Guid groupID, int giveawayID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var member = group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var giveaway = group.GetGiveaway(giveawayID);
            if (giveaway == null)
            {
                return NotFound();
            }


            giveaway.Claim(member);

            return StatusCode(HttpStatusCode.Accepted);
        }

        /// <summary>
        /// Gets past polls from sorted from latest to earliest.
        /// </summary>
        /// <param name="groupID">The group's ID</param>
        /// <param name="pageSize">Optional limit per page.</param>
        /// <param name="pageNumber">Optional page number</param>
        [HttpGet]
        [Route("{groupID}/history")]
        [ResponseType(typeof(IEnumerable<GroupGiveawayRollResult>))]
        public IHttpActionResult GetHistory(Guid groupID, int? pageSize = null, int? pageNumber = null)
        {
            var events = GroupEventManager.GetByGroupAndType(groupID.ToString(), GroupEventType.GiveawayRoll, pageSize ?? 10, pageNumber ?? 1);
            return Ok(events.Select(e =>
            {
                return new GroupGiveawayRollResult
                {
                    GroupID = groupID,
                    GiveawayID = e.GiveawayDetails.GiveawayID,
                    Title = e.GiveawayDetails.Title,
                    ValidWinner = e.GiveawayDetails.RollStatus != GroupGiveawayRollStatus.Invalid,
                    ClaimedPrize = e.GiveawayDetails.RollStatus == GroupGiveawayRollStatus.Claimed,
                    RollTimestamp = e.Timestamp,
                    TotalEntries = e.GiveawayDetails.Entries,
                    WinnerUserID = e.GiveawayDetails.WinnerUserID,
                    WinnerUsername = e.GiveawayDetails.WinnerUsername
                };
            }));
        }

        /// <summary>
        /// Search past polls.
        /// </summary>
        /// <param name="groupID">The group's ID</param>
        /// <param name="request">The search parameters.</param>
        [HttpPost]
        [Route("{groupID}/history/search")]
        [ResponseType(typeof(IEnumerable<GroupGiveawayRollResult>))]
        public IHttpActionResult SearchHistory(Guid groupID, [FromBody] GetGiveawayHistoryRequest request)
        {
            var events = GroupEventManager.GetByGroupAndType(groupID.ToString(), GroupEventType.GiveawayRoll, request.PageSize ?? 10, request.PageNumber ?? 1);
            return Ok(events.Select(e =>
            {
                return new GroupGiveawayRollResult
                {
                    GroupID = groupID,
                    GiveawayID = e.GiveawayDetails.GiveawayID,
                    Title = e.GiveawayDetails.Title,
                    ValidWinner = e.GiveawayDetails.RollStatus != GroupGiveawayRollStatus.Invalid,
                    ClaimedPrize = e.GiveawayDetails.RollStatus == GroupGiveawayRollStatus.Claimed,
                    RollTimestamp = e.Timestamp,
                    TotalEntries = e.GiveawayDetails.Entries,
                    WinnerUserID = e.GiveawayDetails.WinnerUserID,
                    WinnerUsername = e.GiveawayDetails.WinnerUsername
                };
            }));
        }


    }
}