﻿using Elasticsearch.Net.ConnectionPool;
using Nest;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Extensions;
using System.Diagnostics;
using Curse.Friends.Configuration;
using Aerospike.Client;
using System.Threading;
using System.Collections.Concurrent;
using Curse.CloudSearch;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Search;
using Elasticsearch.Net;
using Curse.Friends.Enums;

namespace Curse.Friends.GroupEventTests
{
    class GroupEventsTest
    {
        private static string TemplateName = "groupevents-template";
        private static string TestIndexName = "groupevents-2016m01";
        private const string IndexTypeName = "groupevents";
        private const string DefaultGroupID = "3e8129f3-deb0-49a5-80d8-c138dc90b3ac";

        public static void Test()
        {

            Console.CancelKeyPress += Console_CancelKeyPress;
            StorageConfiguration.Initialize("FrameworkTests", "US-East", ConfigurationMode.Debug, ConfigurationServices.Database | ConfigurationServices.Search);

            try
            {
                while (true)
                {
                    Console.Clear();
                    Console.WriteLine("Welcome to the Elasticsearch test app. This app is designed to quickly prototype out framework-level improvements to our Aerospike wrapper.");
                    Console.WriteLine("0: Import all group events");
                    Console.WriteLine("1. Test index creation and population.");
                    Console.WriteLine("2. Test getting a document via its id.");
                    Console.WriteLine("3. Test querying by group and date range.");
                    Console.WriteLine("4. Test querying by group and event type.");
                    Console.WriteLine("5. Delete all test indexes.");
                    Console.WriteLine("6. Test nested query.");
                    Console.WriteLine("7. Test inner object query.");

                    var selection = Console.ReadKey(true).Key;
                    Console.Clear();

                    switch (selection)
                    {
                        case ConsoleKey.D0:
                            ImportAllGroupEvents();
                            return;
                        case ConsoleKey.Escape:
                            return;
                        case ConsoleKey.D1:
                            TestIndexCreation();
                            break;
                        case ConsoleKey.D2:
                            {
                                var docID = Console.ReadLine();
                                TestGetByID(docID);
                            }
                            break;
                        case ConsoleKey.D3:
                            Console.Write("Enter a group ID: ");
                            {
                                var groupID = Console.ReadLine();
                                groupID = !string.IsNullOrEmpty(groupID) ? groupID : DefaultGroupID;
                                TestQueryingByGroupAndDateRange(groupID, DateTime.UtcNow.AddDays(-60), DateTime.UtcNow);
                            }
                            break;
                        case ConsoleKey.D4:
                            {
                                Console.Write("Enter a group ID: ");
                                var groupID = Console.ReadLine();

                                if (string.IsNullOrWhiteSpace(groupID))
                                {
                                    groupID = "3e8129f3-deb0-49a5-80d8-c138dc90b3ac";
                                }

                                foreach (var type in Enum.GetValues(typeof(GroupEventType)).Cast<GroupEventType>())
                                {
                                    Console.WriteLine("{0} - {1}", (int)type, type);
                                }
                                Console.Write("Enter the group event type integer value: ");
                                var value = Console.ReadLine();
                                TestQueryingByGroupAndType(groupID, (GroupEventType)int.Parse(value));
                            }
                            break;
                        case ConsoleKey.D5:
                            DeleteIndexes();
                            break;
                        case ConsoleKey.D6:
                            TestNestedQuery();
                            break;
                        case ConsoleKey.D7:
                            TestInnerObjectQuery();
                            break;
                        default:
                            Console.WriteLine("Invalidation selection! Press any key to continue...");
                            Console.ReadKey(true);
                            continue;

                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("General error: " + ex.Message);
                Console.ReadLine();
            }

        }

        static void DeleteIndexes()
        {
            var setupClient = GroupEventManager.GetClient();
            for (var year = 2014; year <= DateTime.UtcNow.Year; year++)
            {
                for (var month = 1; month <= 12; month++)
                {
                    setupClient.DeleteIndex(TimeSeriesIndexing.GetIndexName(IndexTypeName, year, month));
                }
            }

            Console.WriteLine("Done!");
            Console.ReadLine();

        }

        //// This presumes that weeks start with Monday.
        //// Week 1 is the 1st week of the year with a Thursday in it.
        //public static int GetIso8601WeekOfYear(DateTime time)
        //{
        //    // Seriously cheat.  If its Monday, Tuesday or Wednesday, then it'll 
        //    // be the same week# as whatever Thursday, Friday or Saturday are,
        //    // and we always get those right
        //    DayOfWeek day = CultureInfo.InvariantCulture.Calendar.GetDayOfWeek(time);
        //    if (day >= DayOfWeek.Monday && day <= DayOfWeek.Wednesday)
        //    {
        //        time = time.AddDays(3);
        //    }

        //    // Return the week of our adjusted day
        //    return CultureInfo.InvariantCulture.Calendar.GetWeekOfYear(time, CalendarWeekRule.FirstFourDayWeek, DayOfWeek.Monday);
        //}


        static void ImportAllGroupEvents()
        {
            KeepImporting = true;
            try
            {
                var logCounter = 0;
                var eventCounter = 0;
                var dict = new ConcurrentDictionary<string, ConcurrentQueue<Data.Search.GroupEvent>>();

                GroupEventLog.BatchOperateLocal(100, (logs) =>
                {

                    foreach (var c in logs)
                    {
                        if (!KeepImporting)
                        {
                            throw new AerospikeException.ScanTerminated();
                        }

                        ++logCounter;

                        IEnumerable<Data.Models.GroupEvent> events;
                        try
                        {
                            events = c.Events.GetValues();
                        }
                        catch (Exception ex)
                        {
                            Console.WriteLine("Failed to get events for group : " + c.GroupID);
                            continue;
                        }

                        var localCounter = 0;
                        var localTotalCount = events.Count();
                        var writeLock = new object();

                        Parallel.ForEach(events, m =>
                        {
                            var newEventCounter = Interlocked.Increment(ref eventCounter);
                            var newLocalCounter = Interlocked.Increment(ref localCounter);
                            if (newLocalCounter % 10 == 0)
                            {
                                Console.Title = "Processing event " + logCounter + "  (Event " + newLocalCounter + " of " + localTotalCount + ", Total Events: " + newEventCounter + ")";
                            }

                            Data.Search.GroupEvent model;

                            try
                            {
                                model = GetModelFromEntry(c.GroupID, m);
                            }
                            catch (Exception ex)
                            {
                                Console.WriteLine("Failed to create model from entity: " + ex.Message);
                                return;
                            }

                            var indexName = TimeSeriesIndexing.GetIndexName(IndexTypeName, m.Timestamp.Year, m.Timestamp.Month);
                            var queue = dict.GetOrAdd(indexName, new ConcurrentQueue<Data.Search.GroupEvent>());
                            queue.Enqueue(model);

                            lock (writeLock)
                            {
                                if (queue.Count <= 5000)
                                {
                                    return;
                                }
                                ProcessQueue(queue, indexName);
                            }

                        });
                    }
                });

                Console.WriteLine("Clearing final queues...");
                foreach (var kvp in dict)
                {
                    ProcessQueue(kvp.Value, kvp.Key);
                }
                Console.WriteLine("Import complete! Press enter to continue...");
                Console.ReadLine();
            }
            finally
            {
            }
        }

        private static void ProcessQueue(ConcurrentQueue<Data.Search.GroupEvent> queue, string indexName)
        {
            Data.Search.GroupEvent groupEvent;
            var operations = new List<IBulkOperation>();

            while (queue.TryDequeue(out groupEvent))
            {
                operations.Add(new BulkIndexOperation<Data.Search.GroupEvent>(groupEvent) { Id = groupEvent.EventID, Index = indexName });
            }

            var request = new BulkRequest()
            {
                Refresh = true,
                Consistency = Consistency.One,
                Operations = operations
            };

            var client = GroupEventManager.GetClient();
            var response = client.Bulk(request);
            if (!response.ConnectionStatus.Success)
            {
                Console.WriteLine("Failed to bulk index!");
            }
        }

        private static bool KeepImporting = false;

        static void Console_CancelKeyPress(object sender, ConsoleCancelEventArgs e)
        {
            e.Cancel = true;
            KeepImporting = false;
        }


        static Data.Search.GroupEvent GetModelFromEntry(Guid groupID, Data.Models.GroupEvent entry)
        {
            return new Data.Search.GroupEvent
            {
                EventID = entry.EventID.ToString(),
                GroupID = groupID,
                Timestamp = entry.Timestamp.ToEpochMilliseconds(),
                InitiatingUserID = entry.InitiatingUserID,
                InitiatingUsername = "Unknown",
                EventType = GetTypeFromLegacy(entry.Type),
                EventCategory = GetCategoryFromLegacy(entry.Type),
                AffectedGroups = new GroupEventGroup[0],
                AffectedRoles = new GroupEventRole[0],
                AffectedUsers = new GroupEventUser[0],
            };
        }

        private static GroupEventCategory GetCategoryFromLegacy(LegacyGroupEventType legacyType)
        {
            switch (legacyType)
            {
                case LegacyGroupEventType.RootGroupCreated:
                case LegacyGroupEventType.SubgroupCreated:
                case LegacyGroupEventType.SubgroupRemoved:
                    return GroupEventCategory.Group;
                case LegacyGroupEventType.UserDemoted:
                case LegacyGroupEventType.UserInvited:
                case LegacyGroupEventType.UserLeft:
                case LegacyGroupEventType.UserPromoted:
                case LegacyGroupEventType.UsersAdded:
                case LegacyGroupEventType.UsersRemoved:
                    return GroupEventCategory.User;
                case LegacyGroupEventType.GroupAvatarChanged:
                case LegacyGroupEventType.GroupTitleChanged:
                default:
                    return GroupEventCategory.CommunityLink;
            }
        }

        private static GroupEventType GetTypeFromLegacy(LegacyGroupEventType legacyType)
        {
            switch (legacyType)
            {
                case LegacyGroupEventType.RootGroupCreated:
                    return GroupEventType.GroupCreated;
                case LegacyGroupEventType.SubgroupCreated:
                    return GroupEventType.SubgroupCreated;
                case LegacyGroupEventType.SubgroupRemoved:
                    return GroupEventType.SubgroupRemoved;
                case LegacyGroupEventType.UserDemoted:
                    return GroupEventType.UserRolesRemoved;
                case LegacyGroupEventType.UserInvited:
                    return GroupEventType.UsersAdded;
                case LegacyGroupEventType.UserLeft:
                    return GroupEventType.UsersRemoved;
                case LegacyGroupEventType.UserPromoted:
                    return GroupEventType.UserRolesAdded;
                case LegacyGroupEventType.UsersAdded:
                    return GroupEventType.UsersRemoved;
                case LegacyGroupEventType.UsersRemoved:
                    return GroupEventType.UsersRemoved;
                case LegacyGroupEventType.GroupAvatarChanged:
                case LegacyGroupEventType.GroupTitleChanged:
                default:
                    return GroupEventType.CommunityUnlinked;
            }
        }

        static void TestIndexCreation()
        {
            var setupClient = GroupEventManager.GetClient();
            Console.WriteLine("Would you like to delete the index and start over? (Y|N)");
            if (Console.ReadKey(true).Key == ConsoleKey.Y)
            {
                setupClient.DeleteIndex(c => c.Index(TestIndexName));
                setupClient.DeleteTemplate(TemplateName);

                var putResponse = setupClient.PutTemplate("", t => t
                    .Name(TemplateName)
                    .Template("groupevents-*")
                    .AddMapping<Data.Search.GroupEvent>(s => s
                        .MapFromAttributes()
                        .SourceField(sf => sf.Enabled(true))
                        .AllField(af => af.Enabled(false))
                    )
                );

                Console.WriteLine("Template created!");
                Console.WriteLine("Request Url: " + putResponse.RequestInformation.RequestUrl);
                Console.WriteLine("Request Parms:" + Encoding.Default.GetString(putResponse.RequestInformation.Request));

            }

            Console.WriteLine("How many groups would you like to create events for?");
            var groupCount = int.Parse(Console.ReadLine());

            Console.WriteLine("How many events would you like to index, per group?");
            var eventCount = int.Parse(Console.ReadLine());

            Console.WriteLine("How many weeks ago to start?");
            var weekCount = int.Parse(Console.ReadLine());

            var totalCount = groupCount * eventCount;

            Console.WriteLine("Inserting docs...");
            var sw = Stopwatch.StartNew();
            var rnd = new Random();

            for (var i = 0; i < groupCount; i++)
            {
                var groupID = i == 0 ? new Guid(DefaultGroupID) : Guid.NewGuid();
                var startingTimestamp = DateTime.UtcNow.AddDays(-7 * weekCount);

                int giveawayID = 0;
                Parallel.For(0, eventCount, j =>
                {
                    var client = GroupEventManager.GetClient();
                    var timestamp = startingTimestamp.AddSeconds(j + rnd.Next(1, 30));

                    if (timestamp > DateTime.UtcNow)
                    {
                        timestamp = DateTime.UtcNow;
                    }

                    var userID = rnd.Next(1, 100);
                    var otherUserID = rnd.Next(101, 200);
                    var roleID = rnd.Next(1, 100);
                    var myGiveawayID = Interlocked.Increment(ref giveawayID);
                    var groupEvent = new Data.Search.GroupEvent
                    {
                        GroupID = groupID,
                        EventID = Guid.NewGuid().ToString(),
                        EventCategory = GroupEventCategory.User,
                        EventType = j % 2 == 0 ? GroupEventType.UserRolesAdded : GroupEventType.UserRolesRemoved,
                        InitiatingUserID = userID,
                        InitiatingUsername = "User" + userID,
                        Timestamp = timestamp.ToEpochMilliseconds(),
                        AffectedUsers = new[] { new GroupEventUser { ID = otherUserID, Username = "User" + otherUserID } },
                        AffectedRoles = new[] { new GroupEventRole { ID = roleID, Name = "Role" + roleID } },
                        AffectedGroups = new GroupEventGroup[0],
                        AffectedGiveaway = new GroupEventGiveaway
                        {
                            GiveawayID = myGiveawayID,
                            ClaimedPrize = j % 2 == 0,
                            Title = "Giveaway" + myGiveawayID,
                            TotalEntries = rnd.Next(1, 20),
                            ValidWinner = true
                        },
                        AffectedLinkedCommunity = new GroupEventLinkedCommunity(),
                        AffectedPoll = new GroupEventPoll()
                    };

                    var resp = client.Index(groupEvent, idx => idx.Index(TestIndexName).Routing(groupEvent.GroupID.ToString()));
                    if (!resp.ConnectionStatus.Success)
                    {
                        Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
                        Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));
                        Console.WriteLine("Failed to index document!");
                    }
                });
            }

            sw.Stop();
            var timePer = sw.ElapsedMilliseconds / (double)totalCount;
            Console.WriteLine("Inserted " + totalCount + " documents in " + sw.ElapsedMilliseconds + "ms (" + timePer + " ms/doc)");
            Console.ReadLine();
        }

        static void TestGetByID(string id)
        {
            var client = GroupEventManager.GetClient();
            var sw = Stopwatch.StartNew();
            var resp = client.Get<Data.Search.GroupEvent>(p => p.Index(TestIndexName).Id(id));
            sw.Stop();
            Console.WriteLine("Retrieved item " + resp.Source.EventID + " in " + sw.ElapsedMilliseconds + "ms");
            Console.ReadLine();
        }

        static void TestQueryingByGroupAndDateRange(string groupID, DateTime startDate, DateTime endDate)
        {

            var client = GroupEventManager.GetClient();
            var pageSize = 100;
            var currentPage = 0;
            while (true)
            {
                var resp = client.Search<Data.Search.GroupEvent>(s => s.Query(q =>
                    q.Bool(b => b.Must(qd => qd.Term(t => t.GroupID, groupID)))
                    && q.Bool(b => b.Must(qd => qd.Range(r => r.OnField(f => f.Timestamp).GreaterOrEquals(startDate.ToEpochMilliseconds()).LowerOrEquals(endDate.ToEpochMilliseconds())))))
                    .Routing(groupID)
                    .Size(pageSize)
                    .Skip(currentPage * pageSize)
                    .SortDescending(p => p.Timestamp));
                Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
                Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));

                var docs = resp.Documents;
                Console.WriteLine("Returned " + docs.Count() + " documents in " + resp.ElapsedMilliseconds + "ms");
                if (docs.Count() < pageSize)
                {
                    break;
                }
                Console.WriteLine("Press enter to continue...");
                Console.ReadLine();
                ++currentPage;
            }

            Console.ReadLine();
        }

        static void TestQueryingByGroupAndType(string groupID, GroupEventType type)
        {
            groupID = string.IsNullOrEmpty(groupID) ? DefaultGroupID : groupID;

            var client = GroupEventManager.GetClient();
            var pageSize = 100;
            var currentPageNumber = 0;
            var indexNames = TimeSeriesIndexing.GetMonthsBetween(DateTime.UtcNow.AddYears(-2), DateTime.UtcNow).Select(m => TimeSeriesIndexing.GetIndexName(IndexTypeName, m.Year, m.Month));
            while (true)
            {

                var resp = client.Search<Data.Search.GroupEvent>(
                    s => s.Filter(
                        q => q.Bool(b => b.Must(qd => qd.Term(t => t.GroupID, groupID))) && q.Bool(b => b.Must(qd => qd.Term(ts => ts.EventType, (int)type))))
                        .Size(pageSize)
                        .Indices(indexNames)
                        .Routing(groupID)
                        .Skip(currentPageNumber * pageSize)
                        .SortDescending(p => p.Timestamp));

                Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
                Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));
                Console.ReadKey(true);

                var docs = resp.Documents;
                Console.WriteLine("Returned " + docs.Count() + " documents in " + resp.ElapsedMilliseconds + "ms");

                if (docs.Count() < pageSize)
                {
                    break;
                }
                Console.WriteLine("Press enter to continue...");
                Console.ReadLine();
                currentPageNumber++;
            }
            Console.ReadLine();
        }

        static void TestNestedQuery()
        {
            var client = GroupEventManager.GetClient();
            var resp = client.Search<Data.Search.GroupEvent>(s => s.Filter(f =>
                f.Term(t => t.GroupID, DefaultGroupID)
                &&
                f.Nested(n =>
                    n.Path(g => g.AffectedUsers)
                        .Filter(f2 => f2.Terms(c => c.AffectedUsers.First().ID, Enumerable.Range(101, 50)))))

                .Index(TestIndexName)
                .Routing(DefaultGroupID)
                .Size(20));

            if (!resp.ConnectionStatus.Success)
            {
                Console.WriteLine("Error requesting affected users with ID between 101 and 150");
                Console.WriteLine(resp.RequestInformation.HttpStatusCode);
                Console.WriteLine(resp.RequestInformation.RequestUrl);
            }
            else
            {
                Console.WriteLine("Affected users with ID between 101 and 150 - {0} returned in {1} ms", resp.Hits.Count(), resp.ElapsedMilliseconds);
                foreach (var hit in resp.Hits.Select(h => h.Source))
                {
                    Console.WriteLine("{0} - {1}", hit.EventID, hit.AffectedUsers.First().ID);
                }
            }

            Console.ReadLine();

            resp = client.Search<Data.Search.GroupEvent>(s => s.Filter(f =>
                f.Term(t => t.GroupID, DefaultGroupID)
                &&
                f.Nested(n =>
                    n.Path(g => g.AffectedUsers)
                        .Filter(f2 => f2.Terms(c => c.AffectedUsers.First().ID, Enumerable.Range(151, 50)))))

                .Index(TestIndexName)
                .Routing(DefaultGroupID)
                .Size(20));

            if (!resp.ConnectionStatus.Success)
            {
                Console.WriteLine("Error requesting affected users with ID between 151 and 200");
                Console.WriteLine(resp.RequestInformation.HttpStatusCode);
                Console.WriteLine(resp.RequestInformation.RequestUrl);
            }
            else
            {
                Console.WriteLine("Affected users with ID between 151 and 200 - {0} returned in {1} ms", resp.Hits.Count(), resp.ElapsedMilliseconds);
                foreach (var hit in resp.Hits.Select(h => h.Source))
                {
                    Console.WriteLine("{0} - {1}", hit.EventID, hit.AffectedUsers.First().ID);
                }
            }

            Console.ReadLine();
        }

        static void TestInnerObjectQuery()
        {
            var client = GroupEventManager.GetClient();
            var resp = client.Search<Data.Search.GroupEvent>(s => s.Filter(f =>
                f.Term(t => t.GroupID, DefaultGroupID)
                &&
                f.Range(r => r.OnField(t => t.AffectedGiveaway.TotalEntries).GreaterOrEquals(1).LowerOrEquals(10))
                )
                .Index(TestIndexName)
                .Routing(DefaultGroupID)
                .Size(20));

            if (!resp.ConnectionStatus.Success)
            {
                Console.WriteLine("Error requesting affected giveaway with 1-10 entries");
                Console.WriteLine(resp.RequestInformation.HttpStatusCode);
                Console.WriteLine(resp.RequestInformation.RequestUrl);
            }
            else
            {
                Console.WriteLine("Affected giveaways with entries between 1 and 10 - {0} returned in {1} ms", resp.Hits.Count(), resp.ElapsedMilliseconds);
                foreach (var hit in resp.Hits.Select(h => h.Source))
                {
                    Console.WriteLine("{0} - {1}", hit.EventID, hit.AffectedGiveaway.TotalEntries);
                }
            }

            Console.ReadLine();

            resp = client.Search<Data.Search.GroupEvent>(s => s.Filter(f =>
                f.Term(t => t.GroupID, DefaultGroupID)
                &&
                f.Range(r => r.OnField(t => t.AffectedGiveaway.TotalEntries).GreaterOrEquals(11).LowerOrEquals(20))
                )
                .Index(TestIndexName)
                .Routing(DefaultGroupID)
                .Size(20));

            if (!resp.ConnectionStatus.Success)
            {
                Console.WriteLine("Error requesting affected giveaway with 11-20 entries");
                Console.WriteLine(resp.RequestInformation.HttpStatusCode);
                Console.WriteLine(resp.RequestInformation.RequestUrl);
            }
            else
            {
                Console.WriteLine("Affected giveaways with entries between 11 and 20 - {0} returned in {1} ms", resp.Hits.Count(), resp.ElapsedMilliseconds);
                foreach (var hit in resp.Hits.Select(h => h.Source))
                {
                    Console.WriteLine("{0} - {1}", hit.EventID, hit.AffectedGiveaway.TotalEntries);
                }
            }

            Console.ReadLine();
        }
    }
}
