﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Elasticsearch.Net;
using Nest;

namespace Curse.Friends.GroupEventTests
{
    class GroupMembersTest
    {
        private static readonly Guid DefaultGroupID = new Guid("795C27BF-6C90-44DE-88F3-DB5655D52EB4");
        public static void Test()
        {

            while (true)
            {
                Console.Clear();
                Console.WriteLine("Group Member Tests");

                Console.WriteLine("0 - Reset Index");
                Console.WriteLine("1 - Get");
                Console.WriteLine("2 - Search by Username");
                Console.WriteLine("3 - Search by Role");
                Console.WriteLine("4 - Import Group Members");

                switch (Console.ReadKey(true).Key)
                {
                    case ConsoleKey.Escape:
                        return;
                    case ConsoleKey.D0:
                        ResetIndex();
                        break;
                    case ConsoleKey.D1:
                        GetAllMembers();
                        break;
                    case ConsoleKey.D2:
                        SearchByUsername();
                        break;
                    case ConsoleKey.D3:
                        SearchByRoleID();
                        break;
                    case ConsoleKey.D4:
                        ImportGroupMembers();
                        break;
                    default:
                        Console.WriteLine("Invalid selection.");
                        break;
                }

                Console.WriteLine("Press enter to continue");
                Console.ReadLine();
            }
        }

        static void ImportGroupMembers()
        {
            var client = GroupMemberManager.GetClient();

            Console.WriteLine("Do you want to delete the index and start over?");
            if (Console.ReadKey(true).Key == ConsoleKey.Y)
            {
                client.DeleteIndex(d => d.Index("groupmembers-index"));
                client.DeleteTemplate("groupmembers-template");
                client.PutTemplate("groupmembers-template", t =>
                    t.Template("groupmembers-*")
                        .AddMapping<GroupMemberSearchModel>(m => m
                            .MapFromAttributes()
                            .SourceField(sf => sf.Enabled(true))
                            .AllField(af => af.Enabled(false))
                        )
                    );
            }

            Console.WriteLine("Enter a group ID or 'all'");
            var result = Console.ReadLine();

            Guid groupID;
            if (Guid.TryParse(result, out groupID))
            {
                Console.WriteLine("Getting members from Aerospike");
                var members = GroupMember.GetAllLocal(g => g.GroupID, groupID);
                Console.WriteLine("Indexing members in Elastic");
                var sw = new Stopwatch();
                sw.Start();
                Parallel.ForEach(members, member =>
                {
                    var memberModel = new GroupMemberSearchModel
                    {
                        GroupID = groupID,
                        UserID = member.UserID,
                        BestRole = member.BestRole,
                        BestRoleRank = member.BestRoleRank,
                        Roles = member.Roles.ToArray(),
                        Username = member.Username,
                        RegionID = member.RegionID,
                        IsDeleted = member.IsDeleted,
                        DateJoined = member.DateJoined.ToEpochMilliseconds(),
                        ModelID = GroupMemberSearchModel.CreateModelID(groupID, member.UserID)
                    };
                    GroupMemberManager.Index(memberModel);
                });
                sw.Stop();
                Console.WriteLine("Finished indexing {0} members in {1} ms", members.Length, sw.ElapsedMilliseconds);
            }
            else if ("all".Equals(result))
            {
                const string idxName = "groupmembers-index";
                var count = 0;
                const int batchSize = 1000;
                Console.WriteLine("Indexing all group members.");
                var sw = new Stopwatch();
                sw.Start();
                GroupMember.BatchOperateLocal(batchSize, members =>
                {
                    var operations = new List<IBulkOperation>();
                    var processed = 0;
                    foreach (var member in members.Where(m=>m.GroupID==m.RootGroupID))
                    {
                        if (member.GroupID == Guid.Empty)
                        {
                            Debugger.Break();
                        }
                        var memberModel = new GroupMemberSearchModel
                        {
                            GroupID = member.GroupID,
                            UserID = member.UserID,
                            BestRole = member.BestRole,
                            BestRoleRank = member.BestRoleRank,
                            Roles = member.Roles.ToArray(),
                            Username = member.Username,
                            RegionID = member.RegionID,
                            IsDeleted = member.IsDeleted,
                            DateJoined = member.DateJoined.ToEpochMilliseconds(),
                            ModelID = GroupMemberSearchModel.CreateModelID(member.GroupID, member.UserID)
                        };
                        operations.Add(new BulkIndexOperation<GroupMemberSearchModel>(memberModel) { Id = memberModel.ModelID, Index = idxName, Routing = member.GroupID.ToString() });
                        processed++;
                    }

                    var request = new BulkRequest
                    {
                        Refresh = true,
                        Consistency = Consistency.One,
                        Operations = operations
                    };
                    client.Bulk(request);

                    Console.Title = count + " Processed";
                    count += processed;
                });
                sw.Stop();
                Console.WriteLine("Finished processing {0} members in {1} minutes",count,sw.Elapsed.TotalMinutes);
            }
            else
            {
                Console.WriteLine("Invalid entry");
            }
        }

        private static void ImportGroup(Guid groupID)
        {
            
        }

        static void ResetIndex()
        {
            Console.WriteLine("Resetting template and index");
            var client = GroupMemberManager.GetClient();
            var deleteIndexResp=client.DeleteIndex(d => d.Index("groupmembers-index"));
            var deleteTemplateResp = client.DeleteTemplate("groupmembers-template");

            var ana = new AnalysisSettings
            {
                Analyzers =
                {
                    {
                        "autocomplete", new CustomAnalyzer
                        {
                            Tokenizer = "standard",
                            Filter = new List<string> {"lowercase", "autocomplete_filter"}
                        }
                    }
                },
                TokenFilters = {{"autocomplete_filter", new EdgeNGramTokenFilter {MinGram = 2, MaxGram = 32}}}
            };

            var putTemplateResp = client.PutTemplate("groupmembers-template", t =>
                t.Template("groupmembers-*")
                    .AddMapping<GroupMemberSearchModel>(m => m
                        .MapFromAttributes()
                        .SourceField(sf => sf.Enabled(true))
                        .AllField(af => af.Enabled(false))
                    )
                    .Settings(dict => dict
                        .Add("index.analysis", ana)
                    )
                );

            Console.WriteLine("How many group members do you want to import?");
            var memberCount = int.Parse(Console.ReadLine());

            var sw = new Stopwatch();
            sw.Start();
            var members = new ConcurrentBag<GroupMember>();
            var groupID = DefaultGroupID;
            var rand = new Random();
            int userID = 0;
            Parallel.For(0, memberCount, j =>
            {
                var myUserID = Interlocked.Increment(ref userID);
                var myRoles = Enumerable.Range(rand.Next(1, 5), rand.Next(6, 10)).ToArray();
                var myBestRole = myRoles[rand.Next(0, myRoles.Length - 1)];
                var member = new GroupMember
                {
                    GroupID = groupID,
                    UserID = myUserID,
                    Username = "User" + myUserID,
                    Roles = new HashSet<int>(myRoles),
                    BestRole = myBestRole,
                    BestRoleRank = myBestRole,
                    DateJoined = DateTime.UtcNow.AddMonths(0 - rand.Next(1, 10)),
                    RegionID = rand.Next(0, 3) * 2 + 1,
                    IsDeleted = false,
                };

                members.Add(member);
                GroupMemberManager.AddOrUpdateMember(member);
            });
            sw.Stop();

            Console.WriteLine("Indexed {0} members in {1}ms", memberCount, sw.ElapsedMilliseconds);
            Console.ReadKey(true);
        }

        static void GetAllMembers()
        {
            int pageSize;
            int pageNumber;
            PromptForPage(out pageSize, out pageNumber);

            var sw = new Stopwatch();
            sw.Start();

            var results = GroupMemberManager.GetByGroup(DefaultGroupID, pageSize, pageNumber);
            sw.Stop();

            Console.WriteLine("Return {0} results in {1} ms", results.Length, sw.ElapsedMilliseconds);

            foreach (var user in results)
            {
                WriteUser(user);
            }

            Console.ReadKey(true);
        }

        static void SearchByUsername()
        {
            Console.WriteLine("Enter username to search for");
            var searchUsername = Console.ReadLine();

            int pageNumber;
            int pageSize;
            PromptForPage(out pageSize, out pageNumber);

            var sw = new Stopwatch();
            sw.Start();
            var searched = GroupMemberManager.SearchGroupMembers(DefaultGroupID, new GroupMemberSearch
            {
                PageNumber = pageNumber,
                PageSize = pageSize,
                Username = searchUsername
            });
            sw.Stop();

            Console.WriteLine("Found {0} matches for username {1} in {2} ms", searched.Length, searchUsername, sw.ElapsedMilliseconds);

            foreach (var user in searched)
            {
                WriteUser(user);
            }

            Console.ReadKey(true);
        }

        static void SearchByRoleID()
        {
            Console.WriteLine("Enter role ID (1-10)");
            var searchRole = int.Parse(Console.ReadLine());

            int pageNumber;
            int pageSize;
            PromptForPage(out pageSize, out pageNumber);

            var sw = new Stopwatch();
            sw.Start();
            var searched2 = GroupMemberManager.SearchGroupMembers(DefaultGroupID, new GroupMemberSearch
            {
                PageSize = pageSize,
                PageNumber = pageNumber,
                RoleID = searchRole
            });
            sw.Stop();

            Console.WriteLine("Found {0} matches for role {1} in {2} ms", searched2.Length, searchRole, sw.ElapsedMilliseconds);

            foreach(var user in searched2)
            {
                WriteUser(user);
            }

            Console.ReadKey(true);

        }

        static void WriteUser(GroupMemberSearchModel user)
        {
            Console.WriteLine("UserID: {0}, Username: {1}, BestRole: {2}, Roles: {3}", user.UserID, user.Username, user.BestRole, string.Join(",", user.Roles));
        }

        static void PromptForPage(out int pageSize, out int pageNumber)
        {
            Console.WriteLine("How many results per page?");
            pageSize = int.Parse(Console.ReadLine());

            Console.WriteLine("Which page?");
            pageNumber = int.Parse(Console.ReadLine());
        }
    }
}
