﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Nest;

namespace Curse.Friends.GroupEventTests
{
    public class GroupsTest
    {
        public static void Test()
        {
            var client = GroupSearchManager.GetClient();

            Console.WriteLine("Do you want to delete everything and start over? (Y|N)");
            if (Console.ReadKey(true).Key == ConsoleKey.Y)
            {
                client.DeleteTemplate("groups-template");
                client.DeleteIndex(t => t.Index("groups-index"));
                client.PutTemplate("groups-template", t => t
                    .Template("groups-*")
                    .AddMapping<GroupSearchModel>(m => m
                        .MapFromAttributes()
                        .AllField(af => af.Enabled(false))
                        .SourceField(sf => sf.Enabled())));
            }

            Console.WriteLine("Do you want to index random data? (Y|N)");
            if (Console.ReadKey(true).Key == ConsoleKey.Y)
            {
                GenerateRandomGroups(client);
            }

            while (true)
            {
                Console.Clear();
                Console.WriteLine("Enter a search query");
                var query = Console.ReadLine();
                Console.WriteLine("Enter a comma-separated list of tags (1-x)");
                var tagsEntry = Console.ReadLine();
                var tags = string.IsNullOrWhiteSpace(tagsEntry) ? null : tagsEntry.Split(',').Select(t => (GroupSearchTag) int.Parse(t)).ToArray();

                var search = new GroupSearch
                {
                    Query = query,
                    Tags = tags
                };
                var results = GroupSearchManager.Search(search);
                WriteResults(results);
                Console.ReadKey(true);

                Console.WriteLine("Would you like to delete these entries? (Y|N)");
                if (Console.ReadKey(true).Key == ConsoleKey.Y)
                {
                    foreach (var result in results)
                    {
                        GroupSearchManager.UpdateSearchability(result.GroupID, false);
                    }

                    client.Refresh(r => r.Index("groups-index"));
                    WriteResults(GroupSearchManager.Search(search));
                }
                else
                {
                    Console.WriteLine("Would you like to update the member count of these results? (Y|N)");
                    if (Console.ReadKey(true).Key == ConsoleKey.Y)
                    {
                        foreach (var result in results)
                        {
                            GroupSearchManager.UpdateMemberCount(result.GroupID, result.MemberCount + 100);
                        }
                    }

                    client.Refresh(r => r.Index("groups-index"));
                    WriteResults(GroupSearchManager.Search(search));
                }
                Console.ReadKey(true);
            }
        }

        static void WriteResults(GroupSearchModel[] results)
        {
            Console.WriteLine("Total Results: {0}",results.Length);
            foreach (var result in results)
            {
                Console.WriteLine("Group {0}, {1}, {2}, {3}", result.GroupID,result.GroupTitle, result.IsPublic?"Public":"Private", result.IsDeleted?"Deleted":"Active");
                Console.WriteLine("Owner {0}, {1}", result.OwnerUserID, result.OwnerUsername);
                Console.WriteLine("Members = {0}, Tags={1}", result.MemberCount, string.Join(",", result.Tags));
                Console.WriteLine(result.Description);
                Console.WriteLine();
            }
        }

        static void GenerateRandomGroups(ElasticClient client)
        {
            var groupsToAdd = new List<GroupSearchModel>();

            var possibleTags = Enum.GetValues(typeof (GroupSearchTag)).Cast<GroupSearchTag>().ToArray();
            var rand = new Random();
            for (var i = 0; i < 100; i++)
            {
                var tags = new List<GroupSearchTag>();
                var currentPossibleTags = new List<GroupSearchTag>(possibleTags);
                for (var j = 0; j < rand.Next(4);j++)
                {
                    var index = rand.Next(0, currentPossibleTags.Count);
                    tags.Add(currentPossibleTags[index]);
                    currentPossibleTags.RemoveAt(index);
                }

                var groupSearchModel = new GroupSearchModel
                {
                    GroupID = Guid.NewGuid(),
                    Description = string.Format("This group is in the {0} grouping", i/10*10),
                    GroupTitle = "Group" + i,
                    IsPublic = i%2 == 0,
                    MemberCount = 100 - i,
                    OwnerUserID = i,
                    OwnerUsername = "User" + (i/3),
                    Tags = tags.ToArray(),
                    IsDeleted = false
                };

                groupsToAdd.Add(groupSearchModel);
            }

            var request = new BulkRequest()
            {
                Index = "groups-index",
                Refresh = true,
                Operations = groupsToAdd.Select(g => (IBulkOperation) new BulkIndexOperation<GroupSearchModel>(g) {Id = g.GroupID.ToString()}).ToArray()
            };

            var result = client.Bulk(request);

            if (!result.ConnectionStatus.Success)
            {
                Console.WriteLine("Errors occurred while bulk indexing!");
                Debugger.Break();
            }
        }
    }
}
