﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Logging;

namespace Curse.Friends.GroupService
{
    public class GroupCache
    {
        /// <summary>
        /// Cached role permissions lookup for this group
        /// </summary>
        public Dictionary<int, GroupRolePermissions> PermissionsByRole { get; private set; }

        public GroupType Type;

        public Group Group { get; private set; }
        public Group RootGroup { get; private set; }
        public ExternalCommunity Community { get; private set; }

        public GroupCache(Group group, Group rootGroup)
        {
            Group = group;
            RootGroup = rootGroup;
            Update(group);
        }

        public bool HasPermission(GroupPermissions permission, GroupMember member)
        {
            // Build up the list of roles this user has (ensuring they always 
            var memberRoleIDs = new HashSet<int>(member.Roles);
            memberRoleIDs.Add(RootGroup.DefaultRoleID);

            // Build up the list of permissions for these roles, from our dictionary
            var memberRolePermissions = new List<GroupRolePermissions>();

            foreach (var roleID in memberRoleIDs)
            {
                GroupRolePermissions rolePermissions;

                if (PermissionsByRole.TryGetValue(roleID, out rolePermissions))
                {
                    memberRolePermissions.Add(rolePermissions);
                }
            }

            // Call into the 
            return Group.HasPermission(permission, member, memberRolePermissions);
        }

        public void Refresh()
        {
            try
            {                
                // Refresh from the DB
                Group.Refresh();
                Update(Group);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to refresh group cache!", new { Group.GroupID, Group.RootGroupID, Group.Title });
            }

        }

        public void Update(Group group)
        {
            Group = group;
            RootGroup = group.RootGroup;
            Type = group.Type;
            PermissionsByRole = Group.GetPermissionsByRole();
            if (!string.IsNullOrEmpty(group.ExternalChannelID))
            {
                Community = ExternalCommunity.GetLocal(group.ExternalChannelID, AccountType.Twitch);
            }
        }

        private volatile bool _isPendingSave;
        private DateTime _dateLastMessaged;

        public void UpdateDateMessaged(DateTime timestamp)
        {
            _dateLastMessaged = timestamp;
            _isPendingSave = true;
        }

        public void SaveChanges()
        {
            if (!_isPendingSave)
            {
                return;
            }

            _isPendingSave = false;
            Group.DateMessaged = _dateLastMessaged;
            Group.Update(p => p.DateMessaged);
        }

        public long GetPermissionsHash(IEnumerable<int> roles)
        {
            var allPermissions = GroupPermissions.None;

            foreach (var roleID in roles)
            {
                GroupRolePermissions rolePermissions;
                if (PermissionsByRole.TryGetValue(roleID, out rolePermissions))
                {
                    if (rolePermissions == null)
                    {
                        Logger.Warn(string.Format("Role {0} permissions not found during a role check on group {1}", roleID, Group.GroupID));
                        continue;
                    }
                    allPermissions = allPermissions | rolePermissions.Permissions;
                }
            }

            return (long) allPermissions;
        }

        public void UpdateCommunity(ExternalCommunity community=null)
        {
            if (community != null)
            {
                Community = community;
            }
            else if (!string.IsNullOrEmpty(Group.ExternalChannelID))
            {
                Community = ExternalCommunity.GetLocal(Group.ExternalChannelID, AccountType.Twitch);
            }
        }
    }
}
