﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Curse.Friends.Data;
using Curse.Logging;
using Curse.Aerospike;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;

namespace Curse.Friends.GroupService
{
    public class GroupMemberCache
    {
        public GroupMember Member { get; private set; }

        private Dictionary<string, GroupMemberEndpointCache> _deliverableEndpoints = new Dictionary<string, GroupMemberEndpointCache>();
        public IReadOnlyCollection<GroupMemberEndpointCache> DeliverableEndpoints { get; private set; }
        private Dictionary<string, GroupMemberEndpointCache> _connectedEndpoints = new Dictionary<string, GroupMemberEndpointCache>();
        public IReadOnlyCollection<GroupMemberEndpointCache> ConnectedEndpoints { get; private set; }


        private readonly Dictionary<Guid, GroupMessageState> _dateMessagedByGroupID = new Dictionary<Guid, GroupMessageState>();
        private readonly object _syncRoot = new object();        
        private DateTime _dateMessageSent;

        private ExternalAccount _externalAccount;
        private DateTime _dateExternalAccountRefreshed;

        public GroupMemberCache(Group group, GroupMember member, IReadOnlyCollection<ClientEndpoint> endpoints, ExternalAccount externalAccount)
        {
            Member = member;
            UpdateEndpoints(group, endpoints ?? new ClientEndpoint[0], false);
            _externalAccount = externalAccount;
            _dateExternalAccountRefreshed = DateTime.UtcNow.AddMinutes(-10);
        }

        public void Refresh(Group group)
        {
            var member = group.GetMember(Member.UserID);
            if (member == null)
            {
                Logger.Warn("Unable to refresh group member. It was not found in the database!", new { Member.GroupID, Member.UserID });
                return;
            }

            UpdateMember(member);
        }

        public void UpdateMember(GroupMember member)
        {
            Member.Nickname = member.Nickname;            
            Member.Roles = member.Roles;
            Member.BestRole = member.BestRole;
            Member.BestRoleRank = member.BestRoleRank;
            Member.IsVoiceDeafened = member.IsVoiceDeafened;
            Member.IsVoiceMuted = member.IsVoiceMuted;
            Member.NotificationFilters = member.NotificationFilters;
            Member.NotificationMuteDate = member.NotificationMuteDate;
            Member.NotificationPreference = member.NotificationPreference;
            Member.DateJoined = member.DateJoined;
            Member.Username = member.Username;
            Member.DisplayName = member.DisplayName;
        }

        public void UpdateEndpoints(Group group, IReadOnlyCollection<ClientEndpoint> endpoints, bool updateActivity)
        {
            var wasConnected = IsConnected;

            var newEndpoints = endpoints.ToDictionary(ep => ep.MachineKey);
            var myEndpoints = new Dictionary<string, GroupMemberEndpointCache>();
            foreach (var ep in newEndpoints)
            {
                GroupMemberEndpointCache cache;
                if (_deliverableEndpoints.TryGetValue(ep.Key, out cache))
                {
                    cache.Update(ep.Value);
                }
                else
                {
                    cache = new GroupMemberEndpointCache(ep.Value);
                }
                myEndpoints[ep.Key] = cache;
            }

            _deliverableEndpoints = myEndpoints;
            DeliverableEndpoints = _deliverableEndpoints.Values.ToArray();
            _connectedEndpoints = myEndpoints.Where(kvp => kvp.Value.Endpoint.IsConnected).ToDictionary(kvp => kvp.Key, kvp => kvp.Value);
            ConnectedEndpoints = _connectedEndpoints.Values.ToArray();

            if (updateActivity && !wasConnected && IsConnected && DateTime.UtcNow.Subtract(Member.DateLastActive) > TimeSpan.FromMinutes(5))
            {
                Logger.Trace("Group member has connected. Saving member activity...");
                SaveActivity(group);
            }
        }        

        public bool IsConnected
        {
            get { return _connectedEndpoints != null && _connectedEndpoints.Count > 0; }
        }

        public void MarkAsRead(Guid groupID, DateTime timestamp)
        {
            lock (_syncRoot)
            {
                GroupMessageState state;

                // Mark the current state as read
                if (_dateMessagedByGroupID.TryGetValue(groupID, out state))
                {
                    state.MarkAsRead(timestamp);
                }
                else
                {
                    _dateMessagedByGroupID.Add(groupID, GroupMessageState.CreateForRead(groupID, timestamp));
                }
            }

        }

        public void UpdateDateMessaged(Guid groupID, DateTime timestamp)
        {
            lock (_syncRoot)
            {
                GroupMessageState state;
                if (_dateMessagedByGroupID.TryGetValue(groupID, out state))
                {
                    _dateMessagedByGroupID[groupID].Increment(timestamp);
                }
                else
                {
                    _dateMessagedByGroupID.Add(groupID, GroupMessageState.CreateForMessaged(groupID, timestamp));

                }
            }
        }

        public void Bump(Guid groupID, DateTime timestamp)
        {
            lock (_syncRoot)
            {
                GroupMessageState state;

                if (_dateMessagedByGroupID.TryGetValue(groupID, out state))
                {
                    _dateMessagedByGroupID[groupID].Bump(timestamp);
                }
                else
                {
                    _dateMessagedByGroupID.Add(groupID, GroupMessageState.CreateForMessaged(groupID, timestamp));
                }
            }
        }

        private class GroupMessageState
        {
            private DateTime? _dateRead;
            public DateTime? DateRead
            {
                get { return _dateRead; }
            }


            private DateTime? _dateMessaged;
            public DateTime? DateMessaged
            {
                get { return _dateMessaged; }
            }

            private int _unreadCounter = 0;
            public int UnreadCounter
            {
                get { return _unreadCounter; }
            }

            private readonly Guid _groupID;

            public Guid GroupID
            {
                get { return _groupID; }
            }

            internal static GroupMessageState CreateForRead(Guid groupID, DateTime dateRead)
            {
                return new GroupMessageState(groupID, dateRead);
            }

            internal static GroupMessageState CreateForMessaged(Guid groupID, DateTime dateMessaged)
            {
                return new GroupMessageState(groupID, dateMessaged, 1);
            }

            private GroupMessageState(Guid groupID, DateTime dateMessaged, int counter)
            {
                _groupID = groupID;
                _dateMessaged = dateMessaged;
                _unreadCounter = counter;
            }

            private GroupMessageState(Guid groupID, DateTime dateRead)
            {
                _groupID = groupID;
                _dateRead = dateRead;
                _unreadCounter = 0;
            }

            public void Increment(DateTime timestamp)
            {
                Interlocked.Increment(ref _unreadCounter);
                _dateMessaged = timestamp;
            }

            public void Bump(DateTime timestamp)
            {
                _dateMessaged = timestamp;
            }

            public void MarkAsRead(DateTime timestamp)
            {
                Interlocked.Exchange(ref _unreadCounter, 0);
                _unreadCounter = 0;
                _dateRead = timestamp;
            }

        }

        public void SaveChanges()
        {
            GroupMessageState[] states;

            lock (_syncRoot)
            {
                // Update the date messages for each channel
                states = _dateMessagedByGroupID.Values.ToArray();
                _dateMessagedByGroupID.Clear();
            }

            foreach (var state in states)
            {
                try
                {
                    if (state.UnreadCounter > 0 && state.DateMessaged.HasValue)
                    {
                        GroupMember.MarkAsUnread(state.GroupID, Member.UserID, state.DateMessaged.Value,
                            state.UnreadCounter);
                    }
                    else if (state.UnreadCounter == 0)
                    {
                        if (state.DateMessaged.HasValue && state.DateRead.HasValue)
                        {
                            GroupMember.MarkAsRead(state.GroupID, Member.UserID, state.DateMessaged.Value,
                                state.DateRead.Value);
                        }
                        else if (state.DateRead.HasValue)
                        {
                            GroupMember.MarkAsRead(state.GroupID, Member.UserID, null, state.DateRead.Value);
                        }
                    }
                }
                catch (AerospikeOperationException ex)
                {
                    Logger.Warn(ex, "Failed to process date messaged for group member", new { Member.UserID, Member.RegionID });
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to process date messaged for group member", new { Member.UserID, Member.RegionID });
                }
            }

        }

        public void SaveActivity(Group group)
        {
            Member.DateLastActive = DateTime.UtcNow;
            Member.Update(UpdateMode.Fast, p => p.DateLastActive);
            GroupMemberIndexWorker.CreateMembersActivity(group, new[] { Member.UserID }, Member.DateLastActive);
        }

        public void UpdateDateMessageSent()
        {
            _dateMessageSent = DateTime.UtcNow;
        }

        public bool CanSendMessage(TimeSpan throttle, out long? retryAfter)
        {
            var timeBetweenSends = DateTime.UtcNow - _dateMessageSent;
            var diff = (long)(throttle - timeBetweenSends).TotalMilliseconds;
            if (diff <= 0)
            {
                retryAfter = null;
                return true;
            }

            retryAfter = diff;
            return false;
        }

        public void UpdateExternalAccount(ExternalAccount externalAccount)
        {
            if (externalAccount !=null && externalAccount.MappedUsers.Contains(Member.UserID))
            {
                _externalAccount = externalAccount;
            }
            else
            {
                _externalAccount = null;
            }
            _dateExternalAccountRefreshed = DateTime.UtcNow;
        }

        public ExternalAccount GetExternalAccount(bool alwaysReturnCached = false)
        {
            if (alwaysReturnCached || DateTime.UtcNow - _dateExternalAccountRefreshed < TimeSpan.FromMinutes(5))
            {
                return _externalAccount;
            }

            var stats = UserStatistics.GetLocal(Member.UserID);
            if (stats == null)
            {
                return _externalAccount;
            }

            _dateExternalAccountRefreshed = DateTime.UtcNow;
            if (string.IsNullOrEmpty(stats.TwitchID))
            {
                _externalAccount = null;
            }
            else
            {
                _externalAccount = ExternalAccount.GetLocal(stats.TwitchID, AccountType.Twitch);
            }

            return _externalAccount;
        }
    }
}
