﻿using System;
using Curse.Friends.Data;
using System.Collections.Generic;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Logging;
using System.Threading.Tasks;
using Curse.Aerospike;
using Curse.Diagnostics;

namespace Curse.Friends.GroupService
{
    public class GroupMembersStore
    {
        private readonly Dictionary<int, GroupMemberCache> _members;
        private readonly object _syncRoot = new object();
        private readonly Guid _groupID;
        private readonly string _groupTitle;

        private static readonly HashSet<string> GroupsOfInterest = new HashSet<string>
        {
            "b7210373-78c5-4e43-9227-5b40e51e9cc2", 
            "fc25ebc9-bf19-421e-bf64-e6f94942ed1d",
            "b1b84936-fc01-4578-a461-05fc774fa944",
            "b5e68c2d-c8b2-45d2-be17-c3148a4c9195",
            "181e284f-7282-4b0f-9db6-59552980218b",
            "445850f7-95fc-4c3b-b119-8fc3eb1a2661",
            "ce41dab9-a921-40b4-8953-be788e9b595b",
            "14dbb897-c190-459c-8fcb-9293b3a5aabb",
            "1d688f5b-21a2-408b-9912-cf00bf9cc639",
            "b169ca29-e483-4298-b26e-7f2d51d861d0"
        };

        public GroupMembersStore(Group group)
        {
            _groupID = group.GroupID;
            _groupTitle = group.Title;

            var allGroupMembers = group.GetAllMembers().Where(p => !p.IsDeleted).ToArray();
            var allGroupMemberIDs = allGroupMembers.Select(p => p.UserID).ToArray();
            var endpointsByUser = ClientEndpoint.MultiGetAllForUserIDs(allGroupMemberIDs);
            var twitchAccountsByUser = ExternalAccount.MultiGetAllTwitchAccountsByUserIDs(allGroupMemberIDs);

            _members = new Dictionary<int, GroupMemberCache>(allGroupMembers.Length);

            foreach (var member in allGroupMembers)
            {
                _members.Add(member.UserID, new GroupMemberCache(group, member, endpointsByUser.GetValueOrDefault(member.UserID), twitchAccountsByUser.GetValueOrDefault(member.UserID)));
            }

            if (GroupsOfInterest.Contains(group.GroupID.ToString()))
            {
                Logger.Info("Created group member store for group of interest: " + group.Title, new { MemberCount = _members.Count, EndpointCount = endpointsByUser.Count });
            }
        }

        public int Count
        {
            get
            {
                lock (_syncRoot)
                {
                    return _members.Count;
                }
            }
        }

        public int ActiveCount
        {
            get
            {
                lock (_syncRoot)
                {
                    return _members.Count(p => p.Value.IsConnected || GroupMember.DetermineActive(p.Value.Member.DateLastActive));
                }
            }
        }

        public void Remove(int userID)
        {
            lock (_syncRoot)
            {
                _members.Remove(userID);
            }
        }

        public bool Add(GroupMemberCache memberCache)
        {
            lock (_syncRoot)
            {
                if (_members.ContainsKey(memberCache.Member.UserID))
                {
                    return false;
                }

                _members.Add(memberCache.Member.UserID, memberCache);
            }

            return true;
        }

        public GroupMemberCache Get(int userID)
        {
            GroupMemberCache member;

            lock (_syncRoot)
            {
                if (!_members.TryGetValue(userID, out member))
                {
                    return null;
                }
            }

            return member;
        }


        public bool CheckMemberPermission(int userID, GroupCache groupCache, GroupPermissions permission)
        {
            var member = Get(userID);

            if (member == null)
            {
                return false;
            }

            return groupCache.HasPermission(permission, member.Member);
        }

        public IReadOnlyCollection<GroupMemberCache> GetAllMembersWithPermission(GroupCache group, GroupPermissions requiredPermissions)
        {
            var members = new List<GroupMemberCache>();

            lock (_syncRoot)
            {
                foreach (var kvp in _members)
                {
                    if(group.HasPermission(requiredPermissions, kvp.Value.Member))
                    {
                        members.Add(kvp.Value);
                    }
                }
            }

            return members;
        }

        public GroupMemberCache[] GetAll()
        {
            lock (_syncRoot)
            {
                return _members.Values.ToArray();
            }
        }

        public void UpdateAllConnected()
        {            
            lock (_syncRoot)
            {
                foreach (var member in _members.Where(p => p.Value.IsConnected))
                {
                    member.Value.Member.DateLastActive = DateTime.UtcNow;                    
                }
            }            
        }

        public HashSet<int> GetConnectedUserIDs()
        {
            var activeMembers = new HashSet<int>();
            lock (_syncRoot)
            {
                foreach (var member in _members.Where(p => p.Value.IsConnected))
                {
                    activeMembers.Add(member.Key);
                }
            }

            return activeMembers;
        }

        public int ConnectedUserCount()
        {
            
            lock (_syncRoot)
            {
                return _members.Count(p => p.Value.IsConnected);                
            }
        }


        public HashSet<int> GetAllUserIDs()
        {
            var userIDs = new HashSet<int>();
            lock (_syncRoot)
            {
                foreach (var member in _members)
                {
                    userIDs.Add(member.Key);
                }
            }

            return userIDs;
        }

        public void SaveChanges()
        {
           
            using (var timer = new SmartStopwatch())
            {
                GroupMemberCache[] allMembers;

                using (timer.Child("Get Members"))
                {
                    allMembers = GetAll();
                }

                using (timer.Child("Save Changes"))
                {

                    Parallel.ForEach(allMembers, new ParallelOptions { MaxDegreeOfParallelism = 12 }, member =>
                    {
                        try
                        {
                            member.SaveChanges();
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex, "Failed to save group member changes");
                        }
                    });
                }

                var maxMilliseconds = allMembers.Length/5.0f;

                if (maxMilliseconds < 100)
                {
                    maxMilliseconds = 100;
                }

                timer.Stop();
                
                if (timer.Elapsed.TotalMilliseconds > maxMilliseconds)
                {
                    Logger.Warn("Saving group member changes has taken " + timer.Elapsed.TotalSeconds.ToString("F2") + " seconds", new { GroupID = _groupID, GroupTitle = _groupTitle, MemberCount = allMembers.Length });
                }
            }


        }
    }
}
