﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;
using System.Threading;
using Curse.Aerospike;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Logging;
using Curse.Friends.Statistics;
using Curse.Logging.Uploader;
using System.Threading.Tasks;
using Timer = System.Timers.Timer;

namespace Curse.Friends.GroupService
{
    public static class GroupServer
    {
        private static readonly LogCategory Logger = new LogCategory("GroupServer") { AlphaLevel = LogLevel.Trace };
        private static readonly ConcurrentDictionary<Guid, GroupSession> _groupSessions =
            new ConcurrentDictionary<Guid, GroupSession>();

        private static volatile bool _isStarted = true;

        private static Timer _periodicTaskTimer;
        private static Timer _instantTaskTimer;
        private static Timer _activityTimer;
        private static Timer _hourlyTasksTimer;
        private static Timer _tenMinsTimer;
        private static Timer _realtimeTaskTimer;
        private static Timer _chattersTimer;

        public static void Start()
        {

            int minThread, maxThreads, availThreads, minCompletionPorts, maxCompletionPorts, availCompletionPorts;
            ThreadPool.GetMaxThreads(out maxThreads, out maxCompletionPorts);
            ThreadPool.GetMinThreads(out minThread, out minCompletionPorts);

            ThreadPool.SetMaxThreads(Environment.ProcessorCount * 200, Environment.ProcessorCount * 400);
            ThreadPool.SetMinThreads(Environment.ProcessorCount * 10, Environment.ProcessorCount * 200);

            ThreadPool.GetAvailableThreads(out availThreads, out availCompletionPorts);


            StartQueueProcessors();

            // Realtime Tasks
            _realtimeTaskTimer = new Timer { Interval = TimeSpan.FromSeconds(1).TotalMilliseconds };
            _realtimeTaskTimer.Elapsed += (sender, args) => DoRealtimeTasks();
            _realtimeTaskTimer.Start();

            // Instant Tasks
            _instantTaskTimer = new Timer { Interval = TimeSpan.FromSeconds(5).TotalMilliseconds };
            _instantTaskTimer.Elapsed += (sender, args) => DoInstantTasks();
            _instantTaskTimer.Start();

            // Periodic Tasks
            _periodicTaskTimer = new Timer { Interval = TimeSpan.FromSeconds(30).TotalMilliseconds };
            _periodicTaskTimer.Elapsed += (sender, args) => DoPeriodicTasks();
            _periodicTaskTimer.Start();

            // Hourly Tasks
            _hourlyTasksTimer = new Timer { Interval = TimeSpan.FromHours(1).TotalMilliseconds };
            _hourlyTasksTimer.Elapsed += (sender, args) => DoHourlyTasks();
            _hourlyTasksTimer.Start();

            // Ten Min Tasks
            _tenMinsTimer = new Timer { Interval = TimeSpan.FromMinutes(10).TotalMilliseconds };
            _tenMinsTimer.Elapsed += (sender, args) => DoTenMinTasks();
            _tenMinsTimer.Start();

            // Host Status
            _activityTimer = new Timer { Interval = TimeSpan.FromSeconds(15).TotalMilliseconds };
            _activityTimer.Elapsed += (sender, args) => UpdateHostActivity();
            _activityTimer.Start();

            // Chatters list, needs to occur more often than every 5 minutes
            _chattersTimer = new Timer {Interval = TimeSpan.FromMinutes(1).TotalMilliseconds};
            _chattersTimer.Elapsed += (sender, args) => UpdateChatters();
            _chattersTimer.Start();

            RegisterHost();

            FriendsStatsManager.Started();
        }

        private static void UpdateChatters()
        {
            _chattersTimer.Stop();
            try
            {
                var chatters = new Dictionary<string, List<string>>();
                foreach (var session in _groupSessions.Values)
                {
                    var sessionChatters = session.GetChatters();
                    foreach (var channel in sessionChatters)
                    {
                        List<string> channelChatters;
                        if (chatters.TryGetValue(channel.Key, out channelChatters))
                        {
                            channelChatters.AddRange(channel.Value);
                        }
                        else
                        {
                            chatters[channel.Key] = channel.Value;
                        }
                    }
                }

                if (chatters.Count == 0)
                {
                    return;
                }

                Logger.Debug("Chatters", new { chatters });
                ChattersListWorker.Create(chatters.ToDictionary(kvp => kvp.Key, kvp => kvp.Value.ToArray()));
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                _chattersTimer.Start();
            }
        }

        private static void DoRealtimeTasks()
        {
            _realtimeTaskTimer.Stop();
            try
            {
                var allSessions = _groupSessions.Values;

                foreach (var session in allSessions)
                {
                    try
                    {
                        session.SaveRealtimeData();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex);
                    }

                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                _realtimeTaskTimer.Start();
            }
        }

        private static readonly LogCategory InstantTasksLogger = new LogCategory("InstantTasks") { Throttle = TimeSpan.FromMinutes(1) };

        private static void DoInstantTasks()
        {
            _instantTaskTimer.Stop();

            try
            {
                var allSessions = _groupSessions.Values.ToArray();

                var sw = Stopwatch.StartNew();
                InstantTasksLogger.Info("Saving periodic data...", new { TotalSessions = allSessions.Length });

                var tasks = allSessions.Select(p => Task.Factory.StartNew(() =>
                {
                    if (!_isStarted)
                    {
                        return;
                    }
                    p.SavePeriodicData();

                })).ToArray();

                Task.WaitAll(tasks);

                //var result = Parallel.ForEach(allSessions, (session) =>
                //{
                //    if (!_isStarted)
                //    {                        
                //        return;
                //    }

                //    try
                //    {
                //        session.SavePeriodicData();
                //    }
                //    catch (Exception ex)
                //    {
                //        Logger.Error(ex);
                //    }
                //});                

                sw.Stop();

                if (sw.Elapsed.TotalSeconds > 1)
                {
                    InstantTasksLogger.Info("Completed saving periodic data!", new { TotalSessions = allSessions.Length, TimeTaken = sw.Elapsed.TotalSeconds.ToString("F2") + " seconds" });
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                _instantTaskTimer.Start();
            }
        }

        private static void DoPeriodicTasks()
        {
            _periodicTaskTimer.Stop();
            try
            {
                FixRehostedGroups();
                SyncMemberIndex();
                CheckGroupIntegrity();
                FixOrphanedGroups();
                CleanupIdleGroups();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                _periodicTaskTimer.Start();
            }

        }

        private static void DoTenMinTasks()
        {
            _tenMinsTimer.Stop();
                              
            try
            {
                
                var allSessions = _groupSessions.Values;

                foreach (var session in allSessions)
                {
                    try
                    {
                        session.SaveTenMinsData();
                        session.RefreshAllGroups();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex);
                    }

                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                _tenMinsTimer.Start();
            }
        }

        /// <summary>
        /// Removes idle groups from inmemory cache, called every 30seconds.
        /// </summary>
        private static void CleanupIdleGroups()
        {
            // Iterate over all groups, and find those that are inctive
            try
            {
                var inactiveGroups = _groupSessions.Where(p => !p.Value.IsActive).Select(p => p.Key).ToArray();

                foreach (var groupID in inactiveGroups)
                {
                    GroupSession removed;
                    if (_groupSessions.TryGetValue(groupID, out removed))
                    {
                        if (removed.IsActive)
                        {
                            Logger.Trace("Group session has become active again, while cleaning up idle groups.");
                            continue;
                        }
                        //remove the group from inmemory cache for data consistency
                        if (_groupSessions.TryRemove(groupID, out removed))
                        {
                            Logger.Debug("Shutting down idle group: " + removed.Group.Title);
                            removed.Shutdown();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

        }

        /// <summary>
        /// Checks the integrity for any groups that need it, up to once per hour.
        /// </summary>
        private static void CheckGroupIntegrity()
        {
            // Iterate over all groups, and find those that are due for an integrity check
            try
            {
                var stableGroups = _groupSessions.Values.Where(p => p.IsStable && p.IsPendingIntegrityCheck).ToArray();

                foreach (var groupSession in stableGroups)
                {
                    Logger.Trace("Performing integrity check for group: " + groupSession.Group.Title);
                    groupSession.DateIntegrityChecked = DateTime.UtcNow;
                    GroupIntegrityWorker.Create(groupSession.Group);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

        }

        private static void SyncMemberIndex()
        {
            try
            {
                var groups = _groupSessions.Values.Where(p => p.IsPendingMemberIndexSync).ToArray();

                foreach (var groupSession in groups)
                {
                    groupSession.DateMemberIndexSynced = DateTime.UtcNow;
                    Logger.Debug("Performing full member index for group:" + groupSession.Group.Title);
                    GroupMemberIndexWorker.CreateFullSync(groupSession.Group);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
        }

        /// <summary>
        /// Updates the DateUpdated timestamp, so that the host can be properly consider online or offline.
        /// </summary>
        private static void UpdateHostActivity()
        {
            try
            {
                if (_host == null || _host.Status == ServiceHostStatus.Offline)
                {
                    return;
                }

                _host.DateUpdated = DateTime.UtcNow;
                _host.Status = ServiceHostStatus.Online;
                _host.Update(p => p.Status, p => p.DateUpdated);
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

        }


        /// <summary>
        /// Identify and shutdown sessions for groups that have a new host
        /// </summary>
        private static void FixRehostedGroups()
        {
            try
            {
                var groups = _groupSessions.Values.ToArray();

                foreach (var groupSession in groups)
                {                    
                    if (groupSession.Group == null)
                    {
                        continue;                        
                    }
                    
                    var groupID = groupSession.Group.GroupID;
                    var group = Group.GetLocal(groupID);

                    if (group == null)
                    {
                        Logger.Warn("Failed to retrieve server from database while fixing rehosted groups: " + groupID);
                        continue;
                    }

                    if (group.MachineName == null || !group.MachineName.Equals(Environment.MachineName))
                    {                      
                        Logger.Warn("A group session for '" + group.Title + "' has been rehosted on another machine '" + group.MachineName + "'. This session will be removed.");
                        
                        GroupSession removed;
                        if (!_groupSessions.TryRemove(group.GroupID, out removed))
                        {
                            Logger.Warn("Failed to remove group session.", new { GroupID = group.GroupID, GroupTitle = group.Title });
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
        }


        private static void FixOrphanedGroups()
        {
            var badGroupHosts = GroupHost.GetAllLocal(p => p.IndexMode, IndexMode.Default).Where(p => p.Status == ServiceHostStatus.Online
                && (DateTime.UtcNow - p.DateUpdated > TimeSpan.FromMinutes(2))
                && !p.IsCleaning && !p.MachineName.Equals(Environment.MachineName)).ToArray();

            try
            {
                // Quickly mark all of that as unhealthy
                foreach (var groupHost in badGroupHosts)
                {
                    Logger.Warn("Found unhealthy group host: " + groupHost.MachineName);
                    groupHost.Status = ServiceHostStatus.Unhealthy;
                    groupHost.DateUnhealthy = DateTime.UtcNow;
                    groupHost.IsCleaning = true;
                    groupHost.Update(p => p.Status, p => p.DateUnhealthy, p => p.IsCleaning);
                }

                // Get all groups hosted by these hosts, and fail them over
                foreach (var groupHost in badGroupHosts)
                {
                    var orphanedGroups = Group.GetAllLocal(p => p.MachineName, groupHost.MachineName);

                    Logger.Warn("Found " + orphanedGroups.Length.ToString("###,##0") + " orphaned groups on host: " + groupHost.MachineName);

                    foreach (var group in orphanedGroups)
                    {
                        group.MachineName = null;
                        group.Update(p => p.MachineName);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                foreach (var groupHost in badGroupHosts)
                {
                    groupHost.IsCleaning = false;
                    groupHost.Update(p => p.IsCleaning);
                }
            }
        }

        private static void DoHourlyTasks()
        {
            _hourlyTasksTimer.Stop();
            try
            {
                ProcessGroupInvitations();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                _hourlyTasksTimer.Start();
            }
        }

        private static void ProcessGroupInvitations()
        {
            foreach (var sessionKvp in _groupSessions)
            {
                try
                {
                    sessionKvp.Value.ProcessGroupInvitations();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, string.Format("Error while purging expired guests from group {0}", sessionKvp.Key));
                }
            }
        }

        private static void StartQueueProcessors()
        {
            GroupMessageCordinator.StartProcessor(GroupMessageCordinator_ProcessMessage);
            GroupChangeCoordinator.StartProcessor(GroupChangeCoordinator_ProcessMessage);
            GroupEndpointCoordinator.StartProcessor(GroupEndpointCoordinator_ProcessMessage);
            GroupMessageChangeCoordinator.StartProcessor(GroupMessageChangeCordinator_ProcessMessage);
            GroupPrivateMessageCoordinator.StartProcessor(GroupPrivateMessageChangeCordinator_ProcessMessage);
            GroupPollChangedCoordinator.StartProcessor(GroupPollChangedNotifier_ProcessMessage);
            GroupGiveawayNotificationCoordinator.StartProcessor(GroupGiveawayNotifier_ProcessMessage);
            GroupMessageLikeCoordinator.StartProcessor(GroupMessageLikeCoordinator_ProcessMessage);
            GroupMessageReadCoordinator.StartProcessor(GroupMessageReadCoordinator_ProcessMessage);
            GroupCallCoordinator.StartProcessor(GroupCallCoordinator_ProcessMessage);
            ExternalCommunityLinkChangedCoordinator.StartProcessor(ExternalCommunityMappingCoordinator_ProcessMessage);
            ExternalMessageCoordinator.StartProcessor(ExternalMessageCoordinator_ProcessMessage);
            GroupMemberTwitchAccountCoordinator.StartProcessor(GroupMemberTwitchAccountCoordinator_ProcessMessage);
            TwitchChatNoticeCoordinator.StartProcessor(TwitchChatNoticeCoordinator_ProcessMessage);
            GroupBulkMessageDeleteCoordinator.StartProcessor(GroupBulkMessageDeleteCoordinator_ProcessMessage);
        }        

        public static void Stop()
        {
            try
            {
                FriendsStatsManager.BeginShutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

            _isStarted = false;

            try
            {
                if (_periodicTaskTimer != null)
                {
                    _periodicTaskTimer.Stop();
                }

                if (_activityTimer != null)
                {
                    _activityTimer.Stop();
                }

                if (_hourlyTasksTimer != null)
                {
                    _hourlyTasksTimer.Stop();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to stop periodic timers.");
            }

            try
            {
                MarkHostOffline();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to mark host as offline.");
            }

            try
            {
                ShutdownSessions();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to shut down sessions");
            }

            try
            {

                StorageConfiguration.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to shutdown storage configuration.");
            }

            try
            {
                LogUploader.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to finish log upload!");
            }

            try
            {

                FriendsStatsManager.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

        }

        private static void ShutdownSessions()
        {
            // Shutdown all hosted group sessions  

            Logger.Info("Shutting down " + _groupSessions.Count + " group sessions...");
            foreach (var groupSession in _groupSessions.Values)
            {
                groupSession.Shutdown();
            }
            Logger.Info("Completed shutting down group sessions!");
        }

        private static GroupHost _host;

        /// <summary>
        /// Marks this GroupServer as offline so no new groups will be asiigned to be processed
        /// by this instance of server.
        /// </summary>
        private static void MarkHostOffline()
        {
            if (_host == null)
            {
                return;
            }

            _host.Status = ServiceHostStatus.Offline;
            _host.Update(p => p.Status);
        }

        /// <summary>
        /// Registers every groupserver on startup as a GroupHost to Aerospike
        /// This helps in updating the mahcine name of the group, when group has no
        /// routed information regarding the GroupServers
        /// Look in GroupHostManager.cs
        /// </summary>
        private static void RegisterHost()
        {
#if CONFIG_LOADTESTING
            const HostEnvironment hostEnvironment = HostEnvironment.Release;
#else
            var hostEnvironment = (HostEnvironment)Enum.Parse(typeof(HostEnvironment), FriendsServiceConfiguration.Mode.ToString(), true);
#endif
            var internalIp = NetworkHelper.GetInternalIpAddress();
            var host = GroupHost.GetLocal(internalIp);

            if (host == null)
            {
                host = new GroupHost
                {
                    InternalIPAddress = internalIp,
                    DateCreated = DateTime.UtcNow,
                    DateOnline = DateTime.UtcNow,
                    DateUpdated = DateTime.UtcNow,
                    MachineName = Environment.MachineName,
                    Status = ServiceHostStatus.Online,
                    Environment = hostEnvironment,

                };
                host.Version = host.CurrentVersion;
                //Inserts to Aerospike local DB
                host.InsertLocal();
            }
            else
            {
                host.DateUpdated = DateTime.UtcNow;
                host.DateOnline = DateTime.UtcNow;
                host.Status = ServiceHostStatus.Online;
                host.Environment = hostEnvironment;
                host.Version = host.CurrentVersion;
                host.Update();
            }

            _host = host;

        }

        private static readonly ConcurrentDictionary<Guid, object> _createSessionLocks = new ConcurrentDictionary<Guid, object>();

        private static GroupSession GetOrAddSession(Guid groupGuid, out bool retrievedFromCache)
        {
            GroupSession session;
            if (_groupSessions.TryGetValue(groupGuid, out session))
            {
                session.LastActivity = DateTime.UtcNow;
                retrievedFromCache = true;
                return session;
            }            

            var createSession = _createSessionLocks.GetOrAdd(groupGuid, new object());
            var lockAcquired = false;
            Monitor.TryEnter(createSession, TimeSpan.FromSeconds(20), ref lockAcquired);

            try
            {
                if (!lockAcquired)
                {
                    throw new InvalidOperationException("Failed to acquire create session lock for group: " + groupGuid);
                }

                lock (createSession)
                {
                    if (_groupSessions.TryGetValue(groupGuid, out session))
                    {
                        session.LastActivity = DateTime.UtcNow;
                        retrievedFromCache = true;
                        return session;
                    }

                    retrievedFromCache = false;
                    return _groupSessions.GetOrAdd(groupGuid, guid =>
                    {
                        var group = Group.GetLocal(groupGuid);
                        if (group == null)
                        {
                            throw new InvalidOperationException("Unable to retrieve group!");
                        }

                        return new GroupSession(group);
                    });
                }
            }
            finally
            {
                if (lockAcquired)
                {
                    Monitor.Exit(createSession);
                }
            }

        }

        /// <summary>
        /// Sends information about any changes made to the Group to NotificationServer
        /// </summary>
        /// <param name="e"></param>
        private static void GroupChangeCoordinator_ProcessMessage(GroupChangeCoordinator e)
        {
            try
            {
                var retrievedFromCache = false;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);

                switch (e.ChangeType)
                {
                    case GroupChangeType.AddUsers:
                        session.AddMembers(e.SenderID, e.UserIDs, retrievedFromCache);
                        break;
                    case GroupChangeType.RemoveUsers:
                        session.RemoveMembers(e.SenderID, e.UserIDs, e.Reason, e.MessageToUsers, retrievedFromCache);
                        break;

                    case GroupChangeType.ChangeInfo:
                        session.ChangeInfo(e.SenderID);
                        break;

                    case GroupChangeType.VoiceSessionStarted:
                    case GroupChangeType.VoiceSessionEnded:
                    case GroupChangeType.VoiceSessionUserJoined:
                    case GroupChangeType.VoiceSessionUserLeft:
                        session.NotifyVoiceSessionChange(e.AffectedGroupIDs.First(), e.ChangeType, e.UserIDs);
                        break;

                    case GroupChangeType.UpdateUsers:
                        session.UpdateMembers(e.SenderID, e.UserIDs);
                        break;

                    case GroupChangeType.CreateGroup:
                        session.NotifyGroupCreation(e.AffectedGroupIDs.First(), e.SenderID, retrievedFromCache);
                        break;

                    case GroupChangeType.RemoveGroup:
                        session.NotifyGroupRemoval(e.AffectedGroupIDs.First(), e.SenderID);
                        break;

                    case GroupChangeType.GroupReorganized:
                        session.NotifyGroupRestructured(e.SenderID);
                        break;
                    case GroupChangeType.PermissionsChanged:
                        session.NotifyRolesChanged(e.SenderID);
                        break;
                    case GroupChangeType.RoleNamesChanged:
                        session.NotifyRolesChanged(e.SenderID);
                        break;
                    case GroupChangeType.UpdateEmoticons:
                        session.NotifyEmoticonsChanged(e.SenderID);
                        break;
                    case GroupChangeType.UpdateUserPresence:
                        session.UpdateMemberPresence(e.PresenceUpdate);
                        break;
                    default:
                        throw new NotImplementedException("Unknown change type: " + e.ChangeType);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to process group change!", e);
            }
        }

        private static void GroupPollChangedNotifier_ProcessMessage(GroupPollChangedCoordinator coordinator)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(coordinator.GroupID, out retrievedFromCache);
                session.NotifyPollChange(coordinator.Notification);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to process poll change!");
            }
        }

        private static void GroupGiveawayNotifier_ProcessMessage(GroupGiveawayNotificationCoordinator notifier)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(notifier.GroupID, out retrievedFromCache);
                session.NotifyGiveawayChanged(notifier.Notification, notifier.TargetUserID);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to process giveaway change!");
            }
        }

        /// <summary>
        /// Broadcasts messages to all the users that belongs to the group
        /// </summary>
        /// <param name="e"></param>
        private static void GroupMessageCordinator_ProcessMessage(GroupMessageCordinator e)
        {
            try
            {
                var retrievedFromCache = false;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.NotifyInstantMessage(e.TargetGroupID, e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process group message.", e);
            }

        }

        /// <summary>
        /// Broadcasts messages to all the users that belongs to the group
        /// </summary>
        /// <param name="e"></param>
        private static void GroupMessageLikeCoordinator_ProcessMessage(GroupMessageLikeCoordinator e)
        {
            try
            {
                var retrievedFromCache = false;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.ProcessMessageLike(e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process like.", e);
            }

        }

        private static void GroupMessageReadCoordinator_ProcessMessage(GroupMessageReadCoordinator e)
        {
            try
            {
                var retrievedFromCache = false;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.ProcessMessageRead(e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process message read.", e);
            }
        }

        /// <summary>
        /// Broadcasts messages to all the users that belongs to the group
        /// </summary>
        /// <param name="e"></param>
        private static void GroupMessageChangeCordinator_ProcessMessage(GroupMessageChangeCoordinator e)
        {
            try
            {
                var retrievedFromCache = false;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.NotifyChangedMessage(e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process group message.", e);
            }

        }

        /// <summary>
        /// Broadcasts messages to all the users that belongs to the group
        /// </summary>
        /// <param name="e"></param>
        private static void GroupPrivateMessageChangeCordinator_ProcessMessage(GroupPrivateMessageCoordinator e)
        {
            try
            {
                var retrievedFromCache = false;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.NotifyPrivateMessage(e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process group message.", e);
            }

        }

        /// <summary>
        /// Updates ClientEndPoints of the member ID for the groupID received
        /// </summary>
        /// <param name="e"></param>
        private static void GroupEndpointCoordinator_ProcessMessage(GroupEndpointCoordinator e)
        {
            try
            {
                GroupSession found;
                if (!_groupSessions.TryGetValue(e.GroupID, out found))
                {
                    return;
                }

                // Update the endpoints for this user
                found.UpdateMemberEndpoints(e.UserID, e.ConnectedEndpoints);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process group endpoint coordinator!", e);
            }
        }

        private static void GroupMemberTwitchAccountCoordinator_ProcessMessage(GroupMemberTwitchAccountCoordinator e)
        {
            try
            {
                GroupSession session;
                if (!_groupSessions.TryGetValue(e.GroupID, out session))
                {
                    return;
                }

                session.UpdateMemberAccount(e.UserID, e.ExternalAccount);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process external account coordinator!", e);
            }
        }

        private static void GroupCallCoordinator_ProcessMessage(GroupCallCoordinator e)
        {
            try
            {
                bool retrievedFromCache;
                var groupSession = GetOrAddSession(e.GroupID, out retrievedFromCache);

                switch (e.Type)
                {
                    case GroupCallCoordinatorType.CallResponded:
                        groupSession.NotifyCallResponded(e);
                        break;
                    case GroupCallCoordinatorType.CallStarted:
                        groupSession.NotifyCallStarted(e);
                        break;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process group call coordinator!", e);
            }
        }

        private static void ExternalCommunityMappingCoordinator_ProcessMessage(ExternalCommunityLinkChangedCoordinator e)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.ProcessCommunityMappingChanged(e.ExternalCommunity, e.Notification);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process ExternalCommunityMappingCoordinator", e);
            }
        }

        private static void ExternalMessageCoordinator_ProcessMessage(ExternalMessageCoordinator e)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.NotifyExternalMessage(e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process external message coordinator", e);
            }
        }

        private static void TwitchChatNoticeCoordinator_ProcessMessage(TwitchChatNoticeCoordinator e)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.NotifyTwitchNotice(e.NotificationTargets, e.Notification);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process Twitch Chat Notice Coordinator", e);
            }
        }

        private static void GroupBulkMessageDeleteCoordinator_ProcessMessage(GroupBulkMessageDeleteCoordinator e)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(e.GroupID, out retrievedFromCache);
                session.NotifyBulkMessageDelete(e.CreateNotification());
            }
            catch(Exception ex)
            {
                Logger.Error(ex, "Failed to process Group Bulk Message Delete Notifier");
            }
        }
    }
}
