﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.ServiceProcess;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Curse.Friends.Configuration;
using Curse.Logging;
using Curse.Friends.Statistics;
using Curse.Friends.Statistics.Models;
using Curse.Logging.Uploader;
using System.Reflection;
using Curse.CloudQueue;
using Curse.Friends.ServiceClients;

namespace Curse.Friends.GroupService
{
    public class GroupService : ServiceBase
    {
        public GroupService()
        {
            ServiceName = "CurseGroupService";
        }

        public void OnDebugStart()
        {
            OnStart(null);

        }

        public void OnDebugStop()
        {
            OnStop();
        }

        protected override void OnStart(string[] args)
        {
            var sw = Stopwatch.StartNew();
            EventLog.WriteEntry("Group Service Starting", EventLogEntryType.Information);

            Logger.Init(new LoggerConfig(@"C:\Curse\Logs") { LogRetainCount = 100, MaxLogSize = Int32.MaxValue });

            LogUploader.Initialize((int) ServiceHostType.FriendsGroupService, FriendsServiceConfiguration.Instance.CentralLogServiceUrl, FriendsServiceConfiguration.Instance.LoggingServiceApiKey);

            Logger.Info("Group Service Starting", new { BuildAge = Assembly.GetExecutingAssembly().GetBuildAge().Hours + " hours ago" });

            // Add the event handler for handling non-UI thread exceptions to the event. 
            AppDomain.CurrentDomain.UnhandledException += CurrentDomain_UnhandledException;
            ReallyStartService(sw);
        }

        IEnumerable<FriendsQueueStats> GetQueueStats()
        {
            return CloudQueueInstanceManager.GetQueueStats().Select(p => new FriendsQueueStats
            {
                QueueName = p.QueueName,
                TotalMessagesProcessed = p.TotalMessagesProcessed,
                CurrentlyProcessing = p.CurrentlyProcessing,
                AverageProcessingTimeMs = (int) p.AverageProcessTime.TotalMilliseconds,
                BestProcessingTimeMs = (int) p.BestProcessTime.TotalMilliseconds,
                WorstProcessingTimeMs = (int) p.WorstProcessTime.TotalMilliseconds,
                AverageQueuedTimeMs = (int) p.AverageQueuedTime.TotalMilliseconds,
                BestQueuedTimeMs = (int) p.BestQueuedTime.TotalMilliseconds,
                WorstQueuedTimeMs = (int) p.WorstQueuedTime.TotalMilliseconds
            }).ToArray();
        }

        void ReallyStartService(object o)
        {
            var sw = o as Stopwatch;
            try
            {
                Logger.Info("Determining current region...");
                var currentRegion = StorageConfiguration.GetCurrentRegion();

                if (currentRegion == null)
                {
                    Logger.Error("Failed to determine current region!");
                }
                else
                {
                    Logger.Info("Current region detected: " + currentRegion.Key);
                }

                Logger.Info("Initializing service client...");
                FriendsServiceClients.Initialize(FriendsServiceConfiguration.Instance.EncryptionKey, FriendsServiceConfiguration.Instance.EncryptionIterations, FriendsServiceConfiguration.Instance.CentralServiceApiKey);
                FriendsStatsManager.Initialize((int)ServiceHostType.FriendsGroupService, ServiceHostType.FriendsGroupService.ToString(), currentRegion != null ? currentRegion.ID : 1, GetQueueStats);    
                FriendsStatsManager.BeginStartup();

                EventLog.WriteEntry("Initializing Configuration", EventLogEntryType.Information);
                StorageConfiguration.Initialize("GroupService");
                EventLog.WriteEntry("Group Server Starting", EventLogEntryType.Information);                           
                GroupServer.Start();                
            }
            catch (Exception ex)
            {
                EventLog.WriteEntry("Failed to start: " + ex.Message, EventLogEntryType.Error);
                Logger.Error(ex, "Notification has failed to start, due to an unhandled exception.");
                throw;
            }

            EventLog.WriteEntry("Group Service Started", EventLogEntryType.Information);

            sw.Stop();
            Logger.Info("Group Service Started in " + sw.Elapsed.TotalSeconds.ToString("###,##0.00") + " seconds");            
        }

        void CurrentDomain_UnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            var exception = e.ExceptionObject as Exception;
            Logger.Error(exception, "Service has crashed!");
        }

        protected override void OnStop()
        {
            try
            {
                base.RequestAdditionalTime(30 * 1000);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to request additional time.");
            }

            try
            {
                GroupServer.Stop();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to cleanly stop service.");
            }

        }

    }
}
