﻿using System;
using System.Collections;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using Curse.Aerospike;
using Curse.Diagnostics;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Logging;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Models;
using Curse.Voice.Helpers;
using Group = Curse.Friends.Data.Group;

namespace Curse.Friends.GroupService
{
    /// <summary>
    /// Contains the details of all the users of the group and their Connection ClientEndPoints[].
    /// Also stores the information regarding the Group itself
    /// All the notifiers created in this class are dequeued and processed by notification Server.
    /// </summary>
    public class GroupSession
    {
        private readonly Group _group;

        private readonly LikeManager _likeManager = new LikeManager();
        private readonly GroupMembersStore _members;

        private static readonly LogCategory Logger = new LogCategory("GroupSession") { DebugLevel = LogLevel.Trace, AlphaLevel = LogLevel.Trace, BetaLevel = LogLevel.Info, ReleaseLevel = LogLevel.Info };

        private readonly ConcurrentDictionary<Guid, GroupCache> _groupCache = new ConcurrentDictionary<Guid, GroupCache>();

        private volatile bool _membersChanged = false;

        private static readonly GroupMemberCache[] EmptyMemberList = new GroupMemberCache[0];

        private IReadOnlyCollection<GroupMemberCache> GetAllMembersWithPermission(Guid groupID, GroupPermissions permission)
        {
            var groupCache = GetGroup(groupID);

            if (groupCache == null)
            {
                return EmptyMemberList;
            }

            return _members.GetAllMembersWithPermission(groupCache, permission);
        }

        private bool CheckMemberPermission(Guid groupID, int userID, GroupPermissions permission)
        {            
            GroupCache groupCache;
            if (!_groupCache.TryGetValue(groupID, out groupCache))
            {
                return false;
            }

            return _members.CheckMemberPermission(userID, groupCache, permission);
        }

        public DateTime DateSessionStarted { get; set; }

        private long _lastActivity;
        public DateTime LastActivity
        {
            get { return DateTime.FromBinary(Interlocked.Read(ref _lastActivity)); }
            set { Interlocked.Exchange(ref _lastActivity, value.ToBinary()); }
        }

        private static readonly TimeSpan IdleTimeout = TimeSpan.FromMinutes(10);

        public bool IsActive
        {
            get
            {
                // Never let super large communities get shutdown!
                if (_members.Count > 100)
                {
                    return true;
                }

                if (_group.Type == GroupType.Large && _members.ConnectedUserCount() > 1)
                {
                    return true;
                }                

                return DateTime.UtcNow - LastActivity <= IdleTimeout;
            }
        }

        private static readonly TimeSpan StableDuration = TimeSpan.FromMinutes(1);

        public bool IsStable
        {
            get { return DateTime.UtcNow - DateSessionStarted >= StableDuration; }
        }

        private static readonly TimeSpan IntegrityCheckThrottle = TimeSpan.FromHours(1);

        public bool IsPendingIntegrityCheck
        {
            get { return DateTime.UtcNow - DateIntegrityChecked >= IntegrityCheckThrottle && _group.Type == GroupType.Large; }
        }

        public DateTime DateIntegrityChecked
        {
            get;
            set;
        }

        private static readonly TimeSpan MemberIndexSyncThrottle = TimeSpan.FromHours(Group.MemberIndexSyncIntervalHours);

        public bool IsPendingMemberIndexSync { get { return DateTime.UtcNow - DateMemberIndexSynced >= MemberIndexSyncThrottle; } }

        public DateTime DateMemberIndexSynced { get; set; }

        private static readonly TimeSpan MemberActivitySyncThrottle = TimeSpan.FromMinutes(5);
        public DateTime DateMemberActivitySaved { get; set; }

        public bool IsPendingMemberActivitySync { get { return DateTime.UtcNow - DateMemberActivitySaved >= MemberActivitySyncThrottle; } }


        public Group Group
        {
            get { return _group; }
        }

        public GroupCache RootGroupCache
        {
            get { return GetGroup(_group.GroupID); }
        }

        public void Shutdown()
        {
            try
            {
                SavePeriodicData();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

            try
            {
                SaveTenMinsData();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

            _group.MachineName = null;
            _group.Update(p => p.MachineName);
        }

        public GroupSession(Group group)
        {
            using (var timer = new SmartStopwatch())
            {
                DateSessionStarted = DateTime.UtcNow;
                DateIntegrityChecked = group.DateChecked;
                LastActivity = DateTime.UtcNow;
                _group = group;

                if (_group.RegionID != StorageConfiguration.CurrentRegion.ID)
                {
                    Logger.Warn("Attempt made to start a group session outside of the group's home region!");
                    throw new InvalidOperationException();
                }       

                using (timer.Child("Members Cache"))
                {
                    // Create a member storem which contains all members
                    _members = new GroupMembersStore(_group);
                }

                using (timer.Child("Group Store"))
                {
                    // Add the root access level
                    _groupCache.TryAdd(_group.GroupID, new GroupCache(group, _group));

                    // Get access levels for all sub groups in this group
                    if (group.CanHaveChildren)
                    {                        
                        var allGroups = group.FastGetRootChildren();

                        foreach (var subGroup in allGroups)
                        {
                            _groupCache.TryAdd(subGroup.GroupID, new GroupCache(subGroup, _group));
                        }

                        Logger.Trace("Completed group store for " + allGroups.Length + " child groups.");
                    }                                       
                }            

                DateMemberIndexSynced = group.DateMembersIndexed;
                DateMemberActivitySaved = group.DateMemberActivitySaved;

                timer.Stop();

                if (_group.Type == GroupType.Large)
                {
                    if (_group.MemberCount > 50)
                    {
                        Logger.Debug("Started new group session for: " + _group.Title, new { Took = timer.Elapsed.TotalMilliseconds.ToString("F2") + "ms", MemberCount = _members.Count, Timers = timer.Results });
                    }
                    else if (_group.MemberCount > 500)
                    {
                        Logger.Info("Started new group session for: " + _group.Title, new { Took = timer.Elapsed.TotalMilliseconds.ToString("F2") + "ms", MemberCount = _members.Count, Timers = timer.Results });
                    }
                    
                }                
            }
            
            Task.Factory.StartNew(() => SaveMemberActivity(true));
        }


        internal void DispatchNotification(Guid targetGroupID, GroupPermissions requiredPermission, HashSet<int> excludeUserIDs, Action<GroupMemberEndpointCache> action)
        {
            var members = GetAllMembersWithPermission(targetGroupID, requiredPermission).Where(member => !excludeUserIDs.Contains(member.Member.UserID)).ToArray();

            if (!members.Any())
            {
                return;
            }

            foreach (var member in members)
            {
                foreach (var ep in member.ConnectedEndpoints)
                {
                    action(ep);
                }
            }
        }


        internal static void DispatchNotification(IEnumerable<GroupMemberCache> members, Action<int, string, HashSet<string>> action, Func<GroupMemberEndpointCache, bool> inclusionRule = null)
        {
            var regionalMachineSessions = new Dictionary<int, Dictionary<string, HashSet<string>>>();
            // Build up a list of distinct session IDs, group by region and server
            foreach (var member in members)
            {
                foreach (var epCache in member.ConnectedEndpoints)
                {
                    var ep = epCache.Endpoint;
                    if (ep.RegionID == 0 || string.IsNullOrEmpty(ep.ServerName) || string.IsNullOrEmpty(ep.SessionID))
                    {
                        continue;
                    }

                    if (inclusionRule != null && !inclusionRule(epCache))
                    {
                        continue;
                    }

                    if (!regionalMachineSessions.ContainsKey(ep.RegionID))
                    {
                        regionalMachineSessions.Add(ep.RegionID, new Dictionary<string, HashSet<string>>());
                    }

                    var region = regionalMachineSessions[ep.RegionID];

                    if (!region.ContainsKey(ep.ServerName))
                    {
                        region.Add(ep.ServerName, new HashSet<string>());
                    }

                    var machine = region[ep.ServerName];

                    machine.Add(ep.SessionID);
                }
            }

            foreach (var regionalMachineSession in regionalMachineSessions)
            {
                var regionID = regionalMachineSession.Key;
                foreach (var machineSession in regionalMachineSession.Value)
                {
                    var serverName = machineSession.Key;
                    var sessionIDs = machineSession.Value;

                    if (sessionIDs.Count > 500)
                    {
                        foreach (var set in sessionIDs.InHashSetsOf(500))
                        {
                            action(regionID, serverName, set);
                        }
                    }
                    else
                    {
                        action(regionID, serverName, sessionIDs);                        
                    }                    
                }
            }
        }

        private void DispatchGroupChangeNotification(GroupChangeNotification notification)
        {
            var allMembers = _members.GetAll();

            // Notify all members that are connected of this event
            DispatchNotification(allMembers, (regionID, serverName, sessionIDs) =>
            {
                GroupMultiSessionChangeNotifier.Create(notification, sessionIDs, serverName, regionID);
            });
        }

        private void DispatchPresenceNotification(GroupPresenceNotification notification)
        {
            var allMembers = _members.GetAll();

            // Notify all members that are connected of this event
            DispatchNotification(allMembers, (regionID, serverName, sessionIDs) =>
            {
                GroupPresenceNotifier.Create(notification, sessionIDs, serverName, regionID);
            }, endpoint =>
            {
                var version = endpoint.Endpoint.GetVersion();
                return version.Major >= 7 && version.Build > 30;
            });
        }

        /// <summary>
        /// Processes invalid 
        /// </summary>
        public void ProcessGroupInvitations()
        {
            _group.InvalidateExpiredInvitations();
            _group.ProcessInvalidatedInvitations();            
        }


        private DateTime _lastMemberCountUpdate;

        public void SavePeriodicData()
        {
            var updateCounts = _membersChanged;
            _membersChanged = false;

            var membersOnline = _members.ActiveCount;
            if (Group.MembersOnline != membersOnline)
            {
                Group.MembersOnline = membersOnline;                
                Group.Update(g => g.MemberCount, g => g.MembersOnline);
            }
            
            if (updateCounts)
            {            
                Group.MemberCount = _members.Count;
                Group.Update(g => g.MemberCount);

                if (_group.Type == GroupType.Large && DateTime.UtcNow.Subtract(_lastMemberCountUpdate) >= TimeSpan.FromSeconds(10))
                {
                    _lastMemberCountUpdate = DateTime.UtcNow;
                    GroupSearchIndexWorker.CreateGroupMemberCountUpdatedWorker(_group, _group.GetServerSearchSettingsOrDefault());
                }

            }

            foreach (var group in _groupCache.Values)
            {
                group.SaveChanges();
            }

            _members.SaveChanges();
        }

        public void SaveRealtimeData()
        {
            var likedMessages = _likeManager.Save();
            if (likedMessages == null)
            {
                return;
            }

            foreach (var message in likedMessages)
            {
                NotifyLikedMessage(message);
            }
        }

        public void SaveTenMinsData()
        {
            SaveMemberActivity(false);
        }

        public void SaveMemberActivity(bool ignoreTimestamp)
        {
            if (ignoreTimestamp && !IsPendingMemberActivitySync)
            {
                return;
            }

            DateMemberActivitySaved = DateTime.UtcNow;
            _members.UpdateAllConnected();
            var activeMembers = _members.GetConnectedUserIDs();

            Logger.Debug("Saving member activity for group: " + _group.Title, new { ActiveMemberCount = activeMembers.Count });

            if (activeMembers.Any())
            {                
                GroupMemberWorker.CreateMemberActivity(_group, activeMembers);
            }
        }

        /// <summary>
        /// Sends notifications to all the users of the group by queueing notifiers to Notification Server.
        /// </summary>
        public void NotifyChangedMessage(GroupMessageChangeCoordinator request)
        {
            // Try to get the group cache
            var targetGroup = GetGroup(request.TargetGroupID);

            if (targetGroup == null)
            {
                return;
            }

            // Get all members of the group with access permissions
            var members = GetAllMembersWithPermission(request.TargetGroupID, GroupPermissions.Access);

            if (!members.Any())
            {
                return;
            }

            // Dispatch this notification to all notification servers with members connected
            DispatchNotification(members, (regionID, serverName, sessions) =>
            {
                GroupMultiSessionMessageNotifier.Create(request.Notification, sessions, serverName, regionID);
            });
        }

        public void NotifyLikedMessage(ConversationMessage message)
        {
            var targetGroupID = Guid.Parse(message.ConversationID);

            // Try to get the group cache
            var targetGroup = GetGroup(targetGroupID);

            if (targetGroup == null)
            {
                return;
            }

            // Get all members of the group with access permissions
            var members = GetAllMembersWithPermission(targetGroupID, GroupPermissions.Access);

            if (!members.Any())
            {
                return;
            }

            var notification = message.ToNotification(0, null, ConversationType.Group, ConversationNotificationType.Liked);

            // Dispatch this notification to all notification servers with members connected
            DispatchNotification(members, (regionID, serverName, sessions) =>
            {
                GroupMultiSessionMessageNotifier.Create(notification, sessions, serverName, regionID);
            });
        }

        private void SendErrorChatResponse(GroupMessageCordinator request, DeliveryStatus status, long? retryAfter = null,
            MessageForbiddenReason forbiddenReason = MessageForbiddenReason.Unknown, GroupPermissions missingPermission = GroupPermissions.None)
        {
            if (request.SenderEndpoint == null || string.IsNullOrEmpty(request.SenderEndpoint.MachineKey))
            {
                return;
            }

            ChatMessageResponseNotifier.Create(request.SenderEndpoint, new ConversationMessageResponse
            {
                ClientID = request.MessageRequest.ClientID,
                ConversationID = request.MessageRequest.ConversationID,
                Status = status,
                RetryAfter = retryAfter,
                ForbiddenReason = forbiddenReason,
                MissingPermission = missingPermission
            });
        }


#if CONFIG_DEBUG || CONFIG_STAGING
        private readonly Random SpamRandom = new Random();
        private static readonly Regex SpamRegex = new Regex(@"^\/spam (?<count>\d{1,3})$", RegexOptions.Compiled);

        private readonly Random StatusRandom = new Random();
        private static readonly Regex StatusRegex = new Regex(@"^\/status (?<count>\d{1,3})$", RegexOptions.Compiled);
#endif

        /// <summary>
        /// Sends notifications to all the users of the group by queueing notifiers to Notification Server.
        /// </summary>
        public void NotifyInstantMessage(Guid targetGroupID, GroupMessageCordinator request)
        {

            try
            {

                ProcessSendMessage(targetGroupID, request);

#if CONFIG_DEBUG || CONFIG_STAGING
                var match = SpamRegex.Match(request.MessageRequest.Message);
                if (match.Success)
                {
                    var members = GetAllMembersWithPermission(targetGroupID, GroupPermissions.Access).ToArray();

                    var originalSender = _members.Get(request.SenderID);
                    var numberOfMessages = int.Parse(match.Groups["count"].Value);

                    for (var i = 0; i < numberOfMessages; i++)
                    {
                        var sender = members[SpamRandom.Next(0, members.Length - 1)];

                        ProcessSendMessage(targetGroupID, new GroupMessageCordinator
                        {
                            GroupID = targetGroupID,
                            MessageRequest = new ConversationMessageRequest
                            {
                                Message = string.Format("{0} made me say this! ({1})", originalSender.Member.GetTitleName(), i),
                                ConversationID = request.MessageRequest.ConversationID,
                                ClientID = Guid.NewGuid(),
                                AttachmentID = Guid.Empty
                            },
                            SenderID = sender.Member.UserID,
                            Timestamp = DateTime.UtcNow,
                        });

                        Thread.Sleep(100);
                    }

                    return;
                }

                match = StatusRegex.Match(request.MessageRequest.Message);
                if (match.Success)
                {
                    var userIDs = _members.GetAllUserIDs().ToArray();
                    var totalStatuses = Enum.GetValues(typeof (UserConnectionStatus)).Length;
                    var notifications = new Dictionary<int, GroupMemberContract>();
                    var selectGames = new List<int> {0, 634, 629, 555, 416, 372, 235, 234, 128, 34, 63};

                    var numberOfStatusChanges = int.Parse(match.Groups["count"].Value);

                    for (var i = 0; i < numberOfStatusChanges; i++)
                    {
                        var userID = userIDs[StatusRandom.Next(0, userIDs.Length)];
                        GroupMemberContract contract;
                        if (!notifications.TryGetValue(userID, out contract))
                        {
                            contract = GetMemberNotifications(new HashSet<int> {userID}).First();
                            notifications[userID] = contract;
                        }

                        var roll = StatusRandom.Next(100);
                        if (roll < 70)
                        {
                            var newStatus = (int) contract.ConnectionStatus + 1%totalStatuses;
                            if (newStatus == 3 || newStatus == 4)
                            {
                                // Don't send invisible or idle to the client, just skip to DnD
                                newStatus = 5;
                            }
                            contract.ConnectionStatus = (UserConnectionStatus) newStatus;
                        }
                        if (roll < 10 || roll >= 70)
                        {
                            // pick a new game from the list
                            int newGame;
                            do
                            {
                                newGame = selectGames[StatusRandom.Next(0, selectGames.Count)];
                            } while (newGame == contract.CurrentGameID);
                            contract.CurrentGameID = newGame;
                        }

                        var sender = _members.Get(userID);
                        var notification = new GroupChangeNotification
                        {
                            ChangeType = GroupChangeType.UpdateUsers,
                            Group = RootGroupCache.Group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl),
                            Members = new[] {contract},
                            SenderID = userID,
                            SenderName = sender?.Member.GetTitleName(),
                        };

                        // Notify all members that are connected of this event
                        DispatchGroupChangeNotification(notification);

                        Thread.Sleep(100);
                    }

                    return;
                }
#endif
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process group message!");
            }
        }

        private void ProcessSendMessage(Guid targetGroupID, GroupMessageCordinator request)
        {
            var sw = Stopwatch.StartNew();

            using (var timer = new SmartStopwatch())
            {

                // Try to get the group cache
                var targetGroup = GetGroup(targetGroupID);
                if (targetGroup == null)
                {
                    var targetGroupModel = _group.GetChildGroup(targetGroupID);

                    if (targetGroupModel == null)
                    {
                        Logger.Warn("User attempted to send a message to a group that is not in the database.");
                        SendErrorChatResponse(request, DeliveryStatus.Error);
                        return;
                    }

                    if (targetGroupModel.IsDeleted)
                    {
                        Logger.Warn("User attempted to send a message to a group that has been deleted.");
                        SendErrorChatResponse(request, DeliveryStatus.Error);
                        return;
                    }

                    targetGroup = new GroupCache(targetGroupModel, _group.RootGroup);
                    if (!_groupCache.TryAdd(targetGroupID, targetGroup))
                    {
                        targetGroup = GetGroup(targetGroupID);
                    }
                }

                if (targetGroup == null)
                {
                    Logger.Warn("User attempted to send a message to a group that is not in the cache.");
                    SendErrorChatResponse(request, DeliveryStatus.Error);
                    return;
                }

                var sender = _members.Get(request.SenderID);
                if (sender == null)
                {

                    SendErrorChatResponse(request, DeliveryStatus.Forbidden, null, MessageForbiddenReason.MissingPermission, GroupPermissions.Access);
                    return;
                }

                if (string.IsNullOrEmpty(targetGroup.Group.ExternalChannelID))
                {
                    ProcessSendConversationMessage(timer, sender, targetGroupID, targetGroup, request);
                }
                else
                {
                    ProcessSendExternalMessage(sender, targetGroupID, targetGroup, request);
                }

                Logger.Trace("Processed group message in " + sw.Elapsed.TotalMilliseconds.ToString("F2") + " ms", timer.Results);
            }
        }

        private void ProcessSendExternalMessage(GroupMemberCache sender, Guid targetGroupID, GroupCache targetGroup, GroupMessageCordinator request)
        {
            var community = targetGroup.Community;
            if (community == null)
            {
                Logger.Warn("Failed to send external message: Community not found", new {targetGroupID, request});
                SendErrorChatResponse(request, DeliveryStatus.Error);
                return;
            }

            var senderAccount = sender.GetExternalAccount();
            if (senderAccount == null)
            {
                Logger.Debug("Failed to send external message: Member did not have a linked account", new {sender});
                SendErrorChatResponse(request, DeliveryStatus.Forbidden, null, MessageForbiddenReason.NoLinkedAccount);
                return;
            }

            if (senderAccount.NeedsReauthentication)
            {
                Logger.Debug("Failed to send external message: Member's account needs reauthentication");
                SendErrorChatResponse(request, DeliveryStatus.Forbidden, null, MessageForbiddenReason.LinkedAccountNeedsReauth);
                return;
            }

            var body = request.MessageRequest.Message ?? string.Empty;
            foreach (var unsupportedCommand in FriendsServiceConfiguration.Instance.UnsupportedChatCommands)
            {
                if (body.ToLowerInvariant().Split(new[] {' '}, 2)[0] == unsupportedCommand)
                {
                    Logger.Debug("Failed to send external message: Unsupported command", new {unsupportedCommand});
                    SendErrorChatResponse(request, DeliveryStatus.Forbidden, null, MessageForbiddenReason.UnsupportedCommand);
                    return;
                }
            }

            ExternalMessageResolver.Create(senderAccount, community, request.SenderEndpoint, request.MessageRequest.Message, request.MessageRequest.ClientID, targetGroupID.ToString());
        }

        private void ProcessSendConversationMessage(SmartStopwatch timer, GroupMemberCache sender, Guid targetGroupID, GroupCache targetGroup, GroupMessageCordinator request)
        {
            // Check Access Or Upload Permission
            var permissionToCheck = request.Attachment == null
                ? GroupPermissions.ChatSendMessages
                : request.Attachment.IsEmbed
                    ? GroupPermissions.ChatUploadPhotos
                    : GroupPermissions.ChatAttachFiles;


            using (timer.Child("CheckPermission"))
            {
                if (!CheckMemberPermission(targetGroupID, request.SenderID, permissionToCheck))
                {
                    SendErrorChatResponse(request, DeliveryStatus.Forbidden, null, MessageForbiddenReason.MissingPermission, permissionToCheck);
                    return;
                }
            }

            using (timer.Child("ChatThrottle"))
            {
                if (Group.ChatThrottleEnabled && Group.ChatThrottleSeconds > 0)
                {
                    // Check if the user has the bypass chat throttle permission
                    if (!CheckMemberPermission(targetGroupID, request.SenderID, GroupPermissions.ChatBypassChatThrottle))
                    {
                        var rootCache = RootGroupCache;

                        // If not, check if they are throttled
                        long? retryAfter;
                        if (!sender.CanSendMessage(TimeSpan.FromSeconds(rootCache.Group.ChatThrottleSeconds), out retryAfter))
                        {
                            if (request.SenderEndpoint != null)
                            {
                                SendErrorChatResponse(request, DeliveryStatus.Throttled, retryAfter);
                            }

                            Logger.Debug("Throttled due to slow mode");
                            return;
                        }
                    }

                }
            }

            using (timer.Child("Mentions"))
            {

                // Resolve Mentions
                var mentions = new HashSet<int>(ConversationParser.GetMentions(request.MessageRequest.Message));

                // Check Mention Permission
                if (_group.Type == GroupType.Large && mentions.Count > 0 &&
                    !CheckMemberPermission(targetGroupID, request.SenderID, GroupPermissions.ChatMentionUsers))
                {
                    SendErrorChatResponse(request, DeliveryStatus.Forbidden, null, MessageForbiddenReason.MissingPermission, GroupPermissions.ChatMentionUsers);
                    return;
                }

                // Check Mention Everyone Permission
                if (_group.Type == GroupType.Large && mentions.Contains(0) &&
                    !CheckMemberPermission(targetGroupID, request.SenderID, GroupPermissions.ChatMentionEveryone))
                {
                    SendErrorChatResponse(request, DeliveryStatus.Forbidden, null, MessageForbiddenReason.MissingPermission, GroupPermissions.ChatMentionEveryone);
                    return;
                }
            }

            var roles = new HashSet<int> { _group.DefaultRoleID };
            if (sender.Member.Roles == null)
            {                
                Logger.Warn(string.Format("ProcessSendMessage: Sender Member roles were null: {0} - {1}", request.GroupID, request.SenderID), new { sender.Member });
            }
            else
            {
                foreach (var role in sender.Member.Roles)
                {
                    roles.Add(role);
                }
            }

            var messageID = Guid.NewGuid();
            ConversationMessage conversationMessage;
            using (timer.Child("ConversationMessage"))
            {
                conversationMessage = ConversationMessage.Create(targetGroup.Group.ConversationID,
                ConversationType.Group,
                messageID,
                request.MessageRequest.Message,
                request.Timestamp,
                sender.Member,
                sender.Member.Roles.ToArray(),
                sender.Member.BestRole,
                targetGroup.GetPermissionsHash(roles),
                0,
                targetGroup.RootGroup.GroupID,
                request.Attachment);
            }

            Logger.Trace("Processing new message", conversationMessage);

            using (timer.Child("ConversationMessageWorker"))
            {
                // Persist this message to storage, in all regions
                if (targetGroup.Group.SupportsOfflineMessaging)
                {
                    ConversationMessageWorker.CreateNewMessage(conversationMessage);
                }
            }

            using (timer.Child("Response"))
            {
                // Let the sender know it's gtg
                if (request.SenderEndpoint != null && !string.IsNullOrEmpty(request.SenderEndpoint.ServerName))
                {
                    ChatMessageResponseNotifier.Create(request.SenderEndpoint, new ConversationMessageResponse
                    {
                        ClientID = request.MessageRequest.ClientID,
                        ConversationID = request.MessageRequest.ConversationID,
                        Status = DeliveryStatus.Successful,
                        ServerID = messageID
                    });
                }
            }

            sender.UpdateDateMessageSent();

            IReadOnlyCollection<GroupMemberCache> members;
            using (timer.Child("Recipients - v2"))
            {
                // Get all members of the group with access permissions
                members = GetAllMembersWithPermission(targetGroupID, GroupPermissions.Access);

                if (!members.Any())
                {
                    return;
                }
            }

            using (timer.Child("Response"))
            {
                // Create the group message contract
                var groupMessageNotification = conversationMessage.ToNotification(request.SenderID,
                    request.MessageRequest.ClientID, ConversationType.Group, ConversationNotificationType.Normal);

                // Dispatch this notification to all notification servers with members connected
                DispatchNotification(members, (regionID, serverName, sessions) =>
                {
                    GroupMultiSessionMessageNotifier.Create(groupMessageNotification, sessions, serverName,
                        regionID);
                });

            }

            // Update the date messages for the group
            targetGroup.UpdateDateMessaged(request.Timestamp);

            using (timer.Child("Member Iteration"))
            {
                var pushEnabled = PushNotificationsEnabled;
                // Update the date messaged for all members
                foreach (var member in members)
                {
                    try
                    {
                        // For the sender, only bump the timestamp
                        if (member.Member.UserID == sender.Member.UserID)
                        {
                            member.Bump(targetGroupID, request.Timestamp);
                        }
                        else
                        {
                            member.UpdateDateMessaged(targetGroupID, request.Timestamp);
                        }

                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex);
                    }

                    if (pushEnabled)
                    {
                        ClientEndpoint.DispatchNotification(member.DeliverableEndpoints.Select(ep=>ep.Endpoint).ToArray(), null, endpoint =>
                        {
                            PushNotificationWorker.ConversationMessage(member.Member.RegionID, endpoint,
                                sender.Member.UserID,
                                sender.Member.GetTitleName(), member.Member.UserID, request.MessageRequest.Message,
                                request.MessageRequest.ConversationID, request.Timestamp, conversationMessage.Mentions);
                        });
                    }
                }
            }

        }

        protected bool PushNotificationsEnabled
        {
            get { return _members.Count <= 100; }
        }

        protected bool PresenceNotificationsEnabled
        {
            get { return _members.Count <= 1000; }
        }

        public void NotifyPrivateMessage(GroupPrivateMessageCoordinator request)
        {

            var targetGroup = GetGroup(request.GroupID);

            // Ensure the recipient is a member of the group
            var recipient = _members.Get(request.RecipientID);
            var sender = _members.Get(request.SenderID);
            if (recipient == null || sender == null)
            {
                if (request.SenderEndpoint != null && request.SenderEndpoint.IsConnected)
                {
                    ChatMessageResponseNotifier.Create(request.SenderEndpoint, new ConversationMessageResponse
                    {
                        ClientID = request.MessageRequest.ClientID,
                        ConversationID = request.MessageRequest.ConversationID,
                        Status = DeliveryStatus.UnknownUser
                    });
                }
                return;
            }

            var permissionToCheck = request.Attachment == null
                ? GroupPermissions.SendPrivateMessage
                : request.Attachment.IsEmbed ? GroupPermissions.ChatUploadPhotos : GroupPermissions.ChatAttachFiles;

            if (!CheckMemberPermission(_group.GroupID, request.SenderID, permissionToCheck))
            {
                if (request.SenderEndpoint != null && request.SenderEndpoint.IsConnected)
                {
                    ChatMessageResponseNotifier.Create(request.SenderEndpoint, new ConversationMessageResponse
                    {
                        ClientID = request.MessageRequest.ClientID,
                        ConversationID = request.MessageRequest.ConversationID,
                        Status = DeliveryStatus.Forbidden,
                        ForbiddenReason = MessageForbiddenReason.MissingPermission,
                        MissingPermission = permissionToCheck
                    });
                }

                return;
            }

            // See if the sender has permission to contact the recipient
            var senderConvo = GroupPrivateConversation.GetOrCreateBySenderIDAndRecipientIDAndGroupID(request.SenderID, request.RecipientID, targetGroup.Group.GroupID);
            var conversationContainer = senderConvo as IConversationContainer;

            if (!conversationContainer.CanSendMessage(request.RecipientID))
            {
                ChatMessageResponseNotifier.Create(request.SenderEndpoint, new ConversationMessageResponse
                {
                    ClientID = request.MessageRequest.ClientID,
                    ConversationID = request.MessageRequest.ConversationID,
                    Status = DeliveryStatus.Forbidden,
                    ForbiddenReason = MessageForbiddenReason.NotFamiliar
                });
            }

            var recipientConvo = GroupPrivateConversation.GetOrCreateBySenderIDAndRecipientIDAndGroupID(request.RecipientID, request.SenderID, targetGroup.Group.GroupID);

            if (senderConvo.IsProvisional)
            {
                senderConvo.MakePermanent(request.Timestamp);
            }

            // Ensure the convo is unhidden
            if (senderConvo.IsHidden)
            {
                senderConvo.IsHidden = false;
                senderConvo.Update(p => p.IsHidden);
            }

            if (recipientConvo.IsProvisional)
            {
                recipientConvo.MakePermanent(request.Timestamp);
            }

            // Ensure the convo is unhidden
            if (recipientConvo.IsHidden)
            {
                recipientConvo.IsHidden = false;
                recipientConvo.Update(p => p.IsHidden);
            }

            var messageID = Guid.NewGuid();

            var conversationMessage = ConversationMessage.Create(senderConvo.ConversationID,
                ConversationType.GroupPrivateConversation,
                messageID,
                request.MessageRequest.Message,
                request.Timestamp,
                sender.Member,
                sender.Member.Roles.ToArray(),
                sender.Member.BestRole,
                targetGroup.GetPermissionsHash(sender.Member.Roles),
                recipient.Member.UserID,
                targetGroup.RootGroup.GroupID,
                request.Attachment);

            // We have permission, process and send the message
            // Let the sender know it's gtg
            ChatMessageResponseNotifier.Create(request.SenderEndpoint, new ConversationMessageResponse
            {
                ClientID = request.MessageRequest.ClientID,
                ConversationID = request.MessageRequest.ConversationID,
                Status = DeliveryStatus.Successful,
                ServerID = messageID
            });

            var senderEndpoints = sender.ConnectedEndpoints.Select(ep => ep.Endpoint).ToArray();
            ClientEndpoint.DispatchNotification(senderEndpoints, endpoint =>
            {
                FriendMessageNotifier.Create(endpoint, conversationMessage.ToNotification(endpoint.UserID, request.MessageRequest.ClientID, ConversationType.GroupPrivateConversation, ConversationNotificationType.Normal));
            }, null);

            // Dispatch this notification to all notification servers with members connected
            ClientEndpoint.DispatchNotification(recipient.ConnectedEndpoints.Select(ep => ep.Endpoint).ToArray(), endpoint =>
            {
                FriendMessageNotifier.Create(endpoint, conversationMessage.ToNotification(endpoint.UserID, request.MessageRequest.ClientID, ConversationType.GroupPrivateConversation, ConversationNotificationType.Normal));
            }, null);

            ClientEndpoint.DispatchNotification(senderEndpoints, endpoint => { }, null);

            // Persist this message to storage, in all regions            
            ConversationMessageWorker.CreateNewMessage(conversationMessage);

            // Mark the convo as unread
            senderConvo.MarkAsUnread(request.Timestamp, 0);
            recipientConvo.MarkAsUnread(request.Timestamp, 1);

        }

        /// <summary>
        /// Sends notifications to all the users of the group by queueing notifiers to corresponding Notification Servers.
        /// </summary>        
        public void NotifyGroupCreation(Guid affectedGroupID, int senderID, bool retrievedFromCache)
        {
            var affectedGroup = Group.GetLocal(affectedGroupID);

            // If the group can't be retrieved from the local region, we have a problem!
            if (affectedGroup == null)
            {
                Logger.Warn("NotifyGroupCreation: Unable to get group from database", new { _group.GroupID, affectedGroupID });
                return;
            }

            // Add the new group to the cache
            if (!_groupCache.TryAdd(affectedGroupID, new GroupCache(affectedGroup, _group)) && retrievedFromCache)
            {
                Logger.Warn("Failed to add new group to group cache: " + affectedGroup.Title);
                return;
            }

            var sender = _members.Get(senderID);
            var notification = new GroupChangeNotification
           {
               ChangeType = GroupChangeType.CreateGroup,
               Group = CreateNotification(true),
               SenderID = senderID,
               SenderName = sender?.Member.GetTitleName(),
           };

            notification.Group.MemberCount = _members.Count;

            // If this is a normal root group, include the members of it
            if (affectedGroup.Type == GroupType.Normal && affectedGroup.IsRootGroup)
            {
                notification.Members = GetMemberNotifications();
            }
            else if(affectedGroup.Type == GroupType.Large && affectedGroup.IsRootGroup)
            {
                notification.Members = GetMemberNotifications(new HashSet<int> {senderID});
            }

            // Notify all members that are connected of this event
            DispatchGroupChangeNotification(notification);

        }

        private GroupMemberContract[] GetMemberNotifications()
        {
            var members = _members.GetAll();
            var userStatistics = UserStatistics.GetAllByUserIDs(members.Select(p => p.Member.UserID)).ToDictionary(p => p.UserID);
            return members.Select(p => p.Member.ToNotification(userStatistics.GetValueOrDefault(p.Member.UserID))).ToArray();
        }

        private GroupMemberContract[] GetMemberNotifications(HashSet<int> userIDs)
        {
            var members = new List<GroupMember>();

            foreach (var id in userIDs)
            {
                var memberCache = _members.Get(id);
                if (memberCache != null)
                {
                    members.Add(memberCache.Member);
                }
            }

            var userStatistics = UserStatistics.GetAllByUserIDs(userIDs).ToDictionary(p => p.UserID);
            return members.Select(p => p.ToNotification(userStatistics.GetValueOrDefault(p.UserID))).ToArray();
        }      
        /// <summary>
        /// Sends notifications to all the users of the group by queueing notifiers to corresponding Notification Servers.
        /// </summary>        
        public void NotifyGroupRemoval(Guid affectedGroupID, int senderID)
        {
            GroupCache removed;
            if (!_groupCache.TryRemove(affectedGroupID, out removed))
            {                
                return;
            }

            Logger.Info("Removed channel from cache: " + removed.Group.Title);

            var sender = _members.Get(senderID);

            var notification = new GroupChangeNotification
            {
                ChangeType = GroupChangeType.RemoveGroup,
                Group = removed.Group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl),
                SenderID = senderID,
                SenderName = sender?.Member.GetTitleName(),
            };

            // Notify all members that are connected of this event
            DispatchGroupChangeNotification(notification);

        }

        /// <summary>
        /// Updates the cache by removing requested members from the group
        /// Also sends notifications to the current users that belong to group
        /// </summary>
        /// <param name="senderID">The user that removed these users to the group.</param>
        /// <param name="userIDs">Ther users being removed from the group.</param>
        /// <param name="retrievedFromCache">Whether or not this group was retrieved from the in-memory cache.</param>
        public void RemoveMembers(int senderID, HashSet<int> userIDs, GroupMemberRemovedReason reason, string message, bool retrievedFromCache)
        {
            var notificationMemberCount = _members.Count;
            GroupMemberContract[] removedNotifications;
            GroupMember sender = null;
            if (retrievedFromCache)
            {
                var senderMemberCache = _members.Get(senderID);
                if (senderMemberCache != null)
                {
                    sender = senderMemberCache.Member;
                }

                removedNotifications = GetMemberNotifications(userIDs);
                notificationMemberCount = _members.Count - removedNotifications.Length;
            }
            else
            {
                // This occurs when the member is removed while the group has no session
                var members = GroupMember.MultiGetLocal(userIDs.Select(id => new KeyInfo(_group.GroupID, id)));
                sender = GroupMember.GetLocal(senderID);
                var userStatistics = UserStatistics.GetAllByUserIDs(userIDs).ToDictionary(p => p.UserID);
                removedNotifications = members.Select(p => p.ToNotification(userStatistics.GetValueOrDefault(p.UserID))).ToArray();
            }

            if (removedNotifications.Length > 0)
            {

                var notification = new GroupChangeNotification
                {
                    ChangeType = GroupChangeType.RemoveUsers,
                    Group = RootGroupCache.Group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl),
                    Members = removedNotifications.ToArray(),
                    SenderID = senderID,
                    SenderName = sender?.GetTitleName(),
                    RemovedReason = reason,
                    MessageToUsers = message
                };

                notification.Group.MemberCount = notificationMemberCount;

                // Notify all members that are connected of this event
                DispatchGroupChangeNotification(notification);
            }

            // Now remove them
            foreach (var id in userIDs)
            {
                _members.Remove(id);
            }

            _membersChanged = true;
        }

        /// <summary>
        /// updates in memory Cache by adding requested members to the Group and notifies all the users about the change
        /// </summary>
        /// <param name="senderID">The user that added this user to the group.</param>
        /// <param name="newUserIDs">Ther users being added to the group.</param>
        /// <param name="retrievedFromCache">Whether or not this group was retrieved from the in-memory cache.</param>
        public void AddMembers(int senderID, HashSet<int> newUserIDs, bool retrievedFromCache)
        {
            // Retrieve the new members from the database
            var newUsers = GroupMember.MultiGetLocal(newUserIDs.Select(p => new KeyInfo(_group.GroupID, p)));


            if (newUsers.Count != newUserIDs.Count)
            {
                Logger.Warn("AddMembers was unable to get one or more members from the datatabase. This is indicative of a replication issue.", new { _group.GroupID, _group.RegionID });
            }

            var newUserStats = UserStatistics.GetAllByUserIDs(newUserIDs).ToDictionary(p => p.UserID);
            var endpointsByUser = ClientEndpoint.MultiGetAllForUserIDs(newUsers.Select(p => p.UserID).ToArray());
            var twitchAccountsByUser = ExternalAccount.MultiGetAllTwitchAccountsByUserStats(newUserStats.Values.ToArray());

            // Added them to the cache
            foreach (var user in newUsers)
            {
                var groupSessionMember = new GroupMemberCache(_group, user, endpointsByUser.GetValueOrDefault(user.UserID), twitchAccountsByUser.GetValueOrDefault(user.UserID));

                if (!_members.Add(groupSessionMember) && retrievedFromCache)
                {
#if DEBUG
                    throw new Exception("Unable to add a member to the group session!");
#endif
                }
            }

            _membersChanged = true;

            var newUserNotifications = newUsers.Select(p => p.ToNotification(newUserStats.GetValueOrDefault(p.UserID))).ToArray();
            var groupNotification = RootGroupCache.Group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl);
            groupNotification.MemberCount = _members.Count;
            var sender = _members.Get(senderID);
            var notification = new GroupChangeNotification
            {
                ChangeType = GroupChangeType.AddUsers,
                Group = groupNotification,
                Members = newUserNotifications.ToArray(),
                SenderID = senderID,
                SenderName = sender?.Member.GetTitleName(),
            };
            notification.Group.MemberCount = _members.Count;

            DispatchGroupChangeNotification(notification);
        }


        /// <summary>
        /// Dispatches presence notifications
        /// </summary>        
        public void UpdateMemberPresence(GroupPresenceContract presenceUpdate)
        {

            if (!PresenceNotificationsEnabled)
            {
                return;
            }

            var notification = new GroupPresenceNotification
            {
                GroupID = _group.GroupID,
                Users = new[] { presenceUpdate }
            };

            // Notify all members that are connected of this event
            DispatchPresenceNotification(notification);
        }

        /// <summary>
        /// Adds or updates members in the cache
        /// </summary>        
        public void UpdateMembers(int senderID, HashSet<int> updatedUserIDs)
        {
            var updatedMembers = new List<GroupMember>();

            // Add or update members in the cache
            foreach (var userID in updatedUserIDs)
            {
                var memberCache = _members.Get(userID);
                if (memberCache == null)
                {
                    continue;
                }

                memberCache.Refresh(_group);
                updatedMembers.Add(memberCache.Member);
            }

            if (updatedUserIDs.Count > 1000)
            {
                Logger.Warn("UpdateMembers attempted to send out a notification for more than 1,000 member changes. This will be supressed.");
                return;
            }

            var memberNotifications = GetMemberNotifications(updatedUserIDs);

            var sender = _members.Get(senderID);
            var notification = new GroupChangeNotification
            {
                ChangeType = GroupChangeType.UpdateUsers,
                Group = RootGroupCache.Group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl),
                Members = memberNotifications,
                SenderID = senderID,
                SenderName = sender?.Member.GetTitleName(),
            };

            // Notify all members that are connected of this event
            DispatchGroupChangeNotification(notification);
        }

        /// <summary>
        /// Updates the cache data with new information about the group
        /// </summary>
        public void ChangeInfo(int senderID)
        {
            Logger.Trace("Received group change info coordinator for group: " + _group.Title);
            
            // Refresh all groups
            _group.Refresh();
            foreach (var group in _groupCache.Values)
            {
                group.Refresh();
            }

            var sender = _members.Get(senderID);
            var notification = new GroupChangeNotification
            {
                ChangeType = GroupChangeType.ChangeInfo,
                Group = CreateNotification(true),
                SenderID = senderID,
                SenderName = sender?.Member.GetTitleName(),
            };

            // Notify all members that are connected of this event
            DispatchGroupChangeNotification(notification);
        }


        /// <summary>
        /// updates the ClientEndpoints[] of a user that belongs to the group, when a user logon/logoff
        /// </summary>
        /// <param name="userID"></param>
        /// <param name="endpoints"></param>
        public void UpdateMemberEndpoints(int userID, IReadOnlyCollection<ClientEndpoint> endpoints)
        {
            var member = _members.Get(userID);
            if (member == null)
            {
                return;
            }

            member.UpdateEndpoints(_group, endpoints, true);
        }

        public void UpdateMemberAccount(int userId, ExternalAccount externalAccount)
        {
            var member = _members.Get(userId);
            if (member == null)
            {
                return;
            }

            member.UpdateExternalAccount(externalAccount);
        }


        /// <summary>
        /// notifies users with changes made for voicesessioncode
        /// </summary>
        public void NotifyVoiceSessionChange(Guid affectedGroupID, GroupChangeType type, HashSet<int> userIDs)
        {
            var groupCache = GetGroup(affectedGroupID);

            if (groupCache == null)
            {
                return;
            }

            groupCache.Refresh();

            var memberNotifications = GetMemberNotifications(userIDs);

            var notification = new GroupChangeNotification
            {
                ChangeType = type,
                Group = groupCache.Group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl),
                Members = memberNotifications,
                SenderID = 0
            };

            DispatchGroupChangeNotification(notification);
        }

        public void NotifyGroupRestructured(int senderID)
        {
            var childGroups = _groupCache.Values.Where(p => !p.Group.IsRootGroup);

            var sender = _members.Get(senderID);
            var notification = new GroupChangeNotification
            {
                ChangeType = GroupChangeType.GroupReorganized,
                Group = RootGroupCache.Group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl),
                SenderID = senderID,
                SenderName = sender?.Member.GetTitleName(),
                ChildGroups = childGroups.Select(g => g.Group.ToChannelNotification(null,null)).ToArray()
            };

            DispatchGroupChangeNotification(notification);
        }

        public void RefreshAllGroups()
        {
            // Refresh the caches
            foreach (var group in _groupCache.Values)
            {
                group.Refresh();
            }
        }

        public void NotifyRolesChanged(int senderID)
        {
            // Gets the latest group data from the database
            RefreshAllGroups();

            var group = GetGroup(Group.GroupID);
            if (group == null)
            {
                Logger.Warn("Root group was not in the group cache.");
                return;
            }

            var sender = _members.Get(senderID);
            var notification = new GroupChangeNotification()
            {
                ChangeType = GroupChangeType.RoleNamesChanged,
                SenderID = senderID,
                SenderName = sender?.Member.GetTitleName(),
                Group = CreateNotification(true)
            };

            DispatchGroupChangeNotification(notification);
        }

        public GroupNotification CreateNotification(bool includeChannels = false)
        {
            if (_group.Type == GroupType.Normal)
            {
                return _group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, false, true, true);
            }

            // Construct the notification using the in memory data vs hitting the DB again
            var notification = _group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, false, true, true);

            // Construct channel notifications from the in-memory cache
            var channels = _groupCache.Values.Where(p => !p.Group.IsRootGroup).Select(p => p.Group).ToArray();
            if (includeChannels)
            {
                notification.Channels = channels.Select(p => p.ToChannelNotification(null, null)).ToArray();
            }

            return notification;
        }

        public void NotifyEmoticonsChanged(int senderId)
        {
            var group = GetGroup(Group.GroupID);
            if (group == null)
            {
                Logger.Warn("Root group was not in the group cache.");
                return;
            }
            group.Refresh();

            var sender = _members.Get(senderId);
            var notification = new GroupChangeNotification()
            {
                SenderID = senderId,
                SenderName = sender?.Member.GetTitleName(),
                ChangeType = GroupChangeType.UpdateEmoticons,
                Group = CreateNotification(),
            };

            DispatchGroupChangeNotification(notification);
        }

        public void NotifyPollChange(GroupPollChangedNotification notification)
        {
            var members = _members.GetAll();
            DispatchNotification(members, (regionID, serverName, sessions) => GroupPollMultiSessionChangedNotifier.Create(notification, sessions, serverName, regionID));
        }

        public void NotifyGiveawayChanged(GroupGiveawayChangedNotification notification, int? targetUserID)
        {
            GroupMemberCache[] members;
            if (targetUserID.HasValue)
            {
                var member = _members.Get(targetUserID.Value);
                if (member == null)
                {
                    Logger.Warn(string.Format("Giveaway Notification: Member {0} was not found in the group member cache for {1}", targetUserID.Value, _group.GroupID));
                    return;
                }
                members = new[] {member};
            }
            else
            {
                members = _members.GetAll();
            }
            DispatchNotification(members, (regionID, serverName, sessions) => GroupGiveawayMultiSessionNotifier.Create(notification, sessions, serverName, regionID));
        }

        public void ProcessMessageLike(GroupMessageLikeCoordinator coordinator)
        {
            // Get the like record
            var like = Like.GetByEntityTypeAndEntityAndUser((int)LikeType.ConversationMessage, coordinator.ConversationMessageID, coordinator.UserID);

            if (like == null)
            {
                Logger.Info("Unable to process message like. Like record not found.", coordinator);
                return;
            }

            var memberCache = _members.Get(coordinator.UserID);
            if (memberCache == null)
            {
                Logger.Warn("Unable to process message like. Group member not found in cache.", coordinator);
                return;
            }

            _likeManager.Add(coordinator.ConversationID, coordinator.ConversationMessageID, coordinator.ConversationMessageTimestamp, memberCache.Member.UserID, memberCache.Member.GetTitleName(), like.Unlike);
        }

        public void ProcessMessageRead(GroupMessageReadCoordinator coordinator)
        {
            // Try to get the member from the cache
            var memberCache = _members.Get(coordinator.UserID);
            if (memberCache == null)
            {
                Logger.Warn("Unable to process message read coordinate. Group member not found in cache.", coordinator);
                return;
            }

            // Get the affected group
            var affectedGroupID = coordinator.AffectedGroupID != Guid.Empty
                ? coordinator.AffectedGroupID
                : _group.GroupID;

            // Ensure the user actually has permission to the group
            if (!CheckMemberPermission(affectedGroupID, coordinator.UserID, GroupPermissions.Access))
            {
                return;
            }

            memberCache.MarkAsRead(affectedGroupID, coordinator.MessageTimestamp);
        }


        public void NotifyCallStarted(GroupCallCoordinator coordinator)
        {
            // Try to get the group cache
            var targetGroup = GetGroup(coordinator.TargetGroupID);

            if (targetGroup == null)
            {
                Logger.Warn("Unable to notify group of a call starting. Target group was not in the cache", coordinator);
                return;
            }

            var sender = _members.Get(coordinator.SenderID);

            if (sender == null)
            {
                Logger.Warn("Unable to notify group of a call starting. Target sender was not in the cache", coordinator);
                return;
            }


            var callDetails = coordinator.CallDetails;

            DispatchNotification(coordinator.TargetGroupID, GroupPermissions.Access, new HashSet<int> { callDetails.CreatorID }, (ep) =>
            {
                var accessToken = AccessTokenHelper.CreateAccessToken(callDetails.CallID, ep.Endpoint.UserID);
                CallNotifier.Create(ep.Endpoint, callDetails.ToNotification(accessToken, sender.Member.UserID, sender.Member.GetTitleName()));
            });
        }

        public void NotifyCallResponded(GroupCallCoordinator coordinator)
        {
            // Try to get the group cache
            var targetGroup = GetGroup(coordinator.TargetGroupID);

            if (targetGroup == null)
            {
                return;
            }

            var callDetails = coordinator.CallDetails;

            DispatchNotification(coordinator.TargetGroupID, GroupPermissions.Access, new HashSet<int> { callDetails.CreatorID }, (ep) =>
            {
                CallRespondedNotifier.Create(ep.Endpoint, coordinator.CallRespondedNotification);
            });
        }

        public GroupCache GetGroup(Guid groupID)
        {
            GroupCache groupCache;
            if (_groupCache.TryGetValue(groupID, out groupCache))
            {
                return groupCache;
            }

            return null;
        }

        public void ProcessCommunityMappingChanged(ExternalCommunity community, ExternalCommunityLinkChangedNotification notification)
        {
            Logger.Info("Dispatching stream status notification: " + _group.Title);            
            RefreshAllGroups();

            if (community != null)
            {
                var targetGroup = _groupCache.Select(kvp => kvp.Value).FirstOrDefault(g => g.Group.ExternalChannelID == community.ExternalID);
                if (targetGroup != null)
                {
                    targetGroup.UpdateCommunity(community);
                }
            }
            
            var members = _members.GetAll();
            DispatchNotification(members, (region, server, sessions) => ExternalCommunityLinkChangedMultiSessionNotifier.Create(notification, sessions, server, region));            
        }

        private class ExternalMessageAuthor : IUserIdentity
        {
            public int UserID { get; set; }
            public string Username { get; set; }
            public string DisplayName { get; set; }
            public string Nickname { get; set; }
        }

        public void NotifyExternalMessage(ExternalMessageCoordinator coordinator)
        {
            if (string.IsNullOrEmpty(coordinator.ExternalChannelID))
            {
                Logger.Debug("External Channel ID was null, can't coordinate external message", coordinator);
                return;
            }

            var targetGroup = _groupCache.Where(kvp => kvp.Value.Community!=null && kvp.Value.Community.ExternalID == coordinator.ExternalChannelID).Select(kvp=>kvp.Value).FirstOrDefault();
            if (targetGroup == null)
            {
                return;
            }

            var author = new ExternalMessageAuthor
            {
                UserID = 0,
                Username = coordinator.ExternalUsername,
                DisplayName = coordinator.ExternalUserDisplayName,
            };

            int[] senderRoles = null;
            var senderBestRole = 0;
            var senderPermissions = 0L;
           
            if (coordinator.MappedUserIDs != null)
            {
                foreach (var id in coordinator.MappedUserIDs)
                {
                    var member = _members.Get(id);
                    if (member != null && member.Member != null)
                    {
                        author.UserID = member.Member.UserID;
                        author.Username = member.Member.Username;
                        author.DisplayName = member.Member.DisplayName;
                        author.Nickname = member.Member.Nickname;
                        
                        senderRoles = member.Member.Roles.ToArray();
                        senderBestRole = member.Member.BestRole;
                        senderPermissions = targetGroup.GetPermissionsHash(member.Member.Roles);
                        break;
                    }
                }
            }           
            

            var message = ConversationMessage.Create(targetGroup.Group.ConversationID, ConversationType.Group, coordinator.MessageID, coordinator.MessageBody, coordinator.MessageTimestamp,
                author, senderRoles, senderBestRole, senderPermissions, 0, _group.GroupID, null, coordinator.EmoteSubstitutions);

            // Fill in shim data
            message.ExternalUserID = coordinator.ExternalUserID;
            message.ExternalUsername = coordinator.ExternalUsername;
            message.ExternalUserDisplayName = coordinator.ExternalUserDisplayName;
            message.ExternalChannelID = coordinator.ExternalChannelID;
            message.ExternalUserColor = coordinator.ExternalUserColor;
            message.Badges = coordinator.Badges;
            message.BitsUsed = coordinator.BitsUsed;

            var notification = message.ToNotification(author.UserID, Guid.Empty, ConversationType.Group, ConversationNotificationType.Normal);
            var members = GetAllMembersWithPermission(targetGroup.Group.GroupID, GroupPermissions.Access);
            DispatchFilteredNotification(targetGroup, members, (region, machine, sessions) => GroupMultiSessionMessageNotifier.Create(notification, sessions, machine, region));
        }

        public void NotifyTwitchNotice(int[] notificationTargets, TwitchChatNoticeNotification notification)
        {
            if (string.IsNullOrEmpty(notification.ExternalChannelID))
            {
                Logger.Debug("External Channel ID was null, can't coordinate twitch chat notice", notification);
                return;
            }

            var targetGroup = _groupCache.Where(kvp => kvp.Value.Community!=null && kvp.Value.Community.ExternalID == notification.ExternalChannelID).Select(kvp=>kvp.Value).FirstOrDefault();
            if (targetGroup == null)
            {
                return;
            }

            GroupMemberCache[] groupMembers;
            if (notificationTargets == null || notificationTargets.Length == 0)
            {
                groupMembers = _members.GetAllMembersWithPermission(targetGroup, GroupPermissions.Access).ToArray();
            }
            else
            {
                groupMembers = notificationTargets.Select(id => _members.Get(id)).Where(m => m != null).ToArray();
            }

            DispatchFilteredNotification(targetGroup, groupMembers, (region, machine, sessions) => TwitchChatNoticeMultiSessionNotifier.Create(notification, region, machine, sessions));
        }

        private void DispatchFilteredNotification(GroupCache targetGroup, IEnumerable<GroupMemberCache> members, Action<int, string, HashSet<string>> action)
        {
            DispatchNotification(members, action, ep => ep.CurrentGroupID == targetGroup.Group.GroupID || ep.CurrentGroupID == _group.GroupID);
        }

        public void NotifyBulkMessageDelete(GroupBulkMessageDeleteNotification notification)
        {
            if(notification == null || !notification.Validate())
            {
                Logger.Debug("GroupBulkMessageDeleteNotification was missing required information", notification);
                return;
            }

            // Try to get the group cache
            var targetGroup = GetGroup(notification.GroupID);

            if (targetGroup == null)
            {
                return;
            }

            // Get all members of the group with access permissions
            var members = GetAllMembersWithPermission(notification.GroupID, GroupPermissions.Access);

            if (!members.Any())
            {
                return;
            }

            // Dispatch this notification to all notification servers with members connected
            DispatchNotification(members, (regionID, serverName, sessions) =>
            {
                GroupBulkMessageDeleteNotifier.Create(notification, sessions, serverName, regionID);                
            });
        }

        public Dictionary<string,List<string>> GetChatters()
        {
            var allmembers = _members.GetAll().Select(m => new {Account = m.GetExternalAccount(true), Member = m}).Where(m => m.Account != null);
            var groupChannelMappings = _groupCache.Values.Where(g => !string.IsNullOrEmpty(g.Group.ExternalChannelID)).ToDictionary(g => g.Group.GroupID, g => g.Group.ExternalChannelID);
            var endpoints = allmembers.SelectMany(m =>
                m.Member.ConnectedEndpoints.Where(ep => groupChannelMappings.ContainsKey(ep.CurrentGroupID)).Select(e => new {Group = e.CurrentGroupID, ID = m.Account.ExternalID}));

            var chatters = new Dictionary<string, List<string>>();
            foreach (var group in endpoints.GroupBy(ep=>ep.Group))
            {
                string externalID;
                if (!groupChannelMappings.TryGetValue(group.Key, out externalID))
                {
                    continue;
                }

                List<string> c;
                if (chatters.TryGetValue(externalID, out c))
                {
                    c.AddRange(group.Select(g => g.ID));
                }
                else
                {
                    chatters[externalID] = group.Select(g => g.ID).ToList();
                }
            }
            return chatters;
        }
    }
}
