﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Queues;
using Curse.Logging;

namespace Curse.Friends.GroupService
{
    public class LikeManager
    {
        private readonly ConcurrentDictionary<string, LikeState>  _likeStates = new ConcurrentDictionary<string, LikeState>();

        public void Add(string conversationID, string messageID, long messageTimestamp, int userID, string username, bool unlike)
        {
            var likeState = _likeStates.GetOrAdd(messageID, s => new LikeState(conversationID, messageID, messageTimestamp));
            likeState.LikeUsers.AddOrUpdate(userID, uid => new LikeUser(userID, username, unlike), (i, user) => new LikeUser(userID, username, unlike));
        }

        public ConversationMessage[] Save()
        {
            if (_likeStates.Count == 0)
            {
                return null;
            }

            var savedMessages = new List<ConversationMessage>();

            var keys = _likeStates.Keys.ToArray();

            foreach (var key in keys)
            {
                LikeState likeState;
                if (!_likeStates.TryRemove(key, out likeState))
                {
                    continue;                    
                }

                var message = likeState.Save();
                if (message != null)
                {
                    savedMessages.Add(message);
                }
            }

            return savedMessages.ToArray();
        }
    }

    public class LikeState
    {
        public LikeState(string conversationID, string messageID, long messageTimestamp)
        {
            _conversationID = conversationID;
            _messageID = messageID;
            _messageTimestamp = messageTimestamp;
        }

        private readonly string _conversationID;
        private readonly string _messageID;
        private readonly long _messageTimestamp;
     
        public ConcurrentDictionary<int, LikeUser> LikeUsers = new ConcurrentDictionary<int, LikeUser>();

        public ConversationMessage Save()
        {
            if (LikeUsers.Count == 0)
            {
                return null;
            }


            ConversationMessage conversationMessage;

            try
            {
                conversationMessage = ConversationManager.GetMessageByID(_conversationID, _messageID, _messageTimestamp.FromEpochMilliconds());
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to save mesages likes. The message could not be retrieved from Elastic!", new { ConversationID = _conversationID, MessageID = _messageID, Timestamp = _messageTimestamp });
                return null;
            }


            // Serious problem! No conversation message
            if (conversationMessage == null)
            {
                Logger.Warn("Unable to save mesages likes. The message was not found in Elastic!");
                return null;
            }

          
            // Get a copy of the users and clear the array
            var users = LikeUsers.Values.ToArray();
            LikeUsers.Clear();
            
            
            // Now reconcile the user list
            var currentUserIDs = new HashSet<int>(conversationMessage.LikeUserIDs);
            var currentUsernames = new HashSet<string>(conversationMessage.LikeUsernames);

            foreach (var user in users)
            {
                if (user.Unlike)
                {
                    currentUserIDs.Remove(user.UserID);
                    currentUsernames.Remove(user.Username);
                }
                else
                {
                    currentUserIDs.Add(user.UserID);
                    if (currentUsernames.Count < ConversationMessage.MaxLikeUsernames * 2)
                    {
                        currentUsernames.Add(user.Username);
                    }
                }                
            }

            if (currentUsernames.Count > ConversationMessage.MaxLikeUsernames)
            {
                currentUsernames = new HashSet<string>(currentUsernames.Take(10));
            }

            conversationMessage.LikeCount = currentUserIDs.Count;
            conversationMessage.LikeUserIDs = currentUserIDs.ToArray();
            conversationMessage.LikeUsernames = currentUsernames.ToArray();

            // Our data is reconciled! Save it to the local region
            ConversationManager.UpdateMessageLikes(_conversationID, _messageID, _messageTimestamp.FromEpochMilliconds(), conversationMessage.LikeCount, currentUserIDs, currentUsernames);
            
            // Queue off a worker to each remote region
            ConversationMessageWorker.CreateSetLikeInfo(conversationMessage);

            return conversationMessage;
        }
    }

    public class LikeUser
    {
        public LikeUser(int userID, string username, bool unlike)
        {
            UserID = userID;
            Username = username;
            Unlike = unlike;
        }

        public int UserID { get; set; }
        public string Username { get; set; }
        public bool Unlike { get; set; }
    }
}
