﻿using System;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;

namespace Curse.Friends.GroupsWebService.Contracts
{
    public enum GroupInvitationReadableWordCategory
    {
        None,
        WorldOfWarcraft,
    }

    public class CreateGroupInviteRequest : BaseRequestContract<CreateGroupInvitationErrorType>
    {
        /// <summary>
        /// If specified, the recipient of this invite will be placed in this channel, by default.
        /// </summary>
        public Guid ChannelID { get; set; }

        /// <summary>
        /// When true, members that joined the server with this invite will be removed when the invite expires or is invalidated.
        /// </summary>
        public bool AutoRemoveMembers { get; set; }

        /// <summary>
        /// How long this invitation should be valid.
        /// </summary>
        public int LifespanMinutes { get; set; }

        /// <summary>
        /// Maximum number of uses this invite has.
        /// </summary>
        public int MaxUses { get; set; }

        /// <summary>
        /// Optional description for the invite.
        /// </summary>
        public string AdminDescription { get; set; }

        /// <summary>
        /// Whether or not to make the invite link readable.
        /// </summary>
        public GroupInvitationReadableWordCategory ReadableWordCategory { get; set; }

        protected override bool DoValidate(out CreateGroupInvitationErrorType errorType)
        {
            if (LifespanMinutes < 0 || LifespanMinutes > 1152000)
            {
                _validationMessage = "Lifespan specified is too short or too long.";
                errorType = CreateGroupInvitationErrorType.LifeSpanOutOfRange;
                return false;
            }

            if (MaxUses < 0)
            {
                _validationMessage = "MaxUses must be greater than or equal to 0.";
                errorType = CreateGroupInvitationErrorType.MaxUsesOutOfRange;
                return false;
            }

            if (!AdminDescription.SafeRange(0, GroupInvitation.DescriptionMaxLength))
            {
                _validationMessage = "Description specified is too long";
                errorType = CreateGroupInvitationErrorType.AdminDescriptionOutOfRange;
                return false;
            }

            errorType = CreateGroupInvitationErrorType.Success;
            return true;
        }

    }
}