﻿using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;

namespace Curse.Friends.GroupsWebService.Contracts
{
    public class CreateGuildCommunityRequest : BaseRequestContract<GuildServerCreationErrorType>
    {
        public string Title { get; set; }

        public bool IsPublic { get; set; }

        /// <summary>
        /// Optional.
        /// </summary>
        public int[] Games { get; set; }

        public GuildCommunityRoleContract OwnerRole { get; set; }

        public GuildCommunityRoleContract GuestRole { get; set; }

        /// <summary>
        /// Optional.
        /// </summary>
        public GuildCommunityRoleContract[] AdditionalRoles { get; set; }

        public string GeneralChatName { get; set; }

        public string OfficerChatName { get; set; }

        /// <summary>
        /// Optional.
        /// </summary>
        public string PveChatName { get; set; }

        /// <summary>
        /// Optional.
        /// </summary>
        public string PvpChatName { get; set; }

        protected override bool DoValidate(out GuildServerCreationErrorType errorType)
        {
            if (!Title.SafeRange(1, Group.TitleMaxLength))
            {
                _validationMessage = "Server title length out of range";
                errorType = GuildServerCreationErrorType.TitleLength;
                return false;
            }

            if (OwnerRole == null)
            {
                _validationMessage = "Owner role missing";
                errorType = GuildServerCreationErrorType.OwnerRoleMissing;
                return false;
            } 

            if(!OwnerRole.Name.SafeRange(1, GroupRole.NameMaxLength))
            {
                _validationMessage = "Owner role name length out of range";
                errorType = GuildServerCreationErrorType.OwnerRoleNameLength;
                return false;
            }

            if (GuestRole == null)
            {
                _validationMessage = "Guest role missing";
                errorType = GuildServerCreationErrorType.GuestRoleMissing;
                return false;
            }
            if(!GuestRole.Name.SafeRange(1, GroupRole.NameMaxLength))
            {
                _validationMessage = "Guest role name length out of range";
                errorType = GuildServerCreationErrorType.GuestRoleNameLength;
                return false;
            }

            if (AdditionalRoles != null)
            {
                foreach (var additionalRole in AdditionalRoles)
                {
                    if (!additionalRole.Name.SafeRange(1, GroupRole.NameMaxLength))
                    {
                        _validationMessage = "AdditionalRole name length out of range: " + additionalRole.Name;
                        errorType = GuildServerCreationErrorType.AdditionalRoleNameLength;
                        return false;
                    }
                }
            }

            if (!GeneralChatName.SafeRange(1, Group.TitleMaxLength))
            {
                _validationMessage = "General chat name length out of range";
                errorType = GuildServerCreationErrorType.GeneralChatNameLength;
                return false;
            }

            if (!OfficerChatName.SafeRange(1, Group.TitleMaxLength))
            {
                _validationMessage = "Officer chat name length out of range";
                errorType = GuildServerCreationErrorType.OfficerChatNameLength;
                return false;
            }

            // Allow null for optional channel
            if (PveChatName != null && !PveChatName.SafeRange(1, Group.TitleMaxLength))
            {
                _validationMessage = "PVE chat name length out of range";
                errorType = GuildServerCreationErrorType.PveChatNameLength;
                return false;
            }

            // Allow null for optional channel
            if (PvpChatName != null && !PvpChatName.SafeRange(1, Group.TitleMaxLength))
            {
                _validationMessage = "PVP chat name length out of range";
                errorType = GuildServerCreationErrorType.PvpChatNameLength;
                return false;
            }

            errorType = GuildServerCreationErrorType.NoError;
            return true;
        }
    }

    public class GuildCommunityRoleContract
    {
        public string Name { get; set; }

        public int Color { get; set; }

        public bool IsOfficer { get; set; }

        public bool HasBadge { get; set; }

        public int VanityBadge { get; set; }

        public NewGuildRole ToNewGuildRole()
        {
            return new NewGuildRole
            {
                Name = Name,
                VanityColor = Color,
                VanityBadge = VanityBadge,
                HasBadge = HasBadge,
                IsOfficer = IsOfficer
            };
        }
    }
}