﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.GroupsWebService.Contracts;
using Curse.Friends.MicroService;
using Curse.Logging;

namespace Curse.Friends.GroupsWebService.Controllers
{
#if !CONFIG_DEBUG && !CONFIG_STAGING
    [ApiExplorerSettings(IgnoreApi = true)]
#endif
    [RoutePrefix("groups/api")]
    [AuthenticationFilter(AuthenticationLevel.ApiKey)]
    public class GroupsApiController : ApiController
    {
        [Route("{serverID}/featured")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult SetServerFeatured(Guid serverID, [FromBody] bool isFeatured = true)
        {
            var group = Group.GetByID(serverID);
            if (group == null)
            {
                return NotFound();
            }

            if (!group.IsRootGroup || group.Type != GroupType.Large)
            {
                return BadRequest();
            }

            group.UpdateIsFeatured(isFeatured, DateTime.UtcNow);

            return StatusCode(HttpStatusCode.NoContent);
        }

        [Route("{groupID}/members/owners")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult AddOwner(Guid groupID, [FromBody] int userID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.AddMemberRole(group.OwnerID, group.OwnerRole, userID);

            return StatusCode(HttpStatusCode.NoContent);
        }

        [Route("vanity-urls/blacklists")]
        [HttpPost]
        [ResponseType(typeof (void))]
        public IHttpActionResult BlacklistVanityUrls([FromBody] string[] urlsToBlacklist)
        {
            try
            {
                var distinctUrls = urlsToBlacklist.Distinct();
                foreach (var url in distinctUrls)
                {
                    VanityUrl.BlackList(url);
                }

                return Ok();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unexpected error while blacklisting urls", urlsToBlacklist);
                return InternalServerError();
            }
        }

        [Route("vanity-urls/reserves")]
        [HttpPost]
        [ResponseType(typeof (void))]
        public IHttpActionResult ReserveVanityUrls([FromBody] string[] urlsToReserve)
        {
            try
            {
                var distinctUrls = urlsToReserve.Distinct();
                foreach (var url in distinctUrls)
                {
                    VanityUrl.Reserve(url);
                }

                return Ok();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unexpected error while reserving urls", urlsToReserve);
                return InternalServerError();
            }
        }

        [HttpPost]
        [Route("vanity-urls/groups/{groupID}")]
        [ResponseType(typeof (void))]
        public IHttpActionResult AssignVanityUrl(Guid groupID, [FromBody] string url)
        {
            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.UpdateVanityUrl(url);

            return Ok();
        }

        #region Control Panel

        [Route("discover/search")]
        [HttpPost]
        [ResponseType(typeof(IEnumerable<ApiGroupSearchContract>))]
        public IHttpActionResult SearchGroups([FromBody] SearchServersRequest request)
        {
            request.Validate();

            var results = GroupSearchManager.Search(new GroupSearch
            {
                Games = request.Games,
                GroupTitle = request.GroupTitle,
                IncludeInappropriate = true,
                IsFeatured = request.IsFeatured,
                IsPublic = request.IsPublic,
                IsStreaming = request.IsStreaming,
                MaxMemberCount = request.MaxMemberCount,
                MinMemberCount = request.MinMemberCount,
                OwnerUsername = request.OwnerUsername,
                PageNumber = request.PageNumber,
                PageSize = request.PageSize,
                Query = request.Query,
                SortAscending = request.SortAscending ?? true,
                SortType = request.SortType ?? GroupSearchSortType.Default,
                SubType = request.SubType,
                Tags = request.Tags
            });

            return Ok(results.Select(r => new ApiGroupSearchContract
            {
                Model = r.ToNotification(),
                AvatarUrl = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "groups", r.GroupID),
                GroupCoverUrl = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "groups", r.GroupID) + "/cover"
            }));
        }

        [Route("discover/quarantine")]
        [HttpPost]
        public IHttpActionResult MarkInappropriate([FromBody] MarkInappropriateRequest request)
        {
            request.Validate();

            var server = Group.GetWritable(request.ServerID);
            if (server == null)
            {
                return NotFound();
            }

            server.SetInappropriateFlag(true);
            return StatusCode(HttpStatusCode.Accepted);
        }

        [Route("discover/quarantine/{serverID}")]
        [HttpPost]
        public IHttpActionResult UnmarkInappropriate(Guid serverID)
        {
            var server = Group.GetWritable(serverID);
            if (server == null)
            {
                return NotFound();
            }

            server.SetInappropriateFlag(false);
            return StatusCode(HttpStatusCode.Accepted);
        }

        #endregion
    }
}
