﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Friends.MicroService;
using Curse.Friends.Data;
using Curse.Aerospike;
using Curse.Friends.Configuration;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Queues;
using Curse.Friends.Data.Search;
using Curse.Friends.Statistics;
using Curse.Extensions;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Enums;
using Curse.Friends.GroupsWebService.Contracts;
using Curse.Friends.NotificationContracts;
using Curse.Logging;
using Curse.Friends.MicroService.Filters;
using Curse.Friends.Tracing;

namespace Curse.Friends.GroupsWebService.Controllers
{
    [RoutePrefix("groups")]
    public class GroupsController : MicroServiceController
    {

        #region Creation
        
        /// <summary>
        /// Creates a normal group (allows up to 100 members, and no channels)
        /// </summary>
        [Route("")]
        [HttpPost]
        [ResponseType(typeof(GroupNotification))]
        [SocialBanFilter]
        public IHttpActionResult Create(CreateGroupRequest request)
        {
            request.Validate();

            // See if this user is the owner of too many groups, or too recent of a group:            
            var me = GetCurrentUserAndRegion();

            if (FriendsServiceConfiguration.Mode == ConfigurationMode.Release)
            {
                var memberships = GroupMember.GetAllByUserID(Token.UserID);

                if (memberships.Count(p => p.IsRootGroup && !p.IsDeleted && p.BestRoleRank == 0) >= Group.MaxGroupsOwned)
                    // TODO: Figure out a better way to throttle this
                {
                    Logger.Warn("User attempted to create a group, and is the owner of too many.");
                    return BadRequest("You have too many servers.");
                }
            }

            // Ensure that every user being added to the group is a friend:
            var filteredParticipants = (request.RecipientsUserIDs != null)
                ? Friendship.GetAllConfirmed(me.Region.RegionID, Token.UserID)
                    .Where(p => request.RecipientsUserIDs.Contains(p.OtherUserID))
                    .ToArray()
                : new Friendship[0];

            // If the user is trying to add any users to the group, ensure that they are valid
            if (request.RecipientsUserIDs != null && request.RecipientsUserIDs.Any() && !filteredParticipants.Any())
            {
                Logger.Warn("User attempted to create a group with a user that is not a friend.", new { Token.UserID, request });
                return BadRequest("Invalid recipients");
            }

            // Create the main group record
            var members = filteredParticipants.Select(p => new NewGroupMember(p)).ToArray();
            var creator = new NewGroupMember(me.Region, me.User, null);
            var newGroup = Group.CreateNormalGroup(creator, request.Title, members, request.OwnerRoleName, request.MemberRoleName);

            FriendsStatsManager.Current.GroupsCreatedByType.Track((int)GroupType.Normal);
            
            ProcessProvisionalUsers(me.User, newGroup, members.Select(p => p.UserID).ToArray());

            return Ok(newGroup.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, false, true, true));
        }

        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("GroupsController");

        private void ProcessProvisionalUsers(User user, Group group, int[] allUserIDs)
        {
            var provisionalAdditions = ExternalAccount.MultiGetAllTwitchAccountsByUserIDs(allUserIDs)
                                                      .Where(p => p.Value.IsAutoProvisioned && p.Value.IsMerged)
                                                      .Select(p => p.Value)
                                                      .ToArray();

            if (!provisionalAdditions.Any())
            {
                return;
            }

            var senderID = user.UserID;
            var message = $"I've added you to a new group, {group.Title}. You can join me by downloading the Twitch desktop app at https://download.twitch.tv";
            
            // Let any provisional accounts know about the group, via whisper
            foreach (var recipient in provisionalAdditions)
            {
                var recipientID = recipient.MergedUserID;
                PrivateMessageWorker.Create(senderID, null, PrivateConversation.GenerateConversationID(senderID, recipientID), message, Guid.NewGuid(), null);
            }

            FilteredLogger.Log(user, "Sent whispers to provisional accounts, informing them of their new group membership.", new { accounts = provisionalAdditions.Select(p => p.ExternalUsername).ToArray() });
        }

        /// <summary>
        /// Modify the settings for server
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("{groupID}/settings")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult Settings(Guid groupID, ModifyGroupSettingsRequest request)
        {
            request.Validate();

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return NotFound();
            }        

            group.ChangeGroupSettings(Token.UserID, request.Title, request.MessageOfTheDay, request.ForcePushToTalk);
            return Ok();
        }
     
        #endregion       
        
        #region Member Management

        [Route("{groupID}/members")]
        [HttpGet]
        [ResponseType(typeof(IEnumerable<GroupMemberContract>))]
        public IHttpActionResult GetMembers(Guid groupID, bool actives = true, int page = 1, int pageSize = 50)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (group.Type != GroupType.Large || !group.IsRootGroup)
            {
                return BadRequest("Invalid group");
            }

            if (pageSize < 10 || pageSize > 50)
            {
                return BadRequest("Invalid page size.");
            }

            if (page < 0 || page > 10000)
            {
                return BadRequest("Invalid page number.");
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);
            var members = actives ? GroupMemberManager.GetByGroup(groupID, pageSize, page) : GroupMemberManager.GetInactiveByGroup(groupID, pageSize, page);
            return GetGroupMemberSearchNotifications(members);
        }

        /// <summary>
        /// Adds the user list provided in the request from the group and notifies the current users that belong to group
        /// </summary>
        [Route("{groupID}/members")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult AddMembers(Guid groupID, [FromBody] int[] userIDs)
        {

            if (userIDs == null || userIDs.Length == 0)
            {
                return BadRequest();
            }

            var group = Group.GetWritable(groupID);

            if (group == null)
            {
                return NotFound();
            }

            if (!group.CanHaveMembers)
            {
                return BadRequest();
            }

            var distinctUserIDs = new HashSet<int>(userIDs);

            var me = GetCurrentUserAndRegion();

            // Ensure that every user being added to the group is a friend:
            var filteredParticipants = Friendship.GetAllConfirmed(me.Region.RegionID, Token.UserID)
                .Where(p => distinctUserIDs.Contains(p.OtherUserID))
                .ToArray();

            //if users are not friends
            if (!filteredParticipants.Any())
            {
                return BadRequest();
            }

            filteredParticipants = filteredParticipants.Where(p => !group.IsMember(p.OtherUserID)).ToArray();

            // Remove any users are already exist
            if (!filteredParticipants.Any())
            {
                //If there are no users to add, do not service the request
                return BadRequest();
            }

            var filteredUserIDs = filteredParticipants.Select(p => p.UserID).ToArray();
            var endpoints = ClientEndpoint.MultiGetMostRecentForUserIDs(filteredUserIDs);
            var newGroupMembers = filteredParticipants.Select(p => new NewGroupMember(p, endpoints.GetValueOrDefault(p.UserID), group.DefaultRole)).ToArray();
            
            // Add the invited users
            group.AddUsers(Token.UserID, newGroupMembers);

            ProcessProvisionalUsers(me.User, group, newGroupMembers.Select(p => p.UserID).ToArray());

            return Ok();
        }
        
        private IHttpActionResult GetGroupMemberSearchNotifications(GroupMemberSearchModel[] members)
        {
            var userStats = UserStatistics.GetAllByUserIDs(members.Select(p => p.UserID)).ToDictionary(p => p.UserID);
            return Ok(members.Select(p => p.ToSearchNotification(userStats.GetValueOrDefault(p.UserID))).ToArray());
        }

        [Route("{groupID}/members/simple-search")]
        [HttpGet]
        [ResponseType(typeof(IEnumerable<GroupMemberContract>))]
        public IHttpActionResult QuickSearchMembers(Guid groupID, string query)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var members = GroupMemberManager.SearchGroupMembers(groupID, new GroupMemberSearch
            {
                Username = query,                
                PageSize = 50,
                PageNumber = 1,
                SortAscending = false,
                SortType = GroupMemberSearchSortType.DateLastActive
            });

            return GetGroupMemberSearchNotifications(members);
        }

        [Route("{groupID}/members/search")]
        [HttpPost]
        [ResponseType(typeof(IEnumerable<GroupMemberContract>))]
        public IHttpActionResult SearchMembers(Guid groupID, [FromBody] GroupMemberSearchRequest request)
        {
            request.Validate();

            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var members = GroupMemberManager.SearchGroupMembers(groupID, new GroupMemberSearch
            {
                Username = request.Username,
                RoleID = request.RoleID,
                PageSize = request.PageSize ?? 50,
                PageNumber = request.Page ?? 1,
                SortType = request.SortType ?? GroupMemberSearchSortType.Default,
                SortAscending = request.SortAscending ?? true
            });

            return GetGroupMemberSearchNotifications(members);
        }
        
        [Route("{groupID}/members/{userID}")]
        [HttpGet]
        [ResponseType(typeof(GroupMemberContract))]
        public IHttpActionResult GetMember(Guid groupID, int userID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                //if the requested group does not exist
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var userStats = UserStatistics.GetByUserOrDefault(userID);
            if (userStats == null)
            {
                return NotFound();
            }

            var member = group.GetMember(userID, true);
            if (member == null)
            {
                return NotFound();
            }

            // Get their external accounts
            var externalAccounts = ExternalAccount.GetAllForUser(userID);

            return Ok(member.ToNotification(userStats, externalAccounts));
        }


        /// <summary>
        /// Removes the users provided in the request from the group and notifies the current users that belong to group
        /// </summary>
        [Route("{groupID}/members/{userID}")]
        [HttpDelete]
        [SocialBanFilter]
        public IHttpActionResult RemoveMember(Guid groupID, int userID)
        {
            if (Token.UserID == userID)
            {
                // Treat it as a leave instead of a remove
                return Leave(groupID);
            }

            var group = Group.GetWritable(groupID);
            if (group == null)
            {
                //if the requested group does not exist
                return NotFound();
            }

            var me = GetCurrentUserAndRegion();
            group.RemoveUsers(me.User.UserID, new HashSet<int> { userID });

            var memberCall = group.GetCurrentCallGroup(userID);
            if (memberCall.HasValue)
            {
                ServiceClients.FriendsServiceClients.Instance.CallsAdmin.KickUsersFromCall(new Curse.ServiceClients.Contracts.KickUsersFromCallRequest
                {
                    GroupID = memberCall.Value,
                    RequestorUserID = me.User.UserID,
                    UserIDs = new[] { userID },
                    Reason = Curse.ServiceClients.Contracts.KickReason.UserRemoved
                });
            }
                   
            return Ok();

        }

        /// <summary>
        /// Transfers the authenticated user's ownership to the supplied user.
        /// </summary>
        /// <param name="groupID">The group's ID</param>
        /// <param name="userID">The new owner's ID</param>
        /// <returns></returns>
        [Route("{groupID}/members/transfer-ownership")]
        [HttpPost]
        [ResponseType(typeof (void))]
        public IHttpActionResult TransferOwnership(Guid groupID, [FromBody] int userID)
        {
            var group = Group.GetWritable(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.TransferOwnership(Token.UserID, userID);

            return StatusCode(HttpStatusCode.NoContent);
        }

        #endregion

        #region End User Endpoints

        /// <summary>
        /// Returns a detailed group response. This includes the full member list for small groups.
        /// </summary>
        [Route("{groupID}")]
        [HttpGet]
        [ResponseType(typeof(GroupNotification))]
        public IHttpActionResult Details(Guid groupID, bool showDeletedChannels = false)
        {
            var group = Group.GetByID(groupID);

            if (group == null)
            {
                return NotFound();
            }

            if (!group.IsRootGroup)
            {
                return BadRequest("You can only get group details for root groups");
            }

            // Ensure the user has access
            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            GroupNotification response;

            // For large groups, return the entire channel tree
            if (group.Type == GroupType.Large)
            {
                response = group.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, Token.UserID, showDeletedChannels);
            }
            else
            {
                response = group.ToGroupNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, Token.UserID);

                response.Channels = new ChannelContract[0];
            }


            return Ok(response);
        }

        /// <summary>
        /// Bulk update of favorite groups.
        /// </summary>
        /// <param name="requests">The favorite requests</param>
        [Route("favorites")]
        [HttpPost]
        public IHttpActionResult ToggleFavorites([FromBody] ToggleFavoriteRequest[] requests)
        {
            var validRequests = new List<ToggleFavoriteRequest>();
            foreach (var request in requests)
            {
                try
                {
                    request.Validate();
                    validRequests.Add(request);
                }
                catch (DataValidationException)
                {
                    // Do nothing
                }
            }

            if (validRequests.Count == 0)
            {
                return BadRequest("No requests were valid");
            }

            foreach (var request in validRequests)
            {
                ToggleFavorite(request.GroupID, request.IsFavorite);
            }

            return Ok();
        }

        /// <summary>
        /// Marks the requested group as User's favorite and notifies all the connected Endpoints 
        /// about the change.
        /// </summary>
        /// <returns></returns>
        [Route("{groupID}/favorite")]
        [HttpPost]
        public IHttpActionResult Favorite(Guid groupID)
        {
            return ToggleFavorite(groupID, true);
        }

        private IHttpActionResult ToggleFavorite(Guid groupID, bool isFavorite)
        {
            var group = Group.GetLocal(groupID);

            if (group == null)
            {
                //If the requested group does not exist
                return NotFound();
            }

            var groupMembership = group.CheckPermission(GroupPermissions.Access, Token.UserID, null, true);

            groupMembership.IsFavorite = isFavorite;
            groupMembership.Update(p => p.IsFavorite);

            //fan out to all end points of user
            ClientEndpoint.DispatchNotification(Token.UserID,
                ep => GroupPreferenceChangedNotifier.Create(ep, groupMembership));

            return Ok();
        }

        /// <summary>
        /// Marks the requested group as User's favorite and notifies all the connected Endpoints 
        /// about the change.
        /// </summary>
        /// <returns></returns>
        [Route("{groupID}/unfavorite")]
        [HttpPost]
        public IHttpActionResult Unfavorite(Guid groupID)
        {
            return ToggleFavorite(groupID, false);
        }
        
        /// <summary>
        /// Removes the requested UserId from the groupID of LeaveGroupRequest
        /// </summary>
        [Route("{groupID}/leave")]
        [HttpPost]
        public IHttpActionResult Leave(Guid groupID)
        {
            var currentUser = GetCurrentUserAndRegion();

            var group = Group.GetWritable(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (!group.CanHaveMembers)
            {
                return BadRequest();
            }

            group.Leave(currentUser.User.UserID, currentUser.Region.RegionID);

            var memberCall = group.GetCurrentCallGroup(currentUser.User.UserID);
            if (memberCall.HasValue)
            {
                ServiceClients.FriendsServiceClients.Instance.CallsAdmin.KickUsersFromCall(new Curse.ServiceClients.Contracts.KickUsersFromCallRequest
                {
                    GroupID = memberCall.Value,
                    RequestorUserID = currentUser.User.UserID,
                    UserIDs = new[] { currentUser.User.UserID },
                    Reason = Curse.ServiceClients.Contracts.KickReason.UserLeft
                });
            }

            return Ok();

        }

        /// <summary>
        /// Removes the requested UserId from the groupID of LeaveGroupRequest
        /// </summary>
        [Route("{groupID}/nickname")]
        [HttpPost]
        [ResponseType(typeof(void))]
        [SocialBanFilter]
        public IHttpActionResult Nickname(Guid groupID, GroupMemberNicknameRequest request)
        {
            request.Validate();

            var user = GetUserAndRegion(Token.UserID);

            if (user == null)
            {
                return NotFound();
            }

            var group = Group.GetByID(groupID);
            
            if (group == null)
            {
                return NotFound();
            }

            // Ensure they have access
            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            if (!group.CanHaveMembers)
            {
                return BadRequest();
            }
            
            var status = group.ChangeMemberNickname(Token.UserID, user.User.Username, user.User.DisplayName, request.Nickname);
            
            if (status != GroupMemberNicknameStatus.Success)
            {
                return BadRequest(status);
            }

            return Ok();
        }

        /// <summary>
        /// Bulk update of notification preferences for groups.
        /// </summary>
        /// <param name="preferences">The notification preference requests.</param>
        [HttpPost]
        [Route("notification-preferences")]
        public IHttpActionResult ChangeNotificationPreferencesBulk([FromBody] BulkGroupNotificationPreferences[] preferences)
        {
            var validPreferences = new List<BulkGroupNotificationPreferences>();

            foreach (var preference in preferences)
            {
                try
                {
                    preference.Validate();
                    validPreferences.Add(preference);
                }
                catch (DataValidationException)
                {
                    // Ignore
                }
            }

            if (validPreferences.Count == 0)
            {
                return BadRequest("None of the preference changes are valid");
            }

            var groups = Group.MultiGetLocal(new HashSet<Guid>(validPreferences.Select(p => p.GroupID)).Select(g => new KeyInfo(g))).ToDictionary(g => g.GroupID);
            var myEndpoints = ClientEndpoint.GetAllDeliverable(Token.UserID);
            foreach (var preference in validPreferences)
            {
                Group group;
                if (!groups.TryGetValue(preference.GroupID, out group))
                {
                    continue;
                }

                GroupMember myMembership;
                if (!group.HasPermission(GroupPermissions.Access, Token.UserID, out myMembership, true))
                {
                    continue;
                }

                myMembership.UpdateUserPreference(preference.Preference, preference.FilterSet);

                ClientEndpoint.DispatchNotification(myEndpoints, ep => GroupPreferenceChangedNotifier.Create(ep, myMembership), null);
            }

            return Ok();
        }

        /// <summary>
        /// Changes notification perference for a user to a group. Adds filters for notification.
        /// </summary>
        [Route("{groupID}/notification-preferences")]
        [HttpPost]
        public IHttpActionResult ChangeNotificationPreferences(Guid groupID, GroupNotificationPreferences preferences)
        {
            preferences.Validate();

            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var myMembership = group.CheckPermission(GroupPermissions.Access, Token.UserID, null, true);


            myMembership.UpdateUserPreference(preferences.Preference, preferences.FilterSet);

            //fan out to all endpoints of user
            ClientEndpoint.DispatchNotification(Token.UserID,
                ep => GroupPreferenceChangedNotifier.Create(ep, myMembership));

            return Ok();
        }

        #endregion

        [Route("{groupID}/events")]
        [HttpGet]
        [ResponseType(typeof(IEnumerable<GroupEventContract>))]
        public IHttpActionResult GetGroupEvents(Guid groupID, int? pageSize = null, int? pageNumber = null)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (!group.IsRootGroup)
            {
                return BadRequest();
            }

            group.CheckPermission(GroupPermissions.AccessAdminPanel, Token.UserID);

            var events = GroupEventManager.GetByGroup(groupID, pageSize, pageNumber);
            return Ok(events.Select(e => e.ToContract()));
        }

        [Route("{groupID}/events/search")]
        [HttpPost]
        [ResponseType(typeof (IEnumerable<GroupEventContract>))]
        public IHttpActionResult SearchGroupEvents(Guid groupID, [FromBody] SearchGroupEventsRequest request)
        {
            request.Validate();

            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (!group.IsRootGroup)
            {
                return BadRequest();
            }

            group.CheckPermission(GroupPermissions.AccessAdminPanel, Token.UserID);

            var events = GroupEventManager.Search(new GroupEventSearch
            {
                GroupID = groupID,
                PageSize = request.PageSize,
                PageNumber = request.PageNumber,
                IncludedEventCategories = request.IncludedCategories,
                ExcludedEventCategories = request.ExcludedCategories,
                IncludedEventTypes = request.IncludedEventTypes,
                ExcludedEventTypes = request.ExcludedEventTypes,
                UsernameQuery = request.UsernameQuery,
                SortAscending = request.SortAscending,
                SortType = request.SortType
            });

            return Ok(events.Select(e => e.ToContract()));
        }

    }
}