﻿using System.Web.Http;
using System.Web.Http.Description;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.GroupsWebService.Configuration;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.GroupsWebService.Contracts;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.GroupsWebService.Controllers
{
    [RoutePrefix("invitations/{inviteCode}")]
    public class InvitationsController : MicroServiceController
    {
        /// <summary>
        /// Get information about the specified invitation.
        /// </summary>
        /// <param name="inviteCode">The invite code.</param>
        [HttpGet]
        [ResponseType(typeof(GroupInvitationNotification))]
        [Route("")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult Get(string inviteCode)
        {
            // Get the group in question
            var invite = GroupInvitation.GetByInviteCode(inviteCode);
            if (invite == null || (invite.MaxUses > 0 && invite.TimesUsed >= invite.MaxUses))
            {
                return NotFound();
            }

            // Get the group
            var group = Group.GetByID(invite.GroupID);

            if (group == null)
            {
                return NotFound();
            }

            // Get the channel
            var channel = Group.GetByID(invite.ChannelID);
            if (channel == null)
            {
                return NotFound();
            }

            return Ok(invite.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, group, channel, GroupsWebConfiguration.Instance.GroupInviteUrlFormat));
        }

        [HttpGet]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        [ResponseType(typeof(GetInvitationDisplayDetailsResponse))]
        [Route("display")]
        public IHttpActionResult GetDisplayDetails(string inviteCode)
        {
            var invite = GroupInvitation.GetByInviteCode(inviteCode);
            if (invite == null || invite.Status == GroupInvitationStatus.Defunct)
            {
                return NotFound();
            }
            // Get the group
            var group = Group.GetByID(invite.GroupID);

            if (group == null)
            {
                return NotFound();
            }

            var channel = group.GetChildGroup(invite.ChannelID);
            if (channel == null)
            {
                return NotFound();
            }

            var searchSettings = group.GetServerSearchSettingsOrDefault();

            return Ok(new GetInvitationDisplayDetailsResponse
            {
                InviteCode = invite.InviteCode,
                GroupID = group.GroupID,
                GroupTitle = group.Title,
                ChannelID = channel.GroupID,
                ChannelTitle = channel.Title,
                CreatorID = invite.CreatorID,
                CreatorUsername = invite.CreatorName,
                MemberCount = group.MemberCount,
                Description = searchSettings.Description
            });
        }

        /// <summary>
        /// Redeem a group invite.
        /// </summary>
        /// <param name="inviteCode">The invite code.</param>
        [HttpPost]
        [ResponseType(typeof(GroupInvitationRedeemResponse))]
        [Route("")]
        [SocialBanFilter]
        public IHttpActionResult Redeem(string inviteCode)
        {
            // Get the group invite in question
            var groupInvite = GroupInvitation.GetByInviteCode(inviteCode);
            if (groupInvite == null || (groupInvite.MaxUses > 0 && groupInvite.TimesUsed >= groupInvite.MaxUses))
            {
                return NotFound();
            }

            // Get the group
            var group = Group.GetWritable(groupInvite.GroupID);
            if (group == null || group.IsDeleted)
            {
                return NotFound();
            }

            // Get the user
            var userAndRegion = GetCurrentUserAndRegion();

            // Process the invitation
            group.ProcessInvitation(groupInvite, userAndRegion.User, userAndRegion.Region, GetCurrentIpAddress().ToString());

            var resp = new GroupInvitationRedeemResponse
            {
                Group = group.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, userAndRegion.User.UserID),
                ChannelID = groupInvite.ChannelID
            };

            return Ok(resp);
        }
    }
}