﻿using System;
using System.Linq;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.Models;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.UserEvents;

namespace Curse.Friends.GroupsWebService.Controllers
{
    [RoutePrefix("groups")]
    public class PrivateMessagesController : MicroServiceController
    {
        [Route("{groupID}/private-messages")]
        [HttpGet]
        [ResponseType(typeof(GroupPrivateConversationContract[]))]
        public IHttpActionResult GetAll(Guid groupID)
        {            
            var group = Group.GetByID(groupID);

            if (group == null)
            {
                return NotFound();
            }

            if (!group.CanHavePrivateMessages)
            {
                return BadRequest("This group cannot host private messages.");
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var conversations = GroupPrivateConversation.GetAllByGroupIDAndUserID(groupID,  Token.UserID).Where(p => !p.IsBlockedByThem && !p.IsHidden).ToArray();
            var memberDisplayNames = GroupMember.MultiGetLocal(conversations.Select(c => new KeyInfo(c.GroupID, c.OtherUserID))).ToDictionary(m => m.UserID, m => m.GetTitleName());
            var contracts = conversations.Select(p => p.ToContract(memberDisplayNames.GetValueOrDefault(p.OtherUserID))).ToArray();

            return Ok(contracts);

        }

        /// <summary>
        /// Blocks a user from sending you private messages.
        /// </summary>
        /// <param name="groupID">The group/server ID of the community.</param>
        /// <param name="userID">The id of the user that you wish to block.</param>
        /// <returns></returns>
        [Route("{groupID}/private-messages/{userID}/block")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult Block(Guid groupID, int userID)
        {
            return ToggleBlock(groupID, userID, true);
        }

        /// <summary>
        /// Unblocks a user, allowing them to send you private messages.
        /// </summary>
        /// <param name="groupID">The group/server ID of the community.</param>
        /// <param name="userID">The id of the user that you wish to unblock.</param>
        /// <returns></returns>
        [Route("{groupID}/private-messages/{userID}/unblock")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult Unblock(Guid groupID, int userID)
        {
            return ToggleBlock(groupID, userID, false);
        }

        private IHttpActionResult ToggleBlock(Guid groupID, int userID, bool isBlocked)
        {
            var group = Group.GetByID(groupID);

            if (group == null)
            {
                return NotFound();
            }

            if (!group.CanHavePrivateMessages)
            {
                return BadRequest("This group cannot host private messages.");
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            // Update their side
            var theirConvo = GroupPrivateConversation.GetBySenderIDAndRecipientIDAndGroupID(userID, Token.UserID, groupID)
                             ?? GroupPrivateConversation.Create(groupID, userID, Token.UserID);
            if (theirConvo.IsBlockedByThem != isBlocked)
            {
                theirConvo.IsBlockedByThem = isBlocked;
                theirConvo.Update(c => c.IsBlockedByThem);
            }

            // Update my side
            var myConvo = GroupPrivateConversation.GetBySenderIDAndRecipientIDAndGroupID(Token.UserID, userID, groupID)
                          ?? GroupPrivateConversation.Create(groupID, Token.UserID, userID);
            if (myConvo.IsBlockedByMe != isBlocked)
            {
                myConvo.IsBlockedByMe = isBlocked;
                myConvo.Update(c => c.IsBlockedByMe);
            }

            if (isBlocked)
            {
                new BlockUserEvent { UserID = Token.UserID, OtherUserID = userID, Source = BlockUserSource.PrivateMessage }.Enqueue();
            }
            else
            {
                new UnblockUserEvent { UserID = Token.UserID, OtherUserID = userID }.Enqueue();
            }

            return Ok();
        }


        /// <summary>
        /// Hides the PM conversation
        /// </summary>
        /// <param name="groupID">The group/server ID of the community.</param>
        /// <param name="userID">The id of the user that you wish to block.</param>
        /// <returns></returns>
        [Route("{groupID}/private-messages/{userID}/hide")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult Hide(Guid groupID, int userID)
        {
            return ToggleHidden(groupID, userID, true);
        }

        /// <summary>
        /// Unhides the PM conversation
        /// </summary>
        /// <param name="groupID">The group/server ID of the community.</param>
        /// <param name="userID">The id of the user that you wish to unblock.</param>
        /// <returns></returns>
        [Route("{groupID}/private-messages/{userID}/unhide")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult Unhide(Guid groupID, int userID)
        {
            return ToggleHidden(groupID, userID, false);
        }

        private IHttpActionResult ToggleHidden(Guid groupID, int userID, bool isHidden)
        {
            var group = Group.GetByID(groupID);

            if (group == null)
            {
                return NotFound();
            }

            if (!group.CanHavePrivateMessages)
            {
                return BadRequest("This group cannot host private messages.");
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            
            // Update my side
            var convo = GroupPrivateConversation.GetBySenderIDAndRecipientIDAndGroupID(Token.UserID, userID, groupID);

            if (convo == null)
            {
                return NotFound();
            }

            convo.IsHidden = isHidden;
            convo.Update(p => p.IsHidden);

            return Ok();
        }
    }
}