﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Security.Policy;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Friends.Data.Search;
using Curse.Friends.GroupsWebService.Contracts;
using Curse.Friends.MicroService;
using Curse.Friends.Statistics;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Enums;
using Curse.Friends.GroupsWebService.Configuration;
using Curse.Friends.GroupsWebService.Utils;
using Curse.Logging;
using Curse.Friends.MicroService.Filters;
using Curse.Friends.Tracing;

namespace Curse.Friends.GroupsWebService.Controllers
{
    [RoutePrefix("servers")]
    public class ServersController : MicroServiceController
    {

        private static readonly FilteredUserLogger FilterLogger = new FilteredUserLogger("Servers");

        /// <summary>
        /// Performs a simple server search with an optional query.
        /// </summary>
        [Route("")]
        [HttpGet]
        [ResponseType(typeof (IEnumerable<GroupSearchModelContract>))]
        public IHttpActionResult SimpleSearch(string query=null, int? pageSize=null, int? pageNumber=null)
        {
            return Ok(DoSearch(new GroupSearch {Query = query, PageNumber = pageNumber, PageSize = pageSize}));
        }

        /// <summary>
        /// Advanced group search where more filtering options are available.
        /// </summary>
        /// <param name="request">The search request.</param>
        [Route("search")]
        [HttpPost]
        [ResponseType(typeof (IEnumerable<GroupSearchModelContract>))]
        public IHttpActionResult Search([FromBody] SearchServersRequest request)
        {
            request.Validate();

            return Ok(DoSearch(new GroupSearch
            {
                Games = request.Games,
                GroupTitle = request.GroupTitle,
                IsPublic = request.IsPublic,
                MaxMemberCount = request.MaxMemberCount,
                MinMemberCount = request.MinMemberCount,
                OwnerUsername = request.OwnerUsername,
                PageNumber = request.PageNumber,
                PageSize = request.PageSize,
                Query = request.Query,
                Tags = request.Tags,
                IsFeatured = request.IsFeatured,
                SortType = request.SortType ?? GroupSearchSortType.Default,
                SortAscending = request.SortAscending ?? false,
                IsStreaming = request.IsStreaming,
                IncludeInappropriate = request.IncludeInappropriateContent,
                SubType = request.SubType
            }));
        }

        private static IEnumerable<GroupSearchModelContract> DoSearch(GroupSearch search)
        {
            return GroupSearchManager.Search(search).Select(g => g.ToNotification());
        }

        // TODO: Use this approach once our generator is fixed
        ///// <summary>
        ///// Performs a discover search for servers and communities.
        ///// </summary>
        //[Route("discover")]
        //[HttpGet]
        //[ResponseType(typeof(DiscoverResponse))]
        //public IHttpActionResult Discover([FromUri] DiscoverRequest request)
        //{
        //    return Ok(DoDiscover(request));
        //}

        /// <summary>
        /// Performs a simple discover search for servers and communities with an optional query.
        /// </summary>
        [Route("simple-discover")]
        [HttpGet]
        [ResponseType(typeof(IEnumerable<GroupSearchModelContract>))]
        public IHttpActionResult SimpleDiscover(string query = null, int? pageSize = null, int? pageNumber = null)
        {
            return Ok(DoDiscover(new DiscoverRequest { Query = query, PageNumber = pageNumber, PageSize = pageSize }));
        }

        /// <summary>
        /// Advanced discover search where more filtering options are available.
        /// </summary>
        /// <param name="request">The search request.</param>
        [Route("advanced-discover")]
        [HttpPost]
        [ResponseType(typeof(DiscoverResponse))]
        public IHttpActionResult AdvancedDiscover([FromBody] DiscoverRequest request)
        {
            request.Validate();

            return Ok(DoDiscover(request));
        }

        private static DiscoverResponse DoDiscover(DiscoverRequest request)
        {
            request = request ?? new DiscoverRequest();

            var sortType = request.SortType ?? DiscoverSortType.Default;
            var groupSortType = GetGroupSortType(sortType);
            var groupSearch = new GroupSearch
            {
                Games = request.Games,
                GroupTitle = request.GroupTitle,
                IsPublic = request.IsPublic,
                MaxMemberCount = request.MaxMemberCount,
                MinMemberCount = request.MinMemberCount,
                OwnerUsername = request.OwnerUsername,
                PageNumber = request.PageNumber,
                PageSize = request.PageSize,
                Query = request.Query,
                Tags = request.Tags,
                IsFeatured = request.IsFeatured,
                SortType = groupSortType,
                SortAscending = request.SortAscending ?? false,
                IsStreaming = request.IsStreaming,
                IncludeInappropriate = request.IncludeInappropriateContent,
                SubType = request.SubType
            };
            var groupResults = GroupSearchManager.Search(groupSearch);

            var communitySortType = GetExternalCommunitySortType(sortType);
            var communitySearch = new ExternalCommunitySearch
            {
                Name = request.GroupTitle,
                PageNumber = request.PageNumber,
                PageSize = request.PageSize,
                Query = request.Query,
                SortType = communitySortType,
                SortAscending = request.SortAscending ?? false,
                IsLive = request.IsStreaming,
            };
            var communityResults = ExternalCommunitySearchManager.Search(communitySearch);

            // Filter communities out
            var allGroupIDs = new HashSet<string>(groupResults.Select(g => g.GroupID.ToString()));
            communityResults = communityResults.Where(c => c.MappedGroups == null || !allGroupIDs.Overlaps(c.MappedGroups)).ToArray();

            return new DiscoverResponse
            {
                Groups = groupResults.Select(g => g.ToNotification()).ToArray(),
                ExternalCommunities = communityResults.Select(g => g.ToNotification(FriendsServiceConfiguration.Instance.TwitchChannelUrlFormat)).ToArray()
            };
        }

        private static GroupSearchSortType GetGroupSortType(DiscoverSortType sortType)
        {
            switch (sortType)
            {
                case DiscoverSortType.DateFeatured:
                    return GroupSearchSortType.DateFeatured;
                case DiscoverSortType.DateLive:
                    return GroupSearchSortType.DateStreaming;
                case DiscoverSortType.MemberCount:
                    return GroupSearchSortType.MemberCount;
                case DiscoverSortType.Title:
                    return GroupSearchSortType.Title;
                default:
                    return GroupSearchSortType.Default;
            }
        }

        private static ExternalCommunitySearchSortType GetExternalCommunitySortType(DiscoverSortType sortType)
        {
            switch (sortType)
            {
                case DiscoverSortType.Title:
                    return ExternalCommunitySearchSortType.Title;
                case DiscoverSortType.DateLive:
                    return ExternalCommunitySearchSortType.DateLive;
                default:
                    return ExternalCommunitySearchSortType.Default;
            }
        }

        /// <summary>
        /// Creates a custom server
        /// </summary>
        [Route("creation-options")]
        [HttpGet]
        [ResponseType(typeof (ServerCreationOptionsResponse))]
        public IHttpActionResult CreationOptions()
        {
            var me = GetCurrentUserAndRegion();

            // Show the sync button for partners or users with at least 10 followers
            var resp = new ServerCreationOptionsResponse();
            

            if (me.User.IsMerged)
            {
                var externalAccount = me.User.GetTwitchAccount();
                if (externalAccount != null)
                {
                    // Ensure's the Twitch channel is up to date
                    var externalCommunity = TwitchModelHelper.CreateOrUpdateCommunity(externalAccount.ExternalID, externalAccount);                                        
                    
                    if (externalCommunity != null)
                    {
                        resp.SyncAvailable = true;
                        resp.SyncByDefault = !externalCommunity.HasMappedGroups && (externalCommunity.CanHaveSubs || externalCommunity.Followers >= 10);
                        resp.PublicAvailable = externalCommunity.CanHaveSubs || externalCommunity.Followers >= 5000;
                        resp.PublicByDefault = true;
                        resp.AvatarUrl = externalCommunity.AvatarUrl;
                        resp.ServerTitle = externalCommunity.ExternalDisplayName;
                        resp.ServerUrl = VanityUrl.FindNextAvailable("/" + externalCommunity.ExternalName);
                    }
                }                
            }
            
            return Ok(resp);
        }

        /// <summary>
        /// Creates a custom server
        /// </summary>
        [Route("")]
        [HttpPost]
        [ResponseType(typeof(GroupNotification))]
        [SocialBanFilter]
        public IHttpActionResult Create(CreateServerRequest request)
        {
            request.Validate();            

            // See if this user is the owner of too many groups, or too recent of a group:            
            var me = GetCurrentUserAndRegion();

#if CONFIG_RELEASE

            var memberships = GroupMember.GetAllByUserID(Token.UserID);

            if (memberships.Count(p => p.IsRootGroup && p.BestRoleRank == 1 && !p.IsDeleted) >= Group.MaxServersOwned) // TODO: Figure out a better way to throttle this
            {
                return BadRequest("You have too many servers.");
            }
#endif

            // Only allow merged accounts to create synced servers
            if (request.IsSynced)
            {
                return CreateSyncedServer(request, me);
            }
            else
            {
                return CreateNormalServer(request, me);
            }
            
        }

        private IHttpActionResult CreateNormalServer(CreateServerRequest request, UserRegionInfo me)
        {
            var creator = new NewGroupMember(me.Region, me.User, GetCurrentIpAddress().ToString());
            var newGroup = Group.CreateServer(creator, request.Title, request.TextChannelTitle,
                request.VoiceChannelTitle, request.OwnerRoleName, request.GuestRoleName, request.ModeratorRoleName, request.IsPublic);


            FriendsStatsManager.Current.GroupsCreatedByType.Track((int)GroupType.Large);

            var contract = newGroup.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, Token.UserID);
            FilterLogger.Log(me.User, null, "User created a new non-synced server", new { request, server = contract });

            return Ok(contract);
        }

        private IHttpActionResult CreateSyncedServer(CreateServerRequest request, UserRegionInfo me)
        {
            if (!me.User.IsMerged)
            {
                Logger.Warn("Attempt to create a synced server without having a merged account.", new { me.User.UserID, me.User.Username });
                return BadRequest("Account must be merged to create a synced server.");
            }
            
            var externalAccount = me.User.GetTwitchAccount();

            if (externalAccount == null)
            {
                Logger.Warn("Attempt to create a synced server without a valid external account.", new { me.User.UserID, me.User.Username });
                return BadRequest("Account must be merged with a valid Twitch account.");
            }

            // Ensure's the Twitch channel is up to date
            var externalCommunity = TwitchModelHelper.CreateOrUpdateCommunity(externalAccount.ExternalID, externalAccount);
            
            // If it cannot be retrieved, fail
            if (externalCommunity == null)
            {            
                Logger.Warn("Failed to create or update external community. Unable to proceed with server creation.");
                return BadRequest("Unable to get or create the external community for this Twitch account.");                
            }

            // Now find a custom URL
            var url = VanityUrl.FindNextAvailable(externalCommunity.ExternalName);
            if (url == null)
            {
                Logger.Warn("Failed to find an available custom URL for channel. A custom URL will not be used.", new { externalCommunity.ExternalName, externalCommunity.IsPartnered, externalCommunity.IsAffiliate, externalCommunity.DisplayName, externalCommunity.Followers });
            }
            
            // Create the synced community object
            var communitySync = new NewCommunitySync(externalCommunity, "subscribers");            
            
            // Creator is simply the logging in user
            var creator = new NewGroupMember(me.Region, me.User, GetCurrentIpAddress().ToString());

            // Crate the server titled with the channel's display name
            var server = Group.CreateSyncedCommunityServer(creator, externalAccount.DisplayName, new [] { communitySync }, request.TextChannelTitle, request.VoiceChannelTitle,
                request.OwnerRoleName, request.GuestRoleName, null, request.IsPublic, url);

            FriendsStatsManager.Current.GroupsCreatedByType.Track((int)GroupType.Large);
            var contract = server.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, Token.UserID);

            FilterLogger.Log(me.User, null, "User created a new synced server", new { request, server = contract });

            return Ok(contract);
        }

        [HttpPost]
        [Route("stream-community")]
        [ResponseType(typeof(CreateStreamerCommunityGroupResponse))]
        [SocialBanFilter]
        public IHttpActionResult CreateStreamCommunity(CreateStreamerCommunityGroupRequest request)
        {
            request.Validate();

            var me = GetCurrentUserAndRegion();

            // Ensure requestor owns communities
            var authorizedAccounts = ExternalAccountMapping.MultiGetLocal(request.SyncedStreams.Select(s => new KeyInfo(me.User.UserID, s.Type, s.ExternalAccountID))).Where(a => !a.IsDeleted).ToArray();
            var authorizedCommunities = ExternalCommunity.MultiGetLocal(authorizedAccounts.Select(a => new KeyInfo(a.ExternalID, a.Type))).ToDictionary(a => a.ExternalID);
            var missingCommunityAccounts = ExternalAccount.MultiGetLocal(authorizedAccounts.Where(a => !authorizedCommunities.ContainsKey(a.ExternalID)).Select(a => new KeyInfo(a.ExternalID, a.Type)));
            foreach (var acct in missingCommunityAccounts)
            {
                Logger.Debug("Missing community when creating a stream community: " + acct.ExternalID);
                authorizedCommunities[acct.ExternalID] = TwitchModelHelper.CreateOrUpdateCommunity(acct.ExternalID, acct, ExternalAccount.LocalConfigID);
            }
            if (authorizedCommunities.Count == 0)
            {
                // If none are authorized, fail
                return Forbidden();
            }

            var syncs = authorizedCommunities.Values
                .Select(c => new NewCommunitySync(c, request.SyncedStreams.First(s => s.ExternalAccountID == c.ExternalID && s.Type == c.Type).PremiumChannelName))
                .ToArray();

            // Look up the synced streams, and ensure that they are owned by the requesting user
            if (syncs.Any(s => s.Community.CanHaveSubs && s.PremiumChannelName != null && !s.PremiumChannelName.SafeRange(1, Group.TitleMaxLength)))
            {
                return BadRequest("Invalid sub only channel name.");
            }

            if (syncs.Any(s => !s.Community.CanHaveSubs && !string.IsNullOrEmpty(s.PremiumChannelName)))
            {
                return BadRequest("Unpartnered channels cannot have a sub only channel.");
            }

            if (!string.IsNullOrWhiteSpace(request.CustomUrl))
            {
                if (!authorizedCommunities.Values.Any(p => p.CanHaveSubs || p.Followers >= FriendsServiceConfiguration.Instance.VanityUrlFollowerRequirement))
                {
                    return BadRequest("You do not qualify for a custom url.");
                }

                if (VanityUrl.Exists(request.CustomUrl))
                {
                    return BadRequest("This URL is already in use.");
                }
            }

            var creator = new NewGroupMember(me.Region, me.User, GetCurrentIpAddress().ToString());
            var server = Group.CreateSyncedCommunityServer(creator, request.ServerName, syncs, request.DefaultTextChannelName, request.DefaultVoiceChannelName,
                request.OwnerRoleName, request.GuestRoleName, null, request.IsPublic, request.CustomUrl);

            var synced = new List<SyncResults>();
            foreach (var streamInfo in request.SyncedStreams)
            {
                var authorizedCommunity = authorizedCommunities.Values.FirstOrDefault(c => c.ExternalID == streamInfo.ExternalAccountID && c.Type == streamInfo.Type);
                synced.Add(authorizedCommunity == null
                    ? new SyncResults { StreamID = streamInfo.ExternalAccountID, Type = streamInfo.Type, Succeeded = false }
                    : new SyncResults
                    {
                        StreamID = streamInfo.ExternalAccountID,
                        Type = streamInfo.Type,
                        Succeeded = true,
                        Followers = authorizedCommunity.Followers,
                        Subscribers = authorizedCommunity.Subscribers,
                    });
            }

            FriendsStatsManager.Current.GroupsCreatedByType.Track((int)GroupType.Large);

            return Ok(new CreateStreamerCommunityGroupResponse
            {
                Group = server.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, me.User.UserID),
                SyncedStreams = synced.ToArray()
            });
        }

        [HttpPost]
        [Route("guild-community")]
        [ResponseType(typeof(GroupNotification))]
        [SocialBanFilter]
        public IHttpActionResult CreateGuildCommunity(CreateGuildCommunityRequest request)
        {
            request.Validate();

            // See if this user is the owner of too many groups, or too recent of a group:            
            var me = GetCurrentUserAndRegion();

//#if CONFIG_RELEASE

//            var memberships = GroupMember.GetAllByUserID(Token.UserID);

//            if (memberships.Count(p => p.IsRootGroup && p.BestRoleRank == 1 && !p.IsDeleted) >= Group.MaxServersOwned) // TODO: Figure out a better way to throttle this
//            {
//                return BadRequest("You have too many servers.");
//            }
//#endif

            var creator = new NewGroupMember(me.Region, me.User, GetCurrentIpAddress().ToString());
            var ownerRole = new NewGuildRole
            {
                Name = request.OwnerRole.Name,
                VanityColor = request.OwnerRole.Color
            };
            var defaultRole = new NewGuildRole
            {
                Name = request.GuestRole.Name,
                VanityColor = request.GuestRole.Color
            };

            var additionalRoles = request.AdditionalRoles == null
                ? new NewGuildRole[0]
                : request.AdditionalRoles.Select(r => new NewGuildRole { Name = r.Name, VanityColor = r.Color, IsOfficer = r.IsOfficer }).ToArray();

            var guildInfo = new NewGuildInfo
            {
                Title = request.Title,
                IsSearchable = true,
                IsPublic = request.IsPublic,
                GameIDs = request.Games ?? new int[0],
                Subtype = GroupSubType.Guild,

                OwnerRole = request.OwnerRole.ToNewGuildRole(),
                DefaultRole = request.GuestRole.ToNewGuildRole(),
                OtherRoles = request.AdditionalRoles == null?new NewGuildRole[0] : request.AdditionalRoles.Select(r=>r.ToNewGuildRole()).ToArray(),
                
                DefaultTextChannelName = request.GeneralChatName,
                OtherChannels = new[]
                {
                    new NewChannelInfo
                    {
                        ChannelMode = GroupMode.TextAndVoice,
                        ChannelName = request.OfficerChatName,
                        RestrictToOfficers = true
                    },
                    new NewChannelInfo
                    {
                        ChannelMode = GroupMode.TextAndVoice,
                        ChannelName = request.PveChatName
                    },
                    new NewChannelInfo
                    {
                        ChannelMode = GroupMode.TextAndVoice,
                        ChannelName = request.PvpChatName
                    }, 
                }
            };
            var newGroup = Group.CreateGuildServer(creator, guildInfo);
            FriendsStatsManager.Current.GroupsCreatedByType.Track((int)GroupType.Large);

            return Ok(newGroup.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, Token.UserID));
        }

        [HttpPost]
        [Route("synced-guild")]
        [ResponseType(typeof(CreateSyncedGuildServerResponse))]
        [SocialBanFilter]
        public IHttpActionResult CreateSyncedGuildServer(CreateSyncedGuildServerRequest request)
        {
            request.Validate();

            var me = GetCurrentUserAndRegion();

            // Ensure requestor owns communities
            var accounts = ExternalAccountMapping.GetAllLocal(a => a.UserID, me.User.UserID).Where(a => !a.IsDeleted);
            var characters = accounts.SelectMany(a => ExternalGuildMember.GetAllLocal(m => m.AccountID, a.ExternalID)).ToLookup(m => m.GuildIndex);

            var authorizedSyncs = new List<NewGuildSync>();
            var results = new List<SyncedGuildResult>();
            foreach (var syncedGuild in request.Guilds)
            {
                var syncedRoles = syncedGuild.RolesToCreate.ToDictionary(r => r.Tag);
                var result = new SyncedGuildResult
                {
                    Success = false,
                    Type = syncedGuild.Type,
                    GuildName = syncedGuild.Name,
                    GameServer = syncedGuild.GameServer,
                    GameRegion = syncedGuild.GameRegion,
                };

                var guild = ExternalGuild.Get(new ExternalGuildIdentifier(syncedGuild.Type, syncedGuild.GameRegion, syncedGuild.GameServer, syncedGuild.Name));
                if (guild != null && characters.Contains(guild.GetGuildIndex()))
                {
                    result.Success = true;
                    var rolesToCreate = new List<NewGuildSyncRole>();
                    foreach (var roleTag in ExternalGuildRole.GetRoleTags(guild.Type))
                    {
                        var syncedRole = syncedRoles.GetValueOrDefault(roleTag);
                        var newRole = new NewGuildSyncRole {Tag = roleTag};
                        if (syncedRole != null)
                        {
                            newRole.Name = syncedRole.Name;
                            newRole.IsOfficer = syncedRole.IsOfficer;
                            newRole.HasBadge = syncedRole.HasBadge;
                            newRole.VanityBadge = syncedRole.VanityBadge;
                            newRole.VanityColor = syncedRole.Color;
                        }
                        rolesToCreate.Add(newRole);
                    }
                    authorizedSyncs.Add(new NewGuildSync
                    {
                        Type = guild.Type,
                        Guild = guild,
                        RolesToCreate = rolesToCreate.ToArray(),
                        GameID = ExternalGuild.GetGameIDFromType(guild.Type)
                    });
                }
                results.Add(result);
            }

            if (authorizedSyncs.Count == 0)
            {
                // If none are authorized, fail
                return Forbidden(new CreateSyncedGuildServerResponse
                {
                    Results = results.ToArray()
                });
            }

            var guildInfo = new NewGuildInfo
            {
                Title = request.Title,
                IsPublic = false,
                IsSearchable = false,
                GameIDs = request.Games,
                Subtype = request.Subtype,

                OwnerRole = new NewGuildRole
                {
                    Name = request.OwnerRole.Name,
                    VanityColor = request.OwnerRole.Color
                },
                DefaultRole = new NewGuildRole
                {
                    Name = request.GuestRole.Name,
                    VanityColor = request.GuestRole.Color
                },
                OtherRoles = request.AdditionalRoles == null
                    ? new NewGuildRole[0]
                    : request.AdditionalRoles.Select(r => new NewGuildRole {Name = r.Name, VanityColor = r.Color, IsOfficer = r.IsOfficer}).ToArray(),

                DefaultTextChannelName = request.GeneralChatName,
                OtherChannels = new[]
                {
                    new NewChannelInfo
                    {
                        ChannelMode = GroupMode.TextAndVoice,
                        ChannelName = request.OfficerChatName,
                        RestrictToOfficers = true,
                    },
                    new NewChannelInfo
                    {
                        ChannelMode = GroupMode.TextAndVoice,
                        ChannelName = request.DefaultVoiceChannelName,
                    },
                    new NewChannelInfo
                    {
                        ChannelMode = GroupMode.TextAndVoice,
                        ChannelName = request.PveChatName,
                    },
                    new NewChannelInfo
                    {
                        ChannelMode = GroupMode.TextAndVoice,
                        ChannelName = request.PvpChatName,
                    }
                },

                GuildSyncs = authorizedSyncs.ToArray()
            };
            var creator = new NewGroupMember(me.Region, me.User, GetCurrentIpAddress().ToString());

            var newGroupID = Guid.NewGuid();
            if (!string.IsNullOrEmpty(request.AvatarUrl))
            {
                var resp = ServiceClients.FriendsServiceClients.Instance.AvatarAdmin.ChangeAvatarUrl(Curse.ServiceClients.Contracts.AvatarType.Group, newGroupID.ToString(), request.AvatarUrl);
                if (!resp.Success)
                {
                    Logger.Info("Error while setting an avatar for a synced guild server: " + resp.StatusCode, new {request.AvatarUrl, GroupID = newGroupID});
                }
            }

            Group defaultChannel;
            var server = Group.CreateGuildServer(creator, guildInfo, out defaultChannel, newGroupID);
            var category = GroupInvitationReadableWordCategory.None;
            if (authorizedSyncs.Any(s => s.Type == AccountType.WorldOfWarcraft))
            {
                category = GroupInvitationReadableWordCategory.WorldOfWarcraft;
            }
            var invite = server.CreateGroupInvitation(creator.UserID, false, defaultChannel.GroupID, TimeSpan.Zero, 0, null);
            FriendsStatsManager.Current.GroupsCreatedByType.Track((int)GroupType.Large);

            return Ok(new CreateSyncedGuildServerResponse
            {
                Group = server.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, me.User.UserID),
                Results = results.ToArray(),
                Invitation = invite.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, server, defaultChannel, GroupsWebConfiguration.Instance.GroupInviteUrlFormat, false)
            });
        }

        [HttpGet]
        [Route("url/{url}")]
        [ResponseType(typeof(string))]
        public IHttpActionResult GetFromUrl(string url)
        {
            if (!VanityUrlValidator.IsValid(url))
            {
                return NotFound();
            }

            var groupID = VanityUrl.GetGroupIDFromUrl(url);

            if (groupID == null)
            {
                return NotFound();
            }

            return Ok(groupID);
        }

        [HttpGet]
        [Route("url/{url}/validate")]
        [ResponseType(typeof(ValidateUrlStatus))]
        public IHttpActionResult ValidateUrl(string url)
        {
            if (!VanityUrlValidator.IsValid(url))
            {
                return Ok(ValidateUrlStatus.Invalid);
            }

            if (VanityUrl.Exists(url))
            {
                return Ok(ValidateUrlStatus.Taken);
            }

            return Ok(ValidateUrlStatus.Available);
        }

        /// <summary>
        /// Gets the current general settings for the server
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("{groupID}/settings/general")]
        [ResponseType(typeof(ServerGeneralSettingsContract))]
        [HttpGet]
        public IHttpActionResult GetGeneralSettings(Guid groupID)
        {

            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (!group.IsRootGroup)
            {
                return BadRequest("Only the root group can have settings.");
            }

            group.CheckPermission(GroupPermissions.ManageServer, Token.UserID);

            var settings = new ServerGeneralSettingsContract
            {
                Title = group.Title,
                AfkChannel = group.AfkChannelID != Guid.Empty ? (Guid?) group.AfkChannelID : null,
                AfkTimeMinutes = group.AfkTimerMinutes,
                IsPublic = group.IsPublic,
                VoiceRegion = (VoiceRegion) group.VoiceRegionID,
                ChatThrottleSeconds = group.ChatThrottleSeconds,
                ChatThrottleEnabled = group.ChatThrottleEnabled
            };
            
            return Ok(settings);
        }

        /// <summary>
        /// Modify the settings for server
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("{groupID}/settings/general")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult SetGeneralSettings(Guid groupID, ServerGeneralSettingsContract request)
        {
            request.Validate();

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (request.AfkChannel.HasValue && request.AfkChannel != Guid.Empty)
            {
                var afkGroup = Group.GetByID(request.AfkChannel.Value);
                if (afkGroup == null || afkGroup.Mode != GroupMode.TextAndVoice)
                {
                    return BadRequest("Unknown or invalid AFK group. You must supply an active group that allows voice.");
                }
            }

            group.ChangeServerSettings(Token.UserID, request.Title, request.AfkChannel ?? Guid.Empty, request.AfkTimeMinutes, (int) request.VoiceRegion, request.IsPublic, request.ChatThrottleSeconds, request.ChatThrottleEnabled);
            return Ok();
        }

        /// <summary>
        /// Gets the current search settings for the server
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("{groupID}/settings/search")]
        [ResponseType(typeof(ServerSearchSettingsContract))]
        [HttpGet]
        public IHttpActionResult GetSearchSettings(Guid groupID)
        {

            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (!group.CanHaveSearchSettings)
            {
                return BadRequest();
            }

            group.CheckPermission(GroupPermissions.ManageServer, Token.UserID);
            var searchSettings = group.GetServerSearchSettingsOrDefault();

            var settings = new ServerSearchSettingsContract
            {
                Description = searchSettings.Description,
                Games = searchSettings.Games.ToArray(),
                IsSearchable = searchSettings.IsSearchable,
                MatchAllGames = searchSettings.MatchAllGames,
                SearchTags = searchSettings.SearchTags.Select(i=>(GroupSearchTag)i).ToArray()
            };

            return Ok(settings);
        }

        /// <summary>
        /// Modify the settings for server
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("{groupID}/settings/search")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult SetSearchSettings(Guid groupID, ServerSearchSettingsContract request)
        {
            request.Validate();

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (!group.CanHaveSearchSettings)
            {
                return BadRequest();
            }

            group.ChangeServerSearchSettings(Token.UserID, request.IsSearchable, request.Description, new HashSet<int>(request.SearchTags.Cast<int>()), new HashSet<int>(request.Games),
                request.MatchAllGames);
            return Ok();
        }

        /// <summary>
        /// Creates a channel within a server
        /// </summary>
        [Route("{groupID}/channels")]
        [HttpPost]
        [ResponseType(typeof(ChannelContract))]
        [SocialBanFilter]
        public IHttpActionResult CreateChannel(Guid groupID, CreateChannelRequest request)
        {
            request.Validate();
            var group = Group.GetWritableByID(groupID);

            if (group == null)
            {
                return NotFound();
            }
            
            var me = GetCurrentUserAndRegion();
            var creator = new NewGroupMember(me.Region, me.User, null);
            var category = request.DisplayCategory == null
                ? null
                : new GroupCategoryInfo
                {
                    DisplayRank = request.DisplayCategory.Rank,
                    Name = request.DisplayCategory.Name,
                    ID = request.DisplayCategory.ID
                };
            var channel = group.CreateChannel(creator, GroupType.Large, request.Mode, request.IsPublic, new HashSet<int>(request.AccessRoles), request.Title, category, request.HideNoAccess,
                request.HideCallMembersNoAccess);
            var creatorMembership = channel.GetMember(Token.UserID);
            var notification = channel.ToChannelNotification(creatorMembership, null);
            return Ok(notification);
        }

        /// <summary>
        /// Deletes a channel
        /// </summary>
        [Route("{groupID}/channels/{channelID}")]
        [HttpDelete]
        [SocialBanFilter]
        public IHttpActionResult DeleteChannel(Guid groupID, Guid channelID)
        {
            var server = Group.GetWritableByID(groupID);
            if (server == null)
            {
                return NotFound();
            }

            // Get the child group
            var channel = server.GetChildGroup(channelID);

            if (channel == null)
            {
                return NotFound();
            }

            // Delete it
            server.DeleteChannel(Token.UserID, channel);

            return Ok();
        }

        /// <summary>
        /// Changes the settings for a specific channel
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="channelID"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("{groupID}/channels/{channelID}/settings")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult ChannelSettings(Guid groupID, Guid channelID, ModifyChannelSettingsRequest request)
        {
            request.Validate();

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var channel = group.GetChildGroup(channelID);

            if (channel == null)
            {
                return NotFound();
            }

            channel.ChangeChannelSettings(Token.UserID, request.Title, request.MessageOfTheDay, request.IsPublic, new HashSet<int>(request.AccessRoles), request.AllowTemporaryChildGroups, request.ForcePushToTalk, request.HideNoAccess, request.HideCallMembersNoAccess);
            return Ok();

        }
        
        /// <summary>
        /// Deletes a group (can be restored)
        /// </summary>
        /// <param name="groupID"></param>
        /// <returns></returns>
        [Route("{groupID}")]
        [HttpDelete]
        [SocialBanFilter]
        public IHttpActionResult Delete(Guid groupID)
        {
            var group = Group.GetWritableByID(groupID);
            if (group == null || group.IsDeleted)
            {
                return NotFound();
            }

            // If root group, delete whole group including channels
            if (!group.IsRootGroup)
            {
                return BadRequest("This API can only be used to delete root level groups.");
            }

            group.DeleteRoot(Token.UserID);
            return Ok();

        }

        /// <summary>
        /// Restores the channel that is soft deleted
        /// </summary>
        [Route("{groupID}/restore")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult Restore(Guid groupID)
        {

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (group.IsRootGroup)
            {
                //cannot restore a deleted channel
                return BadRequest("You cannot restore a root group, yet.");
            }
            else
            {
                group.RestoreChannel(Token.UserID);
            }

            return Ok();
        }


        [Route("{groupID}/reorder")]
        [HttpPost]
        [ResponseType(typeof(GroupNotification))]
        [SocialBanFilter]
        public IHttpActionResult Reorder(Guid groupID, ReorganizeGroupRequest request)
        {

            request.Validate();

            var root = Group.GetWritableByID(groupID);
            if (root == null)
            {
                return NotFound();
            }

            var categories = new List<GroupCategoryInfo>();
            if (request.DisplayCategories != null)
            {
                categories.AddRange(request.DisplayCategories.Select(c => new GroupCategoryInfo
                {
                    ID = c.ID,
                    Name = c.Name ?? string.Empty,
                    DisplayRank = c.Rank,
                    Groups = new GroupReorderInfo[0]
                }));
            }

            if (request.ChangedGroups != null)
            {
                foreach (var grouping in request.ChangedGroups.GroupBy(g => g.CategoryID))
                {
                    var category = categories.FirstOrDefault(c => c.ID == grouping.Key);
                    if (category == null)
                    {
                        category = new GroupCategoryInfo
                        {
                            ID = Guid.NewGuid(),
                            Name = string.Empty,
                            DisplayRank = categories.Count + 1,
                        };
                        categories.Add(category);
                    }

                    category.Groups = grouping.Select(g => new GroupReorderInfo
                    {
                        DisplayOrder = g.DisplayOrder,
                        GroupID = g.GroupID,
                        ParentID = g.ParentID
                    }).ToArray();
                }
            }
            root.ReorganizeChildren(categories.ToArray(), Token.UserID);

            return Ok(root.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, Token.UserID));

        }

        /// <summary>
        /// Joins a server if it is not private.
        /// </summary>
        /// <param name="groupID">The ID of the server.</param>
        [Route("{groupID}/join")]
        [HttpPost]
        [ResponseType(typeof(GroupNotification))]
        [SocialBanFilter]
        public IHttpActionResult JoinServer(Guid groupID)
        {
            var root = Group.GetByID(groupID);
            if (root == null)
            {
                return NotFound();
            }

            var me = GetCurrentUserAndRegion();
            root.JoinPublicServer(new NewGroupMember(me.Region, me.User, GetCurrentIpAddress().ToString(), root.DefaultRole));
            return Ok(root.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, me.User.UserID));
        }

        [Route("partners")]
        [HttpGet]
        [SocialBanFilter]
        [ResponseType(typeof(PartnerServerInfo))]
        public IHttpActionResult GetPartnerServerInfo()
        {
            var me = GetCurrentUserAndRegion();
            if (!me.User.IsMerged)
            {
                return ErrorResponse(HttpStatusCode.Forbidden, "user_not_merged");
            }

            var account = ExternalAccount.GetByTwitchUserID(me.User.TwitchID);
            if(account == null || !account.IsPartnered)
            {
                return ErrorResponse(HttpStatusCode.Forbidden, "user_not_partner");
            }

            var server = Group.GetLocal(GroupsWebConfiguration.Instance.PartnerServerID);
            if (server == null)
            {
                return ErrorResponse(HttpStatusCode.InternalServerError, "server_not_available");
            }

            return Ok(PartnerServerInfo.FromModel(server));
        }

        [Route("partners")]
        [HttpPost]
        [SocialBanFilter]
        [ResponseType(typeof(JoinPartnerServerResponse))]
        public IHttpActionResult JoinPartnerServer()
        {
            var me = GetCurrentUserAndRegion();
            if (!me.User.IsMerged)
            {
                return ErrorResponse(HttpStatusCode.Forbidden, "user_not_merged");
            }

            var account = ExternalAccount.GetByTwitchUserID(me.User.TwitchID);
            if(account == null || !account.IsPartnered)
            {
                return ErrorResponse(HttpStatusCode.Forbidden, "user_not_partner");
            }

            var server = Group.GetWritable(GroupsWebConfiguration.Instance.PartnerServerID);
            if (server == null)
            {
                return ErrorResponse(HttpStatusCode.InternalServerError, "server_not_available");
            }

            try
            {
                var newMember = new NewGroupMember(me.Region, me.User, GetCurrentIpAddress().ToString(), server.DefaultRole);
                server.SystemJoinServer(newMember);
                return Ok(new JoinPartnerServerResponse
                {
                    Server = server.ToServerNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, newMember.UserID),
                });
            }
            catch(GroupPermissionException ex)
            {
                return ErrorResponse(HttpStatusCode.Forbidden, "user_banned", ex.Message);
            }
            catch(DataValidationException ex)
            {
                return ErrorResponse(HttpStatusCode.InternalServerError, "server_not_available");
            }
        }


        #region Emoticons

        [Route("{groupID}/emoticons")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult AddEmoticon(Guid groupID, AddEmoticonRequest request)
        {

            request.Validate();

            var group = Group.GetByID(groupID);

            if (group == null)
            {
                return NotFound();
            }


            group.AddEmoticon(Token.UserID, request.Regex, request.Url, EmoticonSource.Curse, request.RequiredRoles ?? new HashSet<int>());
            return Ok();
        }

        [Route("{groupID}/emoticons/{id}")]
        [HttpDelete]
        [SocialBanFilter]
        public IHttpActionResult RemoveEmoticon(Guid groupID, string id)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var emoticon = GroupEmoticon.GetLocal(groupID, id, EmoticonSource.Curse);
            if (emoticon == null || emoticon.Status != EmoticonStatus.Active)
            {
                return NotFound();
            }

            group.RemoveEmoticon(Token.UserID, emoticon);
            return Ok();
        }

        #endregion

        #region Role Management

        [Route("{groupID}/roles")]
        [HttpGet]
        [ResponseType(typeof(GroupRoleDetails[]))]
        public IHttpActionResult GetRoles(Guid groupID)
        {

            var rootGroup = Group.GetByID(groupID);

            if (rootGroup == null)
            {
                return NotFound();
            }

            if (!rootGroup.IsRootGroup)
            {
                return BadRequest();
            }

            if (!rootGroup.HasPermission(GroupPermissions.ManageServer, Token.UserID)
                && !rootGroup.HasPermission(GroupPermissions.ManageChannels, Token.UserID))
            {
                return Forbidden();
            }

            var allRoles = rootGroup.GetRoles().OrderBy(p => p.Rank);
            var allRolePermissions = rootGroup.GetAllChildRolePermissions();
            var roleDetails = allRoles.Select(role => ToRoleDetails(rootGroup, role, allRolePermissions)).ToArray();
            
            return Ok(roleDetails);
        }


        [Route("{groupID}/roles/{roleID}")]
        [HttpGet]
        [ResponseType(typeof(GroupRoleDetails))]
        public IHttpActionResult GetRole(Guid groupID, int roleID)
        {
            var rootGroup = Group.GetByID(groupID);

            if (rootGroup == null)
            {
                return NotFound();
            }

            rootGroup.CheckPermission(GroupPermissions.ManageServer, Token.UserID);

            if (!rootGroup.IsRootGroup)
            {
                return BadRequest();
            }

            var role = rootGroup.GetRole(roleID);
            if (role == null)
            {
                return NotFound();
            }

            return Ok(ToRoleDetails(rootGroup, role));
        }

        private GroupRoleDetails ToRoleDetails(Group rootGroup, GroupRole role, GroupRolePermissions[] allRolePermissions = null)
        {
            var details = new GroupRoleDetails {Role = role.ToNotification()};
            details.Permissions = new Dictionary<Guid, Dictionary<GroupPermissions, GroupPermissionState>>();
            details.EffectivePermissions = new Dictionary<Guid, long>();
            
            var rolePermissions = allRolePermissions == null ? rootGroup.GetAllPermissionsByRoleID(role.RoleID) : allRolePermissions.Where(p => p.RoleID == role.RoleID).ToArray();
            
            foreach (var permission in rolePermissions)
            {
                details.EffectivePermissions[permission.GroupID] = (long)permission.Permissions;

                if (!details.Permissions.ContainsKey(permission.GroupID))
                {
                    details.Permissions.Add(permission.GroupID, new Dictionary<GroupPermissions, GroupPermissionState>());
                }

                foreach (var permissionsState in permission.PermissionsState)
                {
                    details.Permissions[permission.GroupID].Add((GroupPermissions) permissionsState.Key, (GroupPermissionState) permissionsState.Value);
                }

            }

            return details;
        }

        [Route("{groupID}/roles")]
        [HttpPost]
        [ResponseType(typeof(GroupRoleDetails))]
        [SocialBanFilter]
        public IHttpActionResult CreateRole(Guid groupID, CreateGroupRoleRequest request)
        {

            request.Validate();

            var rootGroup = Group.GetWritableByID(groupID);

            if (rootGroup == null)
            {
                return NotFound();
            }

            if (!rootGroup.IsRootGroup)
            {
                return BadRequest();
            }

            rootGroup.CheckPermission(GroupPermissions.ManageServer, Token.UserID);

            var existingRoles = rootGroup.GetRoles();

            if (existingRoles.Any(p => !p.IsSynced && p.Name.Equals(request.Name, StringComparison.InvariantCultureIgnoreCase) && !p.IsDeleted))
            {
                return BadRequest("This server already has a role named '" + request.Name + "'");
            }

            // Ensure the role has access
            if (!request.Permissions.HasFlag(GroupPermissions.Access))
            {
                request.Permissions = request.Permissions | GroupPermissions.Access;
            }

            var role = rootGroup.CreateRole(request.Name, request.Permissions, false, false, false, request.VanityColor, request.VanityBadge, request.HasCustomVanityBadge);

            GroupChangeCoordinator.ChangeInfo(rootGroup, Token.UserID);
            return Ok(ToRoleDetails(rootGroup, role));
        }

        [Route("{groupID}/roles/{roleID}")]
        [HttpPost]
        [ResponseType(typeof(GroupRoleDetails))]
        [SocialBanFilter]
        public IHttpActionResult ModifyRole(Guid groupID, int roleID, ModifyGroupRoleRequest request)
        {
            request.Validate();
            var rootGroup = Group.GetWritableByID(groupID);

            if (rootGroup == null)
            {
                return NotFound();
            }

            // Root level roles should always have access
            request.Permissions[GroupPermissions.Access] = GroupPermissionState.Allowed;                
            
            var role = rootGroup.ModifyRole(Token.UserID, roleID, request.Name, request.VanityColor, request.HasCustomVanityBadge, request.VanityBadge, request.Permissions);            
            GroupChangeCoordinator.ChangeInfo(rootGroup, Token.UserID);
            return Ok(ToRoleDetails(rootGroup, role));
        }

        [Route("{groupID}/roles/ranks")]
        [HttpPost]
        [ResponseType(typeof(GroupRoleDetails[]))]
        [SocialBanFilter]
        public IHttpActionResult ModifyRoleRanks(Guid groupID, [FromBody] Dictionary<int, int> roleRanks)
        {
            if (roleRanks == null)
            {
                return BadRequest("Invalid role ranks!");
            }

            var rootGroup = Group.GetWritableByID(groupID);

            if (rootGroup == null)
            {
                return NotFound();
            }
            
            var roles = rootGroup.ModifyRoleRanks(Token.UserID, roleRanks);
            return Ok(roles.Select(p => ToRoleDetails(rootGroup, p)));
        }

        [Route("{groupID}/roles/{roleID}")]
        [HttpDelete]
        [ResponseType(typeof(void))]
        [SocialBanFilter]
        public IHttpActionResult DeleteRole(Guid groupID, int roleID)
        {
            var rootGroup = Group.GetWritableByID(groupID);

            if (rootGroup == null)
            {
                return NotFound();
            }

            var role = rootGroup.GetRole(roleID);

            if (role == null)
            {
                return NotFound();
            }

            if (!role.CanBeDeleted)
            {
                return Forbidden("This role cannot be deleted.");
            }

            rootGroup.DeleteRole(Token.UserID, roleID);            
            GroupChangeCoordinator.ChangeInfo(rootGroup, Token.UserID);
            return Ok();
        }

        [Route("{groupID}/roles/{roleID}/permissions/{channelID}")]
        [HttpPost]
        [ResponseType(typeof(GroupRoleDetails))]
        [SocialBanFilter]
        public IHttpActionResult ModifyRolePermissions(Guid groupID, Guid channelID, int roleID, ModifyGroupRolePermissionsRequest request)
        {
            request.Validate();
            var rootGroup = Group.GetWritableByID(groupID);

            if (rootGroup == null)
            {
                return NotFound();
            }

            // Ensure that access is enabled if any permissions are specified
            if (request.PermissionsChanges.Any(p => p.Value == GroupPermissionState.Allowed || p.Value == GroupPermissionState.AllowedInherited))
            {
                request.PermissionsChanges[GroupPermissions.Access] = GroupPermissionState.Allowed;
            }

            var role = rootGroup.ModifyRolePermissions(Token.UserID, roleID, channelID, request.PermissionsChanges);
            GroupChangeCoordinator.ChangeInfo(rootGroup, Token.UserID);
            return Ok(ToRoleDetails(rootGroup, role));
        }

        /// <summary>
        /// Adds a role to the specified user
        /// </summary>        
        [Route("{groupID}/members/{userID}/roles")]
        [HttpPost]
        [ResponseType(typeof(GroupMemberContract))]
        [SocialBanFilter]
        public IHttpActionResult AddMemberRole(Guid groupID, int userID, [FromBody] int roleID)
        {

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var groupMember = group.GetMember(userID);
            if (groupMember == null)
            {
                return NotFound();
            }

            var groupRole = group.GetRole(roleID);

            if (groupRole == null)
            {
                return BadRequest("Unknown role!");
            }

            group.AddMemberRole(Token.UserID, groupRole, userID);
            groupMember.Refresh();
            return Ok(groupMember.ToNotification(UserStatistics.GetByUserOrDefault(userID)));
        }

        /// <summary>
        /// Removes a role to the specified user
        /// </summary>
        [Route("{groupID}/members/{userID}/roles/{roleID}")]
        [HttpDelete]
        [ResponseType(typeof(GroupMemberContract))]
        [SocialBanFilter]
        public IHttpActionResult RemoveMemberRole(Guid groupID, int userID, int roleID)
        {

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var groupMember = group.GetMember(userID);
            if (groupMember == null)
            {
                return NotFound();
            }

            var groupRole = group.GetRole(roleID);

            if (groupRole == null)
            {
                return NotFound();
            }

            group.RemoveMemberRole(Token.UserID, groupRole, userID);
            groupMember.Refresh();
            return Ok(groupMember.ToNotification(UserStatistics.GetByUserOrDefault(userID)));            
        }

        #endregion
        
        #region Invite Management

        /// <summary>
        /// Create an invitation that grants access to the group, for a finite period of time.
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [Route("{groupID}/invites")]
        [HttpPost]
        [ResponseType(typeof (GroupInvitationNotification))]
        [SocialBanFilter]
        public IHttpActionResult CreateInvite(Guid groupID, CreateGroupInviteRequest request)
        {
            request.Validate();

            var userRegion = GetCurrentUserAndRegion();

            if (userRegion == null)
            {
                return BadRequest("Unknown requesting user.");
            }

            // Get the group in question
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            FilterLogger.Log(userRegion.User, "Creating a server invite link...", new { group.Title, group.IsPublic, request });

            var invite = group.CreateGroupInvitation(Token.UserID, request.AutoRemoveMembers, request.ChannelID, TimeSpan.FromMinutes(request.LifespanMinutes), request.MaxUses,
                request.AdminDescription);

            return Ok(invite.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, group, Group.GetByID(invite.ChannelID), GroupsWebConfiguration.Instance.GroupInviteUrlFormat, true));
        }

        [Route("{groupID}/invites")]
        [HttpGet]
        [ResponseType(typeof(GroupInvitationNotification[]))]
        public IHttpActionResult GetInvites(Guid groupID, bool myInvitesOnly = false, bool activeInvitesOnly = false)
        {

            // Get the group in question
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var allInvites = group.GetGroupInvitations(Token.UserID, myInvitesOnly, activeInvitesOnly);
            var invites = allInvites.Select(p => p.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, group, Group.GetByID(p.ChannelID, true), GroupsWebConfiguration.Instance.GroupInviteUrlFormat, true)).ToArray();

            return Ok(invites);
        }

        [Route("{groupID}/invites/{inviteCode}")]
        [HttpDelete]
        public IHttpActionResult DeleteInvite(Guid groupID, string inviteCode, [FromUri] bool removeGuests = false)
        {
            var group = Group.GetByID(groupID);

            if (group == null)
            {
                return NotFound();
            }

            group.DeleteInvitation(Token.UserID, inviteCode, removeGuests);

            return Ok();

        }

        #endregion

        /// <summary>
        /// Gets banned users.
        /// </summary>
        /// <param name="serverID">The server's ID.</param>
        /// <param name="query">The optional query string.</param>
        /// <param name="pageSize">The optional page size.</param>
        /// <param name="pageNumber">The optional page number.</param>
        /// <returns></returns>
        [Route("{serverID}/bans")]
        [HttpGet]
        [ResponseType(typeof(IEnumerable<GroupBannedUserContract>))]
        public IHttpActionResult GetBanList(Guid serverID, string query = null, int? pageSize = null, int? pageNumber = null)
        {
            var group = Group.GetByID(serverID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.BanUser, Token.UserID);

            var bans = GroupBannedUserSearchManager.Search(new GroupBannedUserSearch
            {
                GroupID = serverID,
                PageNumber = pageNumber,
                PageSize = pageSize,
                Query = query,
            });

            return Ok(bans.Select(b => GroupBannedUserContract.ToNotification(b, group)));
        }

        /// <summary>
        /// Searches banned users.
        /// </summary>
        /// <param name="serverID">The server's ID.</param>
        /// <param name="request">The search request.</param>
        [Route("{serverID}/bans/search")]
        [HttpPost]
        [ResponseType(typeof(IEnumerable<GroupBannedUserContract>))]
        public IHttpActionResult SearchBannedUsers(Guid serverID, SearchBannedUsersRequest request)
        {
            request.Validate();

            var group = Group.GetByID(serverID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.BanUser, Token.UserID);

            var bans = GroupBannedUserSearchManager.Search(new GroupBannedUserSearch
            {
                GroupID = serverID,
                PageNumber = request.PageNumber,
                PageSize = request.PageSize,
                Query = request.Query,
                Username = request.Username,
                RequestorUsername = request.RequestorUsername,
                Reason = request.Reason,
                SortType = request.SortType ?? GroupBannedUserSortType.Username,
                SortAscending = request.SortAscending
            });

            return Ok(bans.Select(b => GroupBannedUserContract.ToNotification(b, group)));
        }

        /// <summary>
        /// Bans a user, kicking them from the group and preventing them from being added in the future.
        /// </summary>
        /// <param name="serverID">The server's ID.</param>
        /// <param name="request">Additional options for the ban.</param>
        /// <returns></returns>
        [Route("{serverID}/bans")]
        [HttpPost]
        [ResponseType(typeof(GroupBannedUserContract))]
        [SocialBanFilter]
        public IHttpActionResult BanUser(Guid serverID, [FromBody] BanUserRequest request)
        {
            var group = Group.GetWritableByID(serverID);
            if (group == null)
            {
                return NotFound();
            }

            DateTime? deleteMessagesStartDate = null;
            switch (request.MessageDeleteMode)
            {
                case BanUserMessageDeleteMode.LastDay:
                    deleteMessagesStartDate = DateTime.UtcNow.AddDays(-1);
                    break;
                case BanUserMessageDeleteMode.LastWeek:
                    deleteMessagesStartDate = DateTime.UtcNow.AddDays(-7);
                    break;
                case BanUserMessageDeleteMode.All:
                    deleteMessagesStartDate = DateTime.MinValue;
                    break;
            }

            var ip = request.BanIP
                ? ClientEndpoint.GetAllLocal(e => e.UserID, request.UserID).OrderByDescending(p => p.ConnectedDate).Select(p => p.IPAddress).FirstOrDefault()
                : string.Empty;
            var bannedUser = group.BanUser(Token.UserID, request.UserID, request.Reason, ip, deleteMessagesStartDate);
            var memberCall = group.GetCurrentCallGroup(request.UserID);
            if (memberCall.HasValue)
            {
                ServiceClients.FriendsServiceClients.Instance.CallsAdmin.KickUsersFromCall(new Curse.ServiceClients.Contracts.KickUsersFromCallRequest
                {
                    GroupID = memberCall.Value,
                    RequestorUserID = Token.UserID,
                    UserIDs = new[] { request.UserID },
                    Reason = Curse.ServiceClients.Contracts.KickReason.UserRemoved
                });
            }

            return Ok(GroupBannedUserContract.ToNotification(bannedUser, group));
        }

        /// <summary>
        /// Bans a user, kicking them from the group and preventing them from being added in the future.
        /// </summary>
        /// <param name="serverID">The server's ID.</param>
        /// <param name="userID">The user's ID.</param>
        /// <param name="reason">The reason for banning.</param>
        /// <returns></returns>
        [Route("{serverID}/bans/{userID}")]
        [HttpPost]
        [ResponseType(typeof(GroupBannedUserContract))]
        [SocialBanFilter]
        public IHttpActionResult BanUserOld(string serverID, int userID, [FromBody] string reason)
        {
            var group = Group.GetWritableByID(new Guid(serverID));
            if (group == null)
            {
                return NotFound();
            }

            var bannedUser = group.BanUser(Token.UserID, userID, reason, null, null);

            var memberCall = group.GetCurrentCallGroup(userID);
            if (memberCall.HasValue)
            {
                ServiceClients.FriendsServiceClients.Instance.CallsAdmin.KickUsersFromCall(new Curse.ServiceClients.Contracts.KickUsersFromCallRequest
                {
                    GroupID = memberCall.Value,
                    RequestorUserID = Token.UserID,
                    UserIDs = new[] { userID },
                    Reason = Curse.ServiceClients.Contracts.KickReason.UserRemoved
                });
            }
            return Ok(GroupBannedUserContract.ToNotification(bannedUser, group));
        }

        /// <summary>
        /// Unbans a user.
        /// </summary>
        /// <param name="serverID">The server's ID</param>
        /// <param name="userID">The user's ID</param>
        [Route("{serverID}/bans/{userID}")]
        [HttpDelete]
        [ResponseType(typeof (void))]
        [SocialBanFilter]
        public IHttpActionResult UnbanUser(Guid serverID, int userID)
        {
            var group = Group.GetByID(serverID);
            if (group == null)
            {
                return NotFound();
            }

            group.UnbanUser(Token.UserID, userID);

            return StatusCode(HttpStatusCode.NoContent);
        }

        [Route("{serverID}/live-streams")]
        [HttpGet]
        [ResponseType(typeof (IEnumerable<ExternalCommunityPublicContract>))]
        public IHttpActionResult GetLiveStreamDetails(Guid serverID)
        {
            var group = Group.GetByID(serverID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            if (!group.IsStreaming)
            {
                return Ok();
            }

            var mappings = ExternalCommunityMapping.GetAllLocal(c => c.GroupID, serverID).Where(c => !c.IsDeleted);
            var communities = ExternalCommunity.MultiGetLocal(mappings.Select(m => new KeyInfo(m.ExternalID, m.Type)));

            var notifications = communities.Where(c => c.IsLive).Select(c=>c.ToPublicContract()).ToArray();

            return Ok(notifications);
        }
    }
}