﻿using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Net;
using Amazon;
using Amazon.S3.Model;
using Amazon.S3.Transfer;
using Amazon.S3;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Logging;
using ImageResizer;
using ImageResizer.Configuration;
using ImageResizer.Plugins.AnimatedGifs;
using ImageResizer.Plugins.Basic;
using ImageResizer.Plugins.FastScaling;
using ImageResizer.Plugins.PrettyGifs;

namespace Curse.Friends.ImageManager
{
    public static class ImageManager
    {
        /// <summary>
        /// Save the image to the appropriate storage location
        /// </summary>
        public static Guid SaveImage(ImageMetadata metadata, Stream data, bool autoCloseStream = true)
        {
            var key = Guid.NewGuid();

            using (var client = GetClient())
            {
                using (var transfer = new TransferUtility(client))
                {
                    var request = new TransferUtilityUploadRequest
                    {
                        BucketName = ImageManagerConfiguration.Current.BucketName,
                        Key = key.ToString(),
                        InputStream = data,
                        AutoCloseStream = autoCloseStream,
                        ContentType = metadata.MimeType,
                    };
                    
                    request.Metadata["filename"] = metadata.Filename;
                    request.Metadata["mimetype"] = metadata.MimeType;
                    request.Metadata["dateuploaded"] = metadata.DateUploaded.ToString("o");
                    request.Metadata["isanimated"] = metadata.IsAnimated.ToString();
                    request.Metadata["height"] = metadata.Height.ToString();
                    request.Metadata["width"] = metadata.Width.ToString();
                    request.Metadata["totalsize"] = metadata.TotalBytes.ToString();

                    transfer.Upload(request);
                }
            }

            return key;
        }

        /// <summary>
        /// Return the url of an image, given the key and filename
        /// </summary>
        public static string GetImageUrl(string key, string filename)
        {
            return string.Format(ImageManagerConfiguration.Current.UrlFormat, key, filename);
        }

        public static string GetImageUrl(string key, string filename, int region)
        {
            var bucket = ImageManagerConfiguration.Global.Buckets.FirstOrDefault(b => b.RegionIdentifier == region);
            if (bucket == null)
            {
                return null;
            }

            return string.Format(bucket.UrlFormat, key, filename);
        }

        public static ImageMetadata GetImageMetadata(string key, int regionID)
        {
            var bucket = regionID == 0 ? ImageManagerConfiguration.Current : ImageManagerConfiguration.Global.Buckets.FirstOrDefault(p => p.RegionIdentifier == regionID);
            if (bucket == null)
            {
                return null;
            }

            using (var client = GetClient(bucket))
            {
                var request = new GetObjectMetadataRequest
                {
                    BucketName = ImageManagerConfiguration.Current.BucketName,
                    Key = key
                };

                GetObjectMetadataResponse response;

                try
                {
                    response = client.GetObjectMetadata(request);
                }
                catch (AmazonS3Exception ex)
                {
                    if (ex.Message != "The specified key does not exist.")
                    {
                        Logger.Warn(ex, "Failed to get image from S3");
                    }

                    return null;
                }

                if (response.HttpStatusCode != HttpStatusCode.OK)
                {
                    return null;
                }

                return CreateImageMetadata(response.Metadata);
            }
        }

        /// <summary>
        /// Return the bytes of the image
        /// </summary>
        public static Stream GetImageStream(string key, string requestedFilename, out ImageMetadata imageMetadata)
        {
            using (var client = GetClient())
            {
                var request = new GetObjectRequest
                {
                    BucketName = ImageManagerConfiguration.Current.BucketName,
                    Key = key
                };

                GetObjectResponse response;
                
                try
                {
                     response = client.GetObject(request);
                }
                catch (AmazonS3Exception ex)
                {
                    if (ex.Message != "The specified key does not exist.")
                    {
                        Logger.Warn(ex, "Failed to get image from S3");
                    }

                    imageMetadata = null;
                    return null;
                }
                
                if (response.HttpStatusCode != HttpStatusCode.OK)
                {
                    imageMetadata = null;
                    return null;
                }

                var filename = response.Metadata["filename"];
                if (filename != null && !filename.Equals(requestedFilename, StringComparison.InvariantCultureIgnoreCase))
                {
                    imageMetadata = null;
                    return null;
                }
                imageMetadata = CreateImageMetadata(response.Metadata);
                return response.ResponseStream;
            }
        }

        public static ImageMetadata GetImageMetadata(string key)
        {
            using (var client = GetClient())
            {
                var request = new GetObjectMetadataRequest
                {
                    BucketName = ImageManagerConfiguration.Current.BucketName,
                    Key = key
                };

                var response = client.GetObjectMetadata(request);

                if (response.HttpStatusCode != HttpStatusCode.OK)
                {
                    return null;
                }

                return CreateImageMetadata(response.Metadata);
            }
        }

        public static ImageMetadata CreateImageMetadata(MetadataCollection metadata)
        {
            return new ImageMetadata
            {
                MimeType = metadata["mimetype"],
                DateUploaded = DateTime.Parse(metadata["dateuploaded"]),
                IsAnimated = bool.Parse(metadata["isanimated"]),
                Height = int.Parse(metadata["height"]),
                Width = int.Parse(metadata["width"]),
                TotalBytes = int.Parse(metadata["totalsize"]),
                Filename = metadata["filename"]
            };
        }

        public static Stream ProcessImage(string key, string requestedFilename, out string contentType, int? width = null, int? height = null, bool? animate = null)
        {
            contentType = null;

            // Ensure that the width requested is valid
            if (width.HasValue && (width < ImageManagerConfiguration.Global.MinThumbnailWidth || width > ImageManagerConfiguration.Global.MaxThumbnailWidth))
            {
                return null;
            }

            // Ensure that the height requested is valid
            if (height.HasValue && (height < ImageManagerConfiguration.Global.MinThumbnailWidth || height > ImageManagerConfiguration.Global.MaxThumbnailWidth))
            {
                return null;
            }


            ImageMetadata metadata;
            // Get the stream
            var stream = GetImageStream(key, requestedFilename, out metadata);
            if (stream == null)
            {
                return null;
            }

            contentType = metadata.MimeType;
            if (metadata.IsAnimated)
            {
                if (!animate.HasValue)
                {
                    width = metadata.Width;
                    height = metadata.Height;
                    animate = false;
                }
                else // Do not attempt to resize animated GIFs
                {
                    return stream;
                }
                
            }
            
            if (!width.HasValue && !height.HasValue && !animate.HasValue)
            {
                // No processing should be performed
                return stream;
            }

            try
            {
                // Create an image object, and resample it                
                var outputStream = new MemoryStream();
                var r = new ResizeSettings
                {
                    MaxWidth = ImageManagerConfiguration.Global.MaxThumbnailWidth,
                    MaxHeight = ImageManagerConfiguration.Global.MaxThumbnailWidth
                };

                if (width.HasValue)
                {
                    r.Width = width.Value;
                }
                else if (height.HasValue)
                {
                    r.Height = height.Value;
                }
                else
                {
                    r.Width = r.Width;
                }

                r.Scale = ScaleMode.Both;
                r["fastscale"] = "true";
                r["autorotate"] = "true";

                if (!animate.HasValue || !animate.Value)
                {
                    r["frame"] = "0";
                }
                ImageBuilder.Current.Build(stream, outputStream, r);

                outputStream.Seek(0, SeekOrigin.Begin);
                return outputStream;
            }
            catch (SizeLimits.SizeLimitException ex)
            {
                throw new DataValidationException("Image was too large after resizing.", ex);
            }
        }

        public static void Initialize()
        {
            var config = ConfigurationHelper.LoadConfiguration<ImageManagerConfiguration>(FriendsServiceConfiguration.Mode, "ImageManager");
            
            var localConfig = config.Buckets.FirstOrDefault(p => p.RegionIdentifier == StorageConfiguration.CurrentRegion.ID);

            if (localConfig == null)
            {
                throw new Exception("Failed to initialize. No local config is defined!");
            }

            var limits = new SizeLimits
            {
                ImageSize = new Size(config.MaxThumbnailWidth, config.MaxThumbnailWidth),
                TotalSize = new Size(config.MaxImageDimension, config.MaxImageDimension),
                TotalBehavior = SizeLimits.TotalSizeBehavior.ThrowException
            };

            var sizeLimits = Config.Current.Plugins.Get<SizeLimiting>();
            if (sizeLimits == null)
            {
                new SizeLimiting {Limits = limits}.Install(Config.Current);
            }
            else
            {
                sizeLimits.Limits = limits;
            }
            new AnimatedGifs().Install(Config.Current);
            new PrettyGifs().Install(Config.Current);
            new FastScalingPlugin().Install(Config.Current);

            //new DiskCache().Install(Config.Current);
            ImageManagerConfiguration.Global = config;
            ImageManagerConfiguration.Current = localConfig;            
        }

        private static AmazonS3Client GetClient(ImageManagerBucketConfiguration bucket = null)
        {
            bucket = bucket ?? ImageManagerConfiguration.Current;

            var region = RegionEndpoint.GetBySystemName(bucket.RegionEndpoint);

            if (region == null)
            {
                throw new Exception("Failed to initialize. No region endpoint found:" + bucket.RegionEndpoint);
            }


            return new AmazonS3Client(bucket.AccessKeyID, bucket.SecretAccessKey, region);
        }

        public static ImageValidationStatus IsValidImage(Stream data, string filename, out ImageMetadata imageMetadata)
        {
            try
            {
             
                using (var img = Image.FromStream(data))
                {
                    data.Seek(0, SeekOrigin.Begin);

                    var type = img.ImageType();
                    imageMetadata = new ImageMetadata
                    {
                        Width = img.Width,
                        Height = img.Height,
                        MimeType = img.GetMimeType(),
                        IsAnimated = img.RawFormat.Equals(ImageFormat.Gif) && img.GetFrameCount(FrameDimension.Time) > 1,
                        Filename = Path.ChangeExtension(filename, "." + type),
                        DateUploaded = DateTime.UtcNow,
                        TotalBytes = (int) data.Length,
                        ImageType = type
                    };
                }

                data.Seek(0, SeekOrigin.Begin);

                if (imageMetadata.Width < ImageManagerConfiguration.Global.MinImageDimension)
                {
                    return ImageValidationStatus.WidthTooSmall;
                }
                if (imageMetadata.Width > ImageManagerConfiguration.Global.MaxImageDimension)
                {
                    return ImageValidationStatus.WidthTooLarge;
                }
                if (imageMetadata.Height < ImageManagerConfiguration.Global.MinImageDimension)
                {
                    return ImageValidationStatus.HeightTooSmall;
                }
                if (imageMetadata.Height > ImageManagerConfiguration.Global.MaxImageDimension)
                {
                    return ImageValidationStatus.HeightTooLarge;
                }
                if (!ImageManagerConfiguration.Global.AllowedImageTypes.Contains(imageMetadata.ImageType, StringComparer.InvariantCultureIgnoreCase))
                {
                    return ImageValidationStatus.UnsupportedFormat;
                }

                return ImageValidationStatus.Valid;
            }
            catch
            {
                imageMetadata = null;
                return ImageValidationStatus.UnexpectedError;
            }
        }

        public static bool DeleteImage(string key, int regionID = 0)
        {

            var bucket = regionID == 0 ? ImageManagerConfiguration.Current : ImageManagerConfiguration.Global.Buckets.FirstOrDefault(p => p.RegionIdentifier == regionID);
            if (bucket == null)
            {
                return false;
            }

            using (var client = GetClient(bucket))
            {
                var request = new DeleteObjectRequest
                {
                    BucketName = bucket.BucketName,
                    Key = key
                };

                var response = client.DeleteObject(request);

                return response.HttpStatusCode == HttpStatusCode.OK || response.HttpStatusCode == HttpStatusCode.NoContent; 
            }
        }
    }
}
