﻿using System;
using System.Net;
using System.Threading.Tasks;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.CloudFlare;
using Curse.Friends.ImageManager;
using Curse.Friends.ImagesWebService.Configuration;
using Curse.Friends.ImagesWebService.Contracts;
using Curse.Friends.MicroService;

namespace Curse.Friends.ImagesWebService.Controllers
{
    [RoutePrefix("images/api")]
    [AuthenticationFilter(AuthenticationLevel.ApiKey)]
#if !CONFIG_DEBUG && !CONFIG_STAGING
    [ApiExplorerSettings(IgnoreApi = true)]
#endif
    public class ImagesAdminController  : MicroServiceController
    {
        [Route("{region}/{imageKey}")]
        [HttpGet]
        [ResponseType(typeof(ImageMetadataResponse))]
        public IHttpActionResult FindImage(FileRegion region, string imageKey)
        {
            var metadata = ImageManager.ImageManager.GetImageMetadata(imageKey, (int) region);
            if (metadata == null)
            {
                return NotFound();
            }

            return Ok(ImageMetadataResponse.FromMetadata(metadata, ImageManager.ImageManager.GetImageUrl(imageKey, metadata.Filename, (int) region)));
        }

        [Route("{region}/{imageKey}")]
        [HttpDelete]
        public IHttpActionResult PurgeImage(FileRegion region, string imageKey)
        {
            var metadata = ImageManager.ImageManager.GetImageMetadata(imageKey, (int) region);
            if (metadata == null)
            {
                return NotFound();
            }

            if (!ImageManager.ImageManager.DeleteImage(imageKey, (int) region))
            {
                return NotFound();
            }

            if (ImageServiceConfiguration.Instance.CloudFlareEnabled)
            {
                // Clear CloudFlare
                Task.Delay(ImageCache.CacheDuration).ContinueWith(t =>
                {
                    CloudFlareApi.InvalidateUrl(ImageServiceConfiguration.Instance.CloudFlareDomain, ImageManager.ImageManager.GetImageUrl(imageKey, metadata.Filename, (int) region));
                });
            }
            return StatusCode(HttpStatusCode.Accepted);
        }
    }
}