﻿using System;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Text;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.ImageManager;
using Curse.Friends.MicroService;
using Curse.Friends.MicroService.Attributes;
using Curse.Friends.MicroService.Exceptions;
using Curse.Friends.MicroService.Extensions;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Statistics;
using Curse.Logging;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.ImagesWebService.Controllers
{
    [RoutePrefix("")]
    [ExcludeFromGenerator]
    public class ImagesController : MicroServiceUploadController
    {
        private static readonly LogCategory Logger = new LogCategory("AvatarConroller") { AlphaLevel = LogLevel.Trace };

        protected override string[] FileExtensionsFilter
        {
            get { return ImageManagerConfiguration.Global.AllowedImageTypes.ToArray(); }
        }

        /// <summary>
        ///     Upload an image, via a standard HTTP form file.
        /// </summary>
        [HttpPost]
        [Route("upload")]
        [SocialBanFilter]
        public AttachmentNotification Upload()
        {
            FriendsStatsManager.Current.ImagesUploaded.Track();            
            var attachment = UploadAttachment();
            return attachment.ToNotification();
        }

        [HttpGet]
        [HttpHead]
        [Route("")]
        [ApiExplorerSettings(IgnoreApi = true)]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult HomeRouteFix()
        {
            return ResponseMessage(new HttpResponseMessage(HttpStatusCode.OK)
            {
                Content = new StringContent("Hello from the " + MicroServiceConfiguration.ServiceType + "! You are accessing me on service node " + Environment.MachineName + " in " +
                                            FriendsServiceConfiguration.Instance.LastKnownRegion + ".", Encoding.UTF8, "text/plain")
            });
        }

        /// <summary>
        ///     Legacy upload endpoint (used by the mobile app)
        /// </summary>
        [HttpPost]
        [Route("")]
        [Obsolete]
        [ApiExplorerSettings(IgnoreApi = true)]
        [SocialBanFilter]
        public IHttpActionResult LegacyUpload()
        {
            FriendsStatsManager.Current.ImagesUploaded.Track();            
            var attachment = UploadAttachment();
            return Json(new { url = attachment.Url });
        }

        private Attachment UploadAttachment()
        {
            Logger.Trace("Uploading attachment...");

            var file = GetPostedFile();
            ImageMetadata imageMetadata;
            var validationStatus = ImageManager.ImageManager.IsValidImage(file.InputStream, file.FileName.RemoveSpecialCharacters(), out imageMetadata);

            Logger.Trace("Validated upload: " + validationStatus, imageMetadata);

            if(validationStatus!=ImageValidationStatus.Valid)
            {
                switch (validationStatus)
                {
                    case ImageValidationStatus.HeightTooLarge:
                    case ImageValidationStatus.WidthTooLarge:
                        throw new FileUploadException(FileUploadFailureReason.TooLarge);
                    case ImageValidationStatus.HeightTooSmall:
                    case ImageValidationStatus.WidthTooSmall:
                        throw new FileUploadException(FileUploadFailureReason.TooSmall);
                    case ImageValidationStatus.UnsupportedFormat:
                    case ImageValidationStatus.UnexpectedError:
                    default:
                        throw new FileUploadException(FileUploadFailureReason.UnsupportedFormat);
                }
            }

            var id = ImageManager.ImageManager.SaveImage(imageMetadata, file.InputStream);
            var url = ImageManager.ImageManager.GetImageUrl(id.ToString(), imageMetadata.Filename);

            // Put it in Aerospike
            var attachment = new Attachment
            {
                FileID = id,
                ConversationID = null,
                Status = AttachmentStatus.Available,
                Url = url,
                StorageRegionID = ImageManagerConfiguration.Current.RegionIdentifier,
                FileType = imageMetadata.MimeType,
                DateUploaded = imageMetadata.DateUploaded,
                Filename = imageMetadata.Filename,
                FileSize = imageMetadata.TotalBytes,
                UploaderUserID = Token.UserID,

                // Embed Only Properties
                IsEmbed = true,
                IsAnimated = imageMetadata.IsAnimated,
                Height = imageMetadata.Height,
                Width = imageMetadata.Width
            };

            attachment.InsertLocal();

            return attachment;
        }

        /// <summary>
        ///     Returns a response stream for the requested image, optionally sized and animated per the request args.
        /// </summary>
        [HttpGet]
        [Route("{id}/{filename}")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        [CacheControl(43200)]
        public IHttpActionResult Details(string id, string filename, int? width = null, int? height = null, bool? animate = null)
        {

            Logger.Trace("Getting image details: " + id + "/" + filename, new { id, filename, width, height, animate });

            var thumbnail = ImageCache.GetThumbnail(id, filename, width, height, animate);

            if (thumbnail == null)
            {
                return NotFound();
            }

            var response = new HttpResponseMessage
            {
                StatusCode = HttpStatusCode.OK,
                Content = new ByteArrayContent(thumbnail.ImageContent)
            };

            response.Content.Headers.ContentType = new MediaTypeHeaderValue(thumbnail.ContentType);

            return ResponseMessage(response);

        }

        /// <summary>
        ///     Returns a response stream for the requested image.
        /// </summary>
        [HttpGet]
        [Route("{id}/{filename}/download")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        [CacheControl(43200)]
        public IHttpActionResult Download(string id, string filename)
        {
            Logger.Trace("Downloading image: " + id + "/" + filename);

            ImageMetadata imageMetadata;
            var stream = ImageManager.ImageManager.GetImageStream(id, filename, out imageMetadata);

            if (stream == null)
            {
                return NotFound();
            }
            
            var response = new HttpResponseMessage
            {
                StatusCode = HttpStatusCode.OK,
                Content = new StreamContent(stream),
            };

            response.Content.Headers.ContentType = new MediaTypeHeaderValue(imageMetadata.MimeType);
            response.Content.Headers.ContentDisposition = new ContentDispositionHeaderValue("attachment")
            {
                FileName = imageMetadata.Filename
            };
            return ResponseMessage(response);
        }

        [HttpGet]
        [Route("metadata/{id}")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        [ApiExplorerSettings(IgnoreApi = true)]
        public IHttpActionResult Metadata(string id)
        {
            Logger.Trace("Getting image metadata: " + id);

            var metadata = ImageManager.ImageManager.GetImageMetadata(id);

            var json = new
            {
                type = metadata.MimeType,
                animated = metadata.IsAnimated,
                width = metadata.Width,
                height = metadata.Height,
                size = metadata.TotalBytes,
                datetime = metadata.DateUploaded.ToUniversalTime().ToString("o"),
                filename = metadata.Filename
            };


            return Json(json);


        }
    }
}