﻿using System;
using System.Drawing;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.AuthenticationClient;
using Curse.Friends.LoginsWebService.Contracts;
using Curse.Friends.MicroService;
using Curse.Logging;
using Curse.ServiceEncryption;
using Curse.Friends.AuthenticationClient.AuthService;
using Curse.Friends.Enums;
using Curse.Friends.LoginsWebService.Configuration;


namespace Curse.Friends.LoginsWebService.Authentication
{
    internal class LoginsAuthenticationProvider : AuthenticationProvider
    {

        internal class LoginResult
        {
            public LoginStatus Status { get; set; }
            public int UserID { get; set; }
            public string Username { get; set; }
            public string DisplayName { get; set; }
            public string EmailAddress { get; set; }
            public string SessionID { get; set; }
            public bool HasPremium { get; set; }
            public string Token { get; set; }
            public int SubscriptionToken { get; set; }
            public DateTime Expires { get; set; }
            public bool IsMerged { get; set; }
            public UserBanType Bans { get; set; }
            
            public static LoginResult Failure(LoginStatus status)
            {
                return new LoginResult
                {
                    Status = status
                };
            }

            public static LoginResult Success(string sessionID, User userProfile)
            {
                var emailAddress = userProfile.Profile.Emails.FirstOrDefault();
                if (emailAddress == null)
                {
                    throw new ArgumentNullException("userProfile has no email address");
                }

                EncryptionToken token = null;
                var now = DateTime.UtcNow;
                try
                {
                    token = EncryptionTokenHelper.Create(userProfile.ID, userProfile.Name, userProfile.Profile.Emails.First().Address, now);
                }
                catch (Exception ex)
                {
                    Logger.Error("Failed to create authentication token!", ex);
                }


                if (token == null)
                {
                    throw new ArgumentNullException("Token produced a null value");
                }

                var hasPremium = userProfile.Entitlements.Any(p => p.Active && p.Type == ESubscriptionType.Premium);
                
                return new LoginResult
                {
                    Status = LoginStatus.Success,
                    UserID = userProfile.ID,
                    Username = userProfile.Name,
                    SessionID = sessionID,
                    HasPremium = hasPremium,
                    EmailAddress = emailAddress.Address,
                    Token = token.Value,
                    SubscriptionToken = GetSubscriptionToken(userProfile.ID, hasPremium),
                    Expires = now + AuthenticationContext.TokenDuration,
                    IsMerged = userProfile.IsMapped
                };
            }

            public static LoginResult Success(int userID, string username, string displayName, bool hasPremium, bool isMergedAccount, UserBanType bans, string email)
            {
               
                EncryptionToken token = null;
                var now = DateTime.UtcNow;
                try
                {
                    token = EncryptionTokenHelper.Create(userID, username, string.Empty, now);
                }
                catch (Exception ex)
                {
                    Logger.Error("Failed to create authentication token!", ex);
                }
                
                if (token == null)
                {
                    throw new ArgumentNullException("Token produced a null value");
                }

              
                return new LoginResult
                {
                    Status = LoginStatus.Success,
                    UserID = userID,
                    Username = username,
                    SessionID = string.Empty,
                    HasPremium = hasPremium,
                    EmailAddress = email,
                    Token = token.Value,
                    SubscriptionToken = GetSubscriptionToken(userID, hasPremium),
                    Expires = now + AuthenticationContext.TokenDuration,
                    IsMerged = isMergedAccount,
                    DisplayName = displayName,
                    Bans = bans
                };
            }

            public LoginSession ToSession()
            {
                var hasPremiumPromo = LoginsWebServiceConfiguration.Current.PremiumPromoThrough >= DateTime.UtcNow.ToEpochMilliseconds();

                return new LoginSession
                {
                    UserID = UserID,
                    ActualPremiumStatus = HasPremium,
                    EffectivePremiumStatus = hasPremiumPromo || HasPremium,
                    EmailAddress = EmailAddress,
                    SessionID = SessionID,
                    Username = Username,
                    SubscriptionToken = SubscriptionToken,
                    Token = Token,
                    Expires = Expires.ToEpochMilliseconds(),
                    RenewAfter = (DateTime.UtcNow + RenewAfter).ToEpochMilliseconds(),
                    IsMerged = IsMerged,
                    DisplayName = DisplayName,
                    Bans = Bans
                };
            }
        }

        internal class ClaimAccountResult
        {
            public RegisterStatus Status { get; set; }

            public LoginResult LoginResult { get; set; }

            public static ClaimAccountResult Failure(RegisterStatus status)
            {
                return new ClaimAccountResult
                {
                    Status = status
                };
            }

            public static ClaimAccountResult Failure(LoginStatus status)
            {
                RegisterStatus registerStatus;
                switch (status)
                {
                    case LoginStatus.InvalidPassword:
                        registerStatus = RegisterStatus.InvalidPassword;
                        break;
                    default:
                        registerStatus = RegisterStatus.GeneralError;
                        break;
                }

                return new ClaimAccountResult
                {
                    Status = registerStatus
                };
            }

            public static ClaimAccountResult Success(string sessionID, User userProfile)
            {
                return new ClaimAccountResult
                {
                    Status = RegisterStatus.Success,
                    LoginResult = LoginResult.Success(sessionID, userProfile)
                };
            }
        }

        internal class RenewTokenResult
        {
            public EncryptionToken Token { get; private set; }
            public DateTime Expires { get; private set; }
            public DateTime Issued { get; private set; }

            internal RenewTokenResult(EncryptionToken token)
            {
                Token = token;
                Expires = token.GetExpiration(AuthenticationContext.TokenDuration);
                Issued = token.GetTimestamp();
            }

            public RenewTokenResponseContract ToContract()
            {
                return new RenewTokenResponseContract
                {
                    Token = Token.Value,
                    Expires = Expires.ToEpochMilliseconds(),
                    RenewAfter = (Issued + RenewAfter).ToEpochMilliseconds()
                };
            }
        }

        internal class RegisterResult
        {
            public RegisterStatus Status { get; set; }
            public LoginResult LoginResult { get; set; }

            private static RegisterStatus ToRegisterStatusStatus(EAddUserStatus status)
            {
                switch (status)
                {
                    case EAddUserStatus.Successful:
                        return RegisterStatus.Success;
                    case EAddUserStatus.InvalidPassword:
                        return RegisterStatus.InvalidPassword;
                    case EAddUserStatus.EmailInUse:
                        return RegisterStatus.EmailInUse;
                    case EAddUserStatus.InvalidUsername:
                    case EAddUserStatus.InvalidUsernameWithUnderscores:
                        return RegisterStatus.InvalidUsername;
                    case EAddUserStatus.InvalidProfile:
                        return RegisterStatus.InvalidProfile;
                    case EAddUserStatus.UsernameInUse:
                        return RegisterStatus.UsernameInUse;

                    default:
                        return RegisterStatus.GeneralError;
                }
            }

            public static RegisterResult Error()
            {
                return new RegisterResult
                {
                    Status = RegisterStatus.GeneralError
                };
            }

            public static RegisterResult Failure(EAddUserStatus status)
            {
                return new RegisterResult
                {
                    Status = ToRegisterStatusStatus(status)
                };
            }

            public static RegisterResult Success(string sessionID, User userProfile)
            {
                var loginResult = LoginResult.Success(sessionID, userProfile);

                return new RegisterResult
                {
                    Status = RegisterStatus.Success,
                    LoginResult = loginResult
                };
            }


        }
        
        private static LoginStatus ToAuthenticationStatus(ELoginStatus status)
        {
            switch (status)
            {
                case ELoginStatus.Success:
                    return LoginStatus.Success;
                case ELoginStatus.InvalidPassword:
                    return LoginStatus.InvalidPassword;
                case ELoginStatus.UnauthorizedLogin:
                    return LoginStatus.UnauthorizedLogin;
                case ELoginStatus.UnknownEmail:
                    return LoginStatus.UnknownEmail;
                case ELoginStatus.UnknownUsername:
                    return LoginStatus.UnknownUsername;
                default:
                    return LoginStatus.GeneralError;
            }
        }

        internal static LoginResult CreateLoginResult(int userID, string username, string displayName, bool hasPremium, bool isMergedAccount, UserBanType bans, string email)
        {
            return LoginResult.Success(userID, username, displayName, hasPremium, isMergedAccount, bans, email);
        }

        internal static LoginSession CreateLoginSession(int userID, string username, string displayName, bool hasPremium, bool isMergedAccount, UserBanType bans, string email)
        {
            return LoginResult.Success(userID, username, displayName, hasPremium, isMergedAccount, bans, email).ToSession();
        }


        internal static LoginResult LoginUser(string username, string plainTextPassword)
        {
            string encryptedPassword;

            // If the password cannot be encrypted, it's invalid!
            try
            {
                encryptedPassword = _outgoingCipher.Encrypt(plainTextPassword);
            }
            catch
            {
                return new LoginResult { Status = LoginStatus.GeneralError };
            }

            try
            {
                using (var authService = CreateServiceClient())
                {
                    var result = authService.v2ValidateClientUser(_siteID, username, encryptedPassword);
                    
                    if (result.Status != ELoginStatus.Success)
                    {
                        return LoginResult.Failure(ToAuthenticationStatus(result.Status));
                    }

                    var profile = authService.v2GetUserProfile(_siteID, result.UserId);                                        
                    return LoginResult.Success(result.SessionId, profile);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to make auth service call.");
                return LoginResult.Failure(LoginStatus.GeneralError);
            }
        }

        public static ClaimAccountResult ClaimAccount(int userID, string desiredEmail, string desiredPassword, string redirect)
        {
            try
            {
                string encryptedPassword;

                // If the password cannot be encrypted, it's invalid!
                try
                {
                    encryptedPassword = _outgoingCipher.Encrypt(desiredPassword);
                }
                catch
                {
                    return ClaimAccountResult.Failure(LoginStatus.InvalidPassword);
                }

                using (var authService = CreateServiceClient())
                {

                    var result = authService.v2ConvertTempAccount(_siteID, userID, encryptedPassword, desiredEmail);

                    switch (result.Status)
                    {
                        case EClaimTempAccountResultStatus.EmailInUse:
                            return ClaimAccountResult.Failure(RegisterStatus.EmailInUse);
                        case EClaimTempAccountResultStatus.InvalidEmail:
                            return ClaimAccountResult.Failure(RegisterStatus.InvalidEmail);
                        case EClaimTempAccountResultStatus.InvalidPassword:
                            return ClaimAccountResult.Failure(RegisterStatus.InvalidPassword);                            
                        case EClaimTempAccountResultStatus.Error:
                        case EClaimTempAccountResultStatus.InsufficientPermissions:
                            return ClaimAccountResult.Failure(RegisterStatus.GeneralError);

                    }
                    
                    return ClaimAccountResult.Success(result.SessionID, result.UserProfile);
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to make auth service call.");
                return null;
            }
        }

        internal static RenewTokenResult RenewAuthToken(string tokenValue)
        {
            var token = EncryptionToken.FromValue(tokenValue);

            // This should never happen
            if (!token.IsValid)
            {
                Logger.Warn("Attempt to renew invalid token", new { Token = tokenValue });
                return null;
            }
            
            // This will only let users renew their token once every X days
            if (!token.IsRenewable(AuthenticationContext.TokenDuration, RenewAfter))
            {
                Logger.Debug("Token will not be renewed. It is not within the renewal window.");
                return new RenewTokenResult(token);
            }

            // Token is renewable, so generate a new one
            var newToken = token.Renew(DateTime.UtcNow);

            return new RenewTokenResult(newToken);
        }

        public static RegisterResult RegisterUser(string username, string password, string email, bool newsletter)
        {
            using (var authService = CreateServiceClient())
            {
                try
                {
                    var result = authService.v2AddUser(_siteID,
                        username,
                        _outgoingCipher.Encrypt(password),
                        null,
                        null,
                        null,
                        null,
                        "US",
                        null,
                        null,
                        email,
                        newsletter,
                        null,
                        null,
                        null);

                    if (result.Status != EAddUserStatus.Successful)
                    {
                        return RegisterResult.Failure(result.Status);
                    }

                    var profile = authService.v2GetUserProfile(_siteID, result.UserID);

                    return RegisterResult.Success(result.SessionID, profile);


                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "AuthenticationManager - Failed to register user.", ex);
                    return RegisterResult.Error();
                }

            }
        }


        public static LoginResult LoginWithNetworkSession(int siteId, string sessionId)
        {
            using (var authService = CreateServiceClient())
            {
                try
                {
                    var userID = authService.v2ValidateUserSession(siteId, sessionId);
                    if (userID <= 0)
                    {
                        return LoginResult.Failure(LoginStatus.GeneralError);
                    }

                    var userProfile = authService.v2GetUserProfile(siteId, userID);
                    return LoginResult.Success(sessionId, userProfile);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "AuthenticationManager - Failed to login user with network session.");
                    return LoginResult.Failure(LoginStatus.GeneralError);
                }
            }
        }
    }    

}