﻿using System;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.CloudServices.Models;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.LoginsWebService.Authentication;
using Curse.Friends.LoginsWebService.Contracts;
using Curse.Friends.MicroService.Exceptions;
using Curse.Logging;
using Newtonsoft.Json;
using Curse.Friends.TwitchApi;
using Curse.Friends.TwitchIdentityMerge;
using Curse.Friends.LoginsWebService.Configuration;
using Curse.Friends.Tracing;
using User = Curse.Friends.Data.User;

namespace Curse.Friends.LoginsWebService.Controllers
{
    [RoutePrefix("login")]
    [AuthenticationFilter(AuthenticationLevel.Anonymous)]
    public class LoginController : MicroServiceController
    {

        private static readonly LogCategory FailedLoginsLogger = new LogCategory("FailedLogins") { Throttle = TimeSpan.FromSeconds(30), ReleaseLevel = LogLevel.Debug };
        private static readonly LogCategory FuelLogger = new LogCategory("Fuel") {Throttle = TimeSpan.FromSeconds(30), ReleaseLevel = LogLevel.Debug};
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("Logins");

        [HttpPost]
        [Route("")]
        [ResponseType(typeof(LoginResponse))]
        public IHttpActionResult Login(LoginRequest loginRequest)
        {
            loginRequest.Validate();

            var resp = LoginsAuthenticationProvider.LoginUser(loginRequest.Username, loginRequest.Password);

            if (resp.Status == LoginStatus.Success)
            {
                var userInfo = GetOrCreateUser(resp.UserID, resp.Username, resp.EmailAddress);
                EnsureBestRegion(userInfo);
                return Ok(new LoginResponse { Status = LoginStatus.Success, Session = resp.ToSession(), TwitchUsernameReservationToken = GetTwitchUsernameReservationToken(userInfo) });
            }

            var requestIp = GetCurrentIpAddress();

            if (resp.Status == LoginStatus.UnauthorizedLogin) // Account locked out. Log data for forensics.
            {
                FailedLoginsLogger.Debug("Attempt to login to locked out account", new { loginRequest.Username, IpAddress = requestIp });
            }
            else if (resp.Status == LoginStatus.InvalidPassword)
            {
                FailedLoginsLogger.Debug("Attempt to login with invalid password - " + requestIp, new { loginRequest.Username, IpAddress = requestIp });
            }
            

            var response = Request.CreateResponse(HttpStatusCode.BadRequest, new LoginResponse { Status = resp.Status, StatusMessage = resp.Status.ToString() });
            return ResponseMessage(response);
        }

        private static string GetTwitchUsernameReservationToken(UserRegionInfo userRegionInfo)
        {
            return IdentityMergeAuthHelper.CreateUsernameHmac(userRegionInfo.User.Username, LoginsWebServiceConfiguration.Current.TwitchLoginHmacSecret);
        }

        private UserRegionInfo GetOrCreateUser(int userID, string username, string email)
        {
            // From their storage region, get their record and update their status            
            var userAndRegion = GetUserAndRegion(userID, true);
            var reindexSearch = false;

            if (userAndRegion.User == null) // If it doesn't exist, insert or replace
            {
                userAndRegion.User = new Curse.Friends.Data.User
                {
                    UserID = userID,
                    Username = username,
                    IsTempAccount = TempAccount.IsTempAccount(userID),
                    GroupMessagePushPreference = PushNotificationPreference.Favorites,
                    FriendMessagePushPreference = PushNotificationPreference.All,
                    FriendRequestPushEnabled = true,
                    EmailAddress = email
                };

                userAndRegion.User.Insert(userAndRegion.Region.RegionID);
                userAndRegion.User = Curse.Friends.Data.User.Get(userAndRegion.Region.RegionID, userID);

                if (userAndRegion.User == null)
                {
                    throw new InvalidOperationException("Failed to retrieve user after inserting it.");
                }

                reindexSearch = true;
            }

            // Upsert their friend hints
            if (reindexSearch || FriendsServiceConfiguration.Instance.AlwaysIndexUserSearch)
            {
                try
                {
                    if (!string.IsNullOrEmpty(username))
                    {
                        new FriendHint() { UserID = userID, Type = FriendHintType.Username, SearchTerm = username, AvatarUrl = userAndRegion.User.AvatarUrl, Verification = FriendHintVerification.Verified }.InsertLocal();
                    }

                    if (!string.IsNullOrEmpty(email))
                    {
                        new FriendHint() { UserID = userID, Type = FriendHintType.Email, SearchTerm = email, AvatarUrl = userAndRegion.User.AvatarUrl, Verification = FriendHintVerification.Verified }.InsertLocal();
                    }

                    // Queue up a job to reindex the user
                    FriendHintSearchIndexer.CreateForUser(userAndRegion.User.UserID);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to create friend hint search indexer for registering user.");
                }
            }

            return userAndRegion;
        }

        /// <summary>
        /// Renews an authentication token near the end of its life.
        /// </summary>
        [HttpPost]
        [Route("renew")]
        [ResponseType(typeof(RenewTokenResponseContract))]
        [AuthenticationFilter]
        public IHttpActionResult RenewToken()
        {
            // Additional check to prevent a race condition            
            var userStats = UserStatistics.GetByUserID(Token.UserID);

            if (userStats == null)
            {
                FilteredLogger.Warn(Token.UserID, "Unable to renew token. User statistics record was not found in local database.", new { Token.UserID, Token.Username });
                return BadRequest("User not found.");
            }

            if (Token.IssuedTimestamp < userStats.TokenTimestamp)
            {
                LogExpiredToken(userStats, Token);                
                return Unauthorized(new {error = "Failed authentication"});
            }

            var renewal = LoginsAuthenticationProvider.RenewAuthToken(Token.Token);

            if (renewal == null)
            {
                return BadRequest("Received an invalid token.");
            }

            return Ok(renewal.ToContract());
        }

        private static void LogExpiredToken(UserStatistics userStats, AuthenticationToken token)
        {
            try
            {
                var tokenAgeMs = userStats.TokenTimestamp - token.IssuedTimestamp;
                var tokenAge = TimeSpan.FromMilliseconds(tokenAgeMs);
                FailedLoginsLogger.Warn("Attempt to renew an expired token. The user will be logged out.", new { TokenAgeMins = tokenAge.TotalMinutes.ToString("N2"), token.UserID, userStats.Username, userStats.DisplayName, token.IssuedTimestamp, userStats.TokenTimestamp });
            }
            catch (Exception ex)
            {
                FailedLoginsLogger.Error(ex, "Failed to log expired token info.");
            }            
        }

        [HttpPost]
        [Route("network-session")]
        [ResponseType(typeof(LoginResponse))]
        public IHttpActionResult LoginWithNetworkSession([FromBody] LoginWithNetworkSessionRequest request)
        {
            var resp = LoginsAuthenticationProvider.LoginWithNetworkSession(request.SiteID, request.SessionID);

            if (resp.Status == LoginStatus.Success)
            {
                return Ok(new LoginResponse { Status = LoginStatus.Success, Session = resp.ToSession() });
            }

            return BadRequest(new LoginResponse { Status = resp.Status, StatusMessage = resp.Status.ToString() });
        }

        [HttpPost]
        [Route("twitch-oauth")]
        [ResponseType(typeof(TwitchOAuthResponse))]
        public IHttpActionResult TwitchOAuth([FromBody] TwitchOAuthRequest request)
        {

            // Ensure the supplied data is valid
            try
            {
                request.Validate();
            }
            catch (RequestValidationException ex)
            {
                Logger.Warn(ex, "Twitch OAuth request not valid");
                return BadRequest(new TwitchOAuthResponse { Status = TwitchOAuthStatus.FailedValidation });
            }

            if (!TwitchApiHelper.IsValidClientID(request.ClientID))
            {
                Logger.Warn("Invalid client ID specified for Twitch OAuth", new { request.ClientID, request.RedirectUri });
                return BadRequest(new TwitchOAuthResponse { Status = TwitchOAuthStatus.FailedValidation });
            }

            var twitchApiClient = TwitchApiHelper.GetClient(request.ClientID);

            // Make an API request to Twitch, given the token
            var tokenResponse = twitchApiClient.GetTwitchToken(request.Code, request.State, request.RedirectUri);

            Logger.Trace("Received Twitch OAuth token response", new { tokenResponse });

            if (tokenResponse.Status != TwitchResponseStatus.Success)
            {
                Logger.Warn("Login Failed. Token Error.", tokenResponse);
                return BadRequest(new TwitchOAuthResponse { Status = TwitchOAuthStatus.FailedInvalidOAuthCode });
            }

            var token = tokenResponse.Value;

            // Get the user info from the Twitch API
            var twitchUserResp = twitchApiClient.GetUser(tokenResponse.Value.AccessToken);

            if (twitchUserResp.Status == TwitchResponseStatus.Unauthorized)
            {
                Logger.Warn("Failed to retrieve Twitch account from API due to bad token", new { twitchUserResp });
                return BadRequest(new TwitchOAuthResponse { Status = TwitchOAuthStatus.FailedInvalidOAuthCode });
            }

            if (twitchUserResp.Status != TwitchResponseStatus.Success)
            {
                Logger.Error("Failed to retrieve Twitch account from API", new { twitchUserResp });
                return ResponseMessage(Request.CreateResponse(HttpStatusCode.ServiceUnavailable, new TwitchOAuthResponse { Status = TwitchOAuthStatus.FailedOAuthError }));
            }

            // Make request to the auth service and local database
            var twitchUser = twitchUserResp.Value;
            var twitchUserInfo = new TwitchUserInfo(twitchUser, token, request.ClientID);
            var mergeState = IdentityMergeAuthHelper.GetIdentityMergeState(twitchUserInfo);
            
            if(mergeState.Status == IdentityMergeStatus.Failed)
            {
                Logger.Warn("Error retrieving identity merge state", new { mergeState, TwitchID = twitchUser.ID, TwitchName = twitchUser.Name });
                return StatusCode(HttpStatusCode.InternalServerError, new TwitchOAuthResponse { Status = TwitchOAuthStatus.FailedOAuthError, });
            }

            if (mergeState.Status == IdentityMergeStatus.Merged)
            {
                // Ensure that the user's data is local
                EnsureBestRegion(mergeState.CurseUserID);

                try
                {
                    // per https://jira.twitch.com/browse/CRSBE-73 Bandaid the invalid merge state and missing external account info on login. 
                    IdentityMergeAuthHelper.UpdateMergedUserAccountInfo(mergeState, twitchUserInfo);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error UpdateMergedUserAccountInfo on twitch oauth", new
                    {
                        request.ClientID,
                        curseUserId = mergeState.CurseUserID,
                        twitchUserId = mergeState.TwitchUserID,                        
                    });
                }


                return Ok(new TwitchOAuthResponse
                {
                    Status = TwitchOAuthStatus.Success,
                    Session = CreateLoginSessionFromMergeState(mergeState, twitchUser.Email),
                    Timestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                    TwitchUsername = twitchUser.Name,
                    TwitchDisplayName = twitchUser.DisplayName,
                    TwitchAvatar = twitchUser.Logo,
                    TwitchUserID = twitchUser.ID,
                });
            }
            
            // If the account has not yet been merged, create a OAuthState for this session
            var oAuthState = OAuthState.CreateForSeed(token, 600, request.ClientID);

            return Ok(new TwitchOAuthResponse
            {
                Status = TwitchOAuthStatus.RequiresMerge,
                MergeToken = oAuthState.Key,
                Timestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                TwitchUsername = twitchUser.Name,
                TwitchDisplayName = twitchUser.DisplayName,
                TwitchAvatar = twitchUser.Logo,
                TwitchUserID = twitchUser.ID,
            });
        }
        
        private static LoginSession CreateLoginSessionFromMergeState(IdentityMergeState mergeState, string email)
        {
            if (mergeState == null)
            {
                throw new ArgumentNullException("mergeState");
            }

            // See if this user has any bans
            var ban = UserBan.GetLocal(mergeState.CurseUserID);
            var bans = ban?.Type ?? UserBanType.None;
            return LoginsAuthenticationProvider.CreateLoginResult(mergeState.CurseUserID, mergeState.Username, mergeState.DisplayName, false, true, bans, email).ToSession();
        }

        private static void EnsureBestRegion(int userID)
        {
            if (userID <= 0)
            {
                return;
            }

            var userRegion = UserRegion.GetByUserID(userID);
            var user = userRegion?.GetUser();

            if (user == null)
            {
                return;                
            }

            EnsureBestRegion(userRegion, user);
        }

        private static void EnsureBestRegion(UserRegionInfo userRegionInfo)
        {
            EnsureBestRegion(userRegionInfo.Region, userRegionInfo.User);
        }

        // Based on a successful login, ensures that a user is in the best region (for latency)
        private static void EnsureBestRegion(UserRegion userRegion, User user)
        {
            if (userRegion == null || user == null)
            {
                return;                    
            }

            string reason;
            if (!userRegion.ShouldRelocate(out reason))
            {
                FilteredLogger.Log(user, "User will not be relocated to another region: " + reason, new { user = user.GetLogData(), userRegion, localRegion = UserRegion.LocalConfigID });
                return;                
            }

            FilteredLogger.Log(user, "Relocating user from remote region to local region", new { user = user.GetLogData(), userRegion, localRegion = UserRegion.LocalConfigID });
            UserRegionChangeWorker.Create(userRegion.UserID, UserRegion.LocalConfigID);

        }

        [HttpPost]
        [Route("twitch-merge-new")]
        [ResponseType(typeof(TwitchMergeResponse))]
        public IHttpActionResult TwitchMergeNew([FromBody] TwitchMergeNewRequest request)
        {
            try
            {
                request.Validate();
            }
            catch (RequestValidationException)
            {
                return BadRequest(new TwitchMergeResponse { Status = TwitchMergeStatus.FailedValidation });
            }


            TwitchUserInfo userInfo;
            var status = TryGetTwitchUser(request.MergeToken, out userInfo);
            if (status != TwitchMergeStatus.Success)
            {
                Logger.Warn("Could not retrieve Twitch user using the provided merge token", new { request, status });
                return BadRequest(new TwitchMergeResponse { Status = status });
            }

            // Get the existing merge state
            var mergeState = IdentityMergeAuthHelper.GetIdentityMergeState(userInfo);           

            // Provision a user, if it hasn't already been merged
            if (mergeState.Status == IdentityMergeStatus.Unmerged || mergeState.Status == IdentityMergeStatus.AutoProvisioned)
            {                               
                mergeState = IdentityMergeAuthHelper.ProvisionMappedUser(userInfo, false);
            }

            if (mergeState.Status != IdentityMergeStatus.Merged)
            {
                Logger.Warn("TwitchMergeNew: Unable to merge via a provisioned account.", new { mergeState, TwitchID = userInfo.UserID, TwitchName = userInfo.Username, request });
                return InternalServerError();
            }

            // Ensure that the user's data is local
            EnsureBestRegion(mergeState.CurseUserID);

            var session = CreateLoginSessionFromMergeState(mergeState, userInfo.Email);

            return Ok(new TwitchMergeResponse { Status = TwitchMergeStatus.Success, Session = session });
        }

        [HttpPost]
        [Route("twitch-merge-existing")]
        [ResponseType(typeof(TwitchMergeResponse))]
        public IHttpActionResult TwitchMergeExisting([FromBody] TwitchMergeExistingRequest request)
        {
            try
            {
                request.Validate();
            }
            catch (RequestValidationException)
            {
                return BadRequest(new TwitchMergeResponse { Status = TwitchMergeStatus.FailedValidation });
            }

            // Get the Twitch user info from the API
            TwitchUserInfo userInfo;
            var status = TryGetTwitchUser(request.MergeToken, out userInfo);
            if (status != TwitchMergeStatus.Success)
            {
                return BadRequest(new TwitchMergeResponse { Status = status });
            }

            // Get the existing merge state
            var mergeState = IdentityMergeAuthHelper.GetIdentityMergeState(userInfo);

            // If it has already been merged, we're done
            if (mergeState.Status == IdentityMergeStatus.Merged)
            {                                
                // The existing merge matches the requested user
                return Ok(new TwitchMergeResponse { Status = TwitchMergeStatus.Success, Session = CreateLoginSessionFromMergeState(mergeState, userInfo.Email) });                
            }

            // Otherwise, try to merge to an existing Curse account
            mergeState = IdentityMergeAuthHelper.MergeAccount(userInfo, request.Username, request.Password);


            if (mergeState.Status != IdentityMergeStatus.Merged)
            {
                return BadRequest(new TwitchMergeResponse { Status = GetMergeStatusFromReason(mergeState.FailureReason) });
            }

            // Ensure that the user's data is local
            EnsureBestRegion(mergeState.CurseUserID);

            return Ok(new TwitchMergeResponse { Status = TwitchMergeStatus.Success, Session = CreateLoginSessionFromMergeState(mergeState, userInfo.Email) });

        }

        [HttpPost]
        [Route("twitch-merge-temp-account")]
        [ResponseType(typeof(TwitchMergeResponse))]
        public IHttpActionResult TwitchMergeTempAccount([FromBody] TwitchMergeTempAccountRequest request)
        {
            try
            {
                request.Validate();
            }
            catch (RequestValidationException)
            {
                return BadRequest(new TwitchMergeResponse { Status = TwitchMergeStatus.FailedValidation });
            }


            // Get the temp account
            var tempAccount = TempAccount.GetLocal(request.TempAccountToken);

            if (tempAccount == null)
            {
                return NotFound();
            }

            if (tempAccount.Status == TempAccountStatus.Claimed)
            {
                return BadRequest("Account is already claimed");
            }

            // Get the Twitch user info from the API
            TwitchUserInfo userInfo;
            var status = TryGetTwitchUser(request.MergeToken, out userInfo);
            if (status != TwitchMergeStatus.Success)
            {
                return BadRequest(new TwitchMergeResponse { Status = status });
            }

            // Get the existing merge state
            var mergeState = IdentityMergeAuthHelper.GetIdentityMergeState(userInfo);

            // If it has already been merged, we're done
            if (mergeState.Status == IdentityMergeStatus.Merged)
            {
                return Ok(new TwitchMergeResponse { Status = TwitchMergeStatus.Success, Session = CreateLoginSessionFromMergeState(mergeState, userInfo.Email) });
            }

            // Otherwise, try to merge to an existing Curse account
            mergeState = IdentityMergeAuthHelper.MergeTempAccount(tempAccount, userInfo);

            if (mergeState.Status != IdentityMergeStatus.Merged)
            {
                return BadRequest(new TwitchMergeResponse { Status = GetMergeStatusFromReason(mergeState.FailureReason) });
            }

            // Ensure that the user's data is local
            EnsureBestRegion(mergeState.CurseUserID);

            return Ok(new TwitchMergeResponse { Status = TwitchMergeStatus.Success, Session = CreateLoginSessionFromMergeState(mergeState, userInfo.Email) });

        }


#if ENABLE_UNMERGE

        [HttpPost]
        [Route("twitch-unmerge/{twitchID}")]
        [ResponseType(typeof(void))]
        public IHttpActionResult TwitchUnmerge(int twitchID)
        {
            IdentityMergeAuthHelper.UnmergeAccount(twitchID);
            // Make the auth call to unmerge it
            return Ok();

        }

        [HttpGet]
        [Route("test-hmac/{username}")]
        [ResponseType(typeof(string))]
        public IHttpActionResult TestHmac(string username)
        {
            // Make the auth call to unmerge it
            return Ok(IdentityMergeAuthHelper.CreateUsernameHmac(username, LoginsWebServiceConfiguration.Current.TwitchLoginHmacSecret));

        }
#endif


        [HttpPost]
        [Route("fuel")]
        [ResponseType(typeof(FuelAuthorizationResponseContract))]
        //[AuthenticationFilter]
        public IHttpActionResult GetFuelToken()
        {
            if (Token.IsAnonymous || Token.IsApiRequest)
            {
                return ErrorResponse(HttpStatusCode.Forbidden, "user_not_authenticated", "Attempt to get a token without authentication");
            }

            var user = GetCurrentUserAndRegion();
            if (!user.User.IsMerged)
            {
                return ErrorResponse(HttpStatusCode.BadRequest, "user_not_merged", "The requesting user is not merged");
            }

            var account = ExternalAccount.GetLocal(user.User.TwitchID, AccountType.Twitch);
            if (account == null)
            {
                return ErrorResponse(HttpStatusCode.InternalServerError, "account_not_found", "The Twitch account for the user cannot be found");
            }

            var clientID = account.ClientID;

            string secret;
            if (string.IsNullOrEmpty(clientID))
            {
                // Fall back to the first valid client ID at least temporarily for those that merged prior to fixing the merge helper to store Client ID
                clientID = LoginsWebServiceConfiguration.Current.TwitchLoginClientMap.First().Key;
            }
            else if (!LoginsWebServiceConfiguration.Current.TwitchLoginClientMap.TryGetValue(clientID, out secret))
            {
                // Fall back to the first valid client ID at least temporarily for those that merged prior to fixing the merge helper to store Client ID
                clientID = LoginsWebServiceConfiguration.Current.TwitchLoginClientMap.First().Key;
            }

            var resp = TwitchApiHelper.GetClient(clientID).AuthorizeFuel(account.AuthToken, LoginsWebServiceConfiguration.Current.FuelScopes);
            if (resp.Status != TwitchResponseStatus.Success)
            {
                FuelLogger.Debug("Fuel API Error " + resp.Status, resp);
                return ErrorResponse(HttpStatusCode.InternalServerError, "fuel_error", resp.Value?.Error);
            }

            return Ok(FuelAuthorizationResponseContract.FromModel(resp.Value));
        }

        private static TwitchMergeStatus GetMergeStatusFromReason(IdentityMergeFailureReason reason)
        {
            switch (reason)
            {
                case IdentityMergeFailureReason.GeneralError:
                    return TwitchMergeStatus.FailedGeneralError;
                case IdentityMergeFailureReason.ValidationError:
                    return TwitchMergeStatus.FailedValidation;
                case IdentityMergeFailureReason.UnknownUser:
                    return TwitchMergeStatus.FailedUnknownUser;
                case IdentityMergeFailureReason.InvalidPassword:
                    return TwitchMergeStatus.FailedInvalidPassword;
                case IdentityMergeFailureReason.CurseAccountMerged:
                    return TwitchMergeStatus.FailedCurseAccountMerged;
                case IdentityMergeFailureReason.TwitchAccountMerged:
                    return TwitchMergeStatus.FailedTwitchAccountMerged;
                case IdentityMergeFailureReason.DeletedAccount:
                    return TwitchMergeStatus.FailedDeletedAccount;
                default:
                    throw new ArgumentOutOfRangeException("reason", reason, null);
            }
        }

        private TwitchMergeStatus TryGetTwitchUser(string mergeToken, out TwitchUserInfo user)
        {
            user = null;
            var oAuthState = OAuthState.GetLocal(mergeToken);

            if (oAuthState == null)
            {
                return TwitchMergeStatus.FailedInvalidMergeToken;
            }

            var token = JsonConvert.DeserializeObject<TwitchAccessTokenResponse>(oAuthState.Data);

            // User the token to retrieve the user from the Twitch API

            // Get the user info from the Twitch API
            var twitchUserResp = TwitchApiHelper.GetClient(oAuthState.ClientID).GetUser(token.AccessToken);

            if (twitchUserResp.Status == TwitchResponseStatus.Unauthorized)
            {
                return TwitchMergeStatus.FailedInvalidOAuthCode;
            }

            if (twitchUserResp.Status != TwitchResponseStatus.Success)
            {
                Logger.Error("Failed to retrieve Twitch account from API", new { twitchUserResp });
                return TwitchMergeStatus.FailedOAuthError;
            }

            user = new TwitchUserInfo(twitchUserResp.Value, token, oAuthState.ClientID);
            return TwitchMergeStatus.Success;
        }
    }
}