﻿using System;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Extensions;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Utils;
using Curse.Friends.LoginsWebService.Contracts;
using Curse.Friends.MicroService;
using Curse.Friends.LoginsWebService.Authentication;
using Curse.Friends.LoginsWebService.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;

namespace Curse.Friends.LoginsWebService.Controllers
{
    [RoutePrefix("register")]
    [AuthenticationFilter(AuthenticationLevel.Anonymous)]
    public class RegisterController : MicroServiceController
    {
        [HttpPost]
        [Route("")]
        [ResponseType(typeof(RegisterResponse))]
        public IHttpActionResult Register(RegisterRequest registerRequest)
        {
            registerRequest.Validate();

            var resp = LoginsAuthenticationProvider.RegisterUser(registerRequest.Username, registerRequest.Password, registerRequest.Email, registerRequest.Newsletter);

            if (resp.Status == RegisterStatus.Success)
            {
                return Ok(new RegisterResponse { Session = resp.LoginResult.ToSession(), Status = RegisterStatus.Success });
            }
            
            return BadRequest(new RegisterResponse { Status = resp.Status, StatusMessage = resp.Status.ToString() });


        }

        [HttpPost]
        [Route("check-username")]
        [ResponseType(typeof(bool))]
        public IHttpActionResult CheckUsername([FromBody] string username)
        {
            if (!username.SafeRange(1, 32))
            {
                return BadRequest("Invalid username!");
            }

            return Ok(LoginsAuthenticationProvider.IsUsernameAvailable(username));
        }

        [HttpPost]
        [Route("create-temp")]
        [ResponseType(typeof(RegisterTempResponse))]
        public IHttpActionResult RegisterTempAccount(RegisterTempRequest registerRequest)
        {
            registerRequest.Validate();

            if (!LoginsAuthenticationProvider.IsUsernameAvailable(registerRequest.Username))
            {
                return BadRequest(new RegisterTempResponse { Status = RegisterStatus.UsernameInUse });
            }

            var currentIpAddress = GetCurrentIpAddress();

            if (!TempAccount.CanCreate(currentIpAddress.ToString(), LoginsWebServiceConfiguration.Current.MaxAccountsPerIpPerHour))
            {
                return BadRequest(new RegisterTempResponse { Status = RegisterStatus.TooManyAccountsSameIp });
            }

            // Create a temp account
            var tempAccount = TempAccount.Create(currentIpAddress.ToString());

            tempAccount.InsertLocal();

            var resp = LoginsAuthenticationProvider.RegisterUser(registerRequest.Username, ShortCodeGenerator.RandomCharacters(32), tempAccount.Token.ToString().Replace("-", string.Empty) + "@cursetemp.com", false);

            if (resp.Status != RegisterStatus.Success)
            {
                return BadRequest(new RegisterTempResponse { Status = resp.Status, StatusMessage = resp.Status.ToString() });
            }

            // Update the temp account to associate it with this user account
            tempAccount.UserID = resp.LoginResult.UserID;
            tempAccount.Status = TempAccountStatus.Active;
            tempAccount.Update();

            var session = resp.LoginResult.ToSession();
            session.IsTemporaryAccount = true;

            return Ok(new RegisterTempResponse { TempAccountToken = tempAccount.Token.ToString(), Session = session, Status = RegisterStatus.Success });

        }

        [HttpPost]
        [Route("claim-temp")]
        [ResponseType(typeof(RegisterResponse))]
        public IHttpActionResult ClaimTempAccount(ClaimTempRequest claimRequest)
        {
            claimRequest.Validate();

            // Get the temp account
            var tempAccount = TempAccount.GetLocal(claimRequest.TempAccountToken);

            if (tempAccount == null)
            {
                return NotFound();
            }


            var userAndRegion = GetUserAndRegion(tempAccount.UserID);

            if (tempAccount.Status != TempAccountStatus.Active)
            {
                return BadRequest(new RegisterResponse {Status = RegisterStatus.GeneralError, StatusMessage = "The temp account specified is no longer active, and cannot be claimed."});
            }
            
            // Look up the account in auth
            var claimResult = LoginsAuthenticationProvider.ClaimAccount(tempAccount.UserID, claimRequest.Email, claimRequest.Password, LoginsWebServiceConfiguration.Current.EmailValidationRedirect);

            if (claimResult.Status != RegisterStatus.Success)
            {
                return BadRequest(new RegisterResponse {Status = claimResult.Status});
            }
            

            // Update the temp account to associate it with this user account
            tempAccount.Status = TempAccountStatus.Claimed;
            tempAccount.Update();

            userAndRegion.User.IsTempAccount = false;
            userAndRegion.User.Update(p => p.IsTempAccount);

            userAndRegion.User.UpdateEmail(claimRequest.Email);

            return Ok(new RegisterResponse { Session = claimResult.LoginResult.ToSession(), Status = RegisterStatus.Success });

        }
    }
}