﻿using System;
using System.Net.Http;
using System.Web;
using Curse.CloudServices.Models;
using Curse.Friends.Data.Models;
using Curse.Friends.MicroService.Extensions;
using Curse.Logging;
using Curse.ServiceEncryption;
using Curse.Extensions;

namespace Curse.Friends.MicroService
{
    public static class AuthenticationContext
    {
        public static TimeSpan TokenDuration = TimeSpan.FromDays(90);

        private static readonly LogCategory Logger = new LogCategory("AuthenticationContext") { Throttle = TimeSpan.FromMinutes(1) };

        public static AuthenticationToken GetTokenFromRequest(HttpRequestMessage request)
        {
            if (request == null)
            {
                throw new Exception("Unable to establish an authentication context. Request is null!");
            }

            // Try to get the token from the header
            var tokenValue = request.GetHeader(AuthenticationToken.HttpHeaderName);
            if (!string.IsNullOrWhiteSpace(tokenValue))
            {
                return ParseToken(tokenValue, false);    
            }
            
            // Next check the legacy header           
            tokenValue = request.GetHeader(AuthenticationToken.LegacyHttpHeaderName);
            if (!string.IsNullOrWhiteSpace(tokenValue))
            {
                return ParseToken(tokenValue, false);
            }
           
            // If still empty, check the cookie
            tokenValue = request.GetCookie(AuthenticationToken.HttpCookieName);
            if (!string.IsNullOrWhiteSpace(tokenValue))
            {
                return ParseToken(tokenValue, true);
            }

            return ParseToken(null, false);   
        }

        public static AuthenticationToken Current
        {
            get
            {
                var httpContext = HttpContext.Current;
                if (httpContext == null)
                {
                    throw new Exception("Unable to establish an authentication context. HttpContext.Current is null!");
                }

                // Try to get the token from the header
                var tokenValue = httpContext.Request.Headers[AuthenticationToken.HttpHeaderName];
                if (!string.IsNullOrWhiteSpace(tokenValue))
                {
                    return ParseToken(tokenValue, false);
                }

                // Next check the legacy header           
                tokenValue = httpContext.Request.Headers[AuthenticationToken.LegacyHttpHeaderName];
                if (!string.IsNullOrWhiteSpace(tokenValue))
                {
                    return ParseToken(tokenValue, false);
                }

                // If still empty, check the cookie
                var cookie = httpContext.Request.Cookies.Get(AuthenticationToken.HttpCookieName);
                if (cookie != null)
                {
                    tokenValue = HttpUtility.UrlDecode(cookie.Value);
                    if (!string.IsNullOrWhiteSpace(tokenValue))
                    {
                        return ParseToken(tokenValue, true);
                    }
                }
                
                return ParseToken(null, false); 
            }
        }

        public static bool IsTempAccount
        {
            get
            {
                if (Current.IsAnonymous)
                {
                    return false;
                }

                try
                {
                    return TempAccount.IsTempAccount(Current.UserID);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to determine if account is temporary");
                    return true;
                }
            }

        }

        

        private static AuthenticationToken ParseToken(string tokenString, bool isFromCookie)
        {
            
            if (string.IsNullOrWhiteSpace(tokenString))
            {
                return new AuthenticationToken { IsAnonymous = true };
            }

            // Special case for client issue
            if (tokenString.Equals("undefined"))
            {
                return new AuthenticationToken { IsAnonymous = true, IsInvalid = true, Token = tokenString };
            }

            var encryptedToken = EncryptionToken.FromValue(tokenString);

            // Invalid Token
            if (!encryptedToken.IsValid)
            {                
                return new AuthenticationToken { IsAnonymous = true, IsInvalid = true, Token = tokenString};
            }

            var timestamp = encryptedToken.GetDateTime("Timestamp");

            // Expired Token
            if (DateTime.UtcNow - timestamp > TokenDuration)
            {
                Logger.Warn("Authentication request with an expired token!", new { TokenValue = tokenString });
                return new AuthenticationToken { IsAnonymous = true };
            }            

            return new AuthenticationToken
            {
                Token = tokenString,                
                UserID = encryptedToken.GetInteger("UserID"),
                Username = encryptedToken.GetValue("Username"),
                ApiKey = encryptedToken.GetValue("ApiKey"),
                IsFromCookie = isFromCookie,
                IssuedTimestamp = timestamp.SafeToEpochMilliseconds()
            };
        }

        public static bool IsTokenValid(string tokenString)
        {
            var encryptedToken = EncryptionToken.FromValue(tokenString);
            return encryptedToken.IsValid;
        }

    }
}
