﻿using System;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Web.Http.Controllers;
using System.Web.Http.Filters;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Logging;

namespace Curse.Friends.MicroService
{
    public enum AuthenticationLevel
    {
        LoggedIn,
        Anonymous,
        ApiKey
    }

    public class AuthenticationFilter : ActionFilterAttribute
    {
        private readonly AuthenticationLevel _authenticationLevel;
        private readonly bool _isGlobal;
        private readonly bool _allowCookies;

        public AuthenticationFilter(bool isGlobal, AuthenticationLevel authenticationLevel, bool allowCookies)
        {
            _authenticationLevel = authenticationLevel;
            _isGlobal = isGlobal;
#if CONFIG_DEBUG
            _allowCookies = true;
#else
            _allowCookies = allowCookies;
#endif

        }

        public AuthenticationFilter(AuthenticationLevel authenticationLevel = AuthenticationLevel.LoggedIn, bool allowCookies = false) : this(false, authenticationLevel, allowCookies)
        {
        }
        
        public override void OnActionExecuting(HttpActionContext actionContext)
        {
            var filter = actionContext.ActionDescriptor.GetCustomAttributes<AuthenticationFilter>().FirstOrDefault();
            if (filter != null && _isGlobal)
            {
                // This has been overridden at a lower level.
                base.OnActionExecuting(actionContext);
                return;
            }
            
            var token = AuthenticationContext.GetTokenFromRequest(actionContext.Request);

            // We have a matching API key
            if (token.ApiKey != null && token.ApiKey == FriendsServiceConfiguration.Instance.CentralServiceApiKey)
            {
                token.IsApiRequest = true;
                return;
            }

            // If the token is invalid, do not log the user out. Just return a forbidden
            if (token.IsInvalid && _authenticationLevel != AuthenticationLevel.Anonymous)
            {
                actionContext.Response = new HttpResponseMessage(HttpStatusCode.Forbidden) { Content = new StringContent("{\"error\":\"Invalid authentication token.\"}") };
            }

            if (token.IsAnonymous && _authenticationLevel != AuthenticationLevel.Anonymous)
            {
                actionContext.Response = new HttpResponseMessage(HttpStatusCode.Unauthorized) { Content = new StringContent("{\"error\":\"Failed Authentication\"}") };
            }

            if (_authenticationLevel == AuthenticationLevel.LoggedIn)
            {
                var stats = UserStatistics.GetByUserOrDefault(token.UserID);

                if (stats.TokenTimestamp > token.ValidThrough)
                {
                    Logger.Warn("Failed authentication, due to revoked token.", new { stats.TokenTimestamp, token.ValidThrough });
                    actionContext.Response = new HttpResponseMessage(HttpStatusCode.Unauthorized) { Content = new StringContent("{\"error\":\"Failed Authentication\"}") };
                }
            }

            if (_authenticationLevel != AuthenticationLevel.Anonymous && token.IsFromCookie && !_allowCookies)
            {
                actionContext.Response = new HttpResponseMessage(HttpStatusCode.Forbidden) { Content = new StringContent("{\"error\":\"Cookie authentication is not permitted for this service endpoint.\"}") };
            }

            base.OnActionExecuting(actionContext);
        }
    }
}
