﻿using System;

namespace Curse.Friends.MicroService.Exceptions
{
    public enum FileUploadFailureReason
    {
        /// <summary>
        /// No file in the request
        /// </summary>
        Missing,
        
        /// <summary>
        /// The user is attempting to upload a file that is too large for the configured limits
        /// </summary>
        TooManyBytes,

        /// <summary>
        /// The user is attempting to upload too frequently        
        /// </summary>
        TooManyUploads,

        // Image specific failures
        /// <summary>
        /// The dimensions of the image are too large
        /// </summary>
        TooLarge = 1000, 

        /// <summary>
        /// The dimensions of the image are too small
        /// </summary>
        TooSmall,

        /// <summary>
        /// The image is an incorrect aspect ratio
        /// </summary>
        IncorrectAspectRatio,

        /// <summary>
        /// The image is in an incompatible format
        /// </summary>
        UnsupportedFormat,   


        InvalidUrl, // The image URL is invalid        ,

        VirusDetected,
        VirusScanError,
    }

    public class FileUploadException : Exception
    {
        public FileUploadFailureReason Reason { get; private set; }

        public TimeSpan? RetryAfter { get; private set; }

        public long RequestedFileSize { get; private set; }

        public long MaxFileSize { get; private set; }


        public override string Message
        {
            get
            {                
                return "File upload failed: " + Reason;                
            }
        }

        public FileUploadException(FileUploadFailureReason reason, TimeSpan? retryAfter = null)        
        {
            Reason = reason;
            RetryAfter = retryAfter;            
        }

        public FileUploadException(long requestedFileSize, long maxFileSize)            
        {
            Reason = FileUploadFailureReason.TooManyBytes;
            RequestedFileSize = requestedFileSize;
            MaxFileSize = maxFileSize;        
        }

        public object ToResponse()
        {
            return new
            {
                Reason,
                RetryAfter = RetryAfter.HasValue ? (int)RetryAfter.Value.TotalSeconds : 0,
                RequestedFileSize,
                MaxFileSize,
                Message
            };
        }
    }
}