﻿using System.Collections.Generic;
using System.Net;
using System.Net.Http;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Http.Filters;
using Curse.Logging;
using System;
using System.Linq;
using Curse.CloudServices.Models;
using Curse.Friends.MicroService.ExceptionHandlers;

namespace Curse.Friends.MicroService.Filters
{
    public class MicroServiceExceptionFilterAttribute : ActionFilterAttribute, IExceptionFilter
    {
        private static readonly Dictionary<Type, IMicroServiceExceptionHandler> ExceptionHandlers =
            new Dictionary<Type, IMicroServiceExceptionHandler>();

        private static void RegisterExceptionHandler(IMicroServiceExceptionHandler handler)
        {
            if (ExceptionHandlers.ContainsKey(handler.ExceptionType))
            {
                return;
            }

            ExceptionHandlers.Add(handler.ExceptionType, handler);
        }

        static MicroServiceExceptionFilterAttribute()
        {
            RegisterExceptionHandler(new DataConflictExceptionHandler());
            RegisterExceptionHandler(new DataNotFoundExceptionHandler());
            RegisterExceptionHandler(new FileUploadExceptionHandler());
            RegisterExceptionHandler(new FriendshipPermissionExceptionHandler());
            RegisterExceptionHandler(new GroupPermissionExceptionHandler());
            RegisterExceptionHandler(new RequestThrottledExceptionHandler());
            RegisterExceptionHandler(new RequestValidationExceptionHandler());
            RegisterExceptionHandler(new DataValidationExceptionHandler());
            RegisterExceptionHandler(new UserNotFoundExceptionHandler());
        }

        public Task ExecuteExceptionFilterAsync(HttpActionExecutedContext actionExecutedContext, CancellationToken cancellationToken)
        {
            if (cancellationToken.IsCancellationRequested)
            {
                // Request was canceled by the framework/client, don't do anything
                return Task.FromResult(0);
            }

            var exception = actionExecutedContext.Exception;

            // If the exception is null for whatever reason, return a generic 500 error
            if (exception == null)
            {
                actionExecutedContext.Response = new HttpResponseMessage(HttpStatusCode.InternalServerError);
                return Task.FromResult(0);
            }

            // See if this exception type has a handler
            IMicroServiceExceptionHandler handler;
            if (ExceptionHandlers.TryGetValue(exception.GetType(), out handler))
            {
                return handler.Handle(exception, actionExecutedContext, cancellationToken);
            }

            handler = ExceptionHandlers.Where(kvp => kvp.Key.IsInstanceOfType(exception)).Select(kvp => kvp.Value).FirstOrDefault();
            if (handler != null)
            {
                return handler.Handle(exception, actionExecutedContext, cancellationToken);
            }

            // Otherwise, return a generic 500 error
            var token = AuthenticationContext.GetTokenFromRequest(actionExecutedContext.Request);
            var initException = exception as TypeInitializationException;
            var aggException = exception as AggregateException;
            if (initException != null)
            {
                Logger.Error(initException.InnerException ?? exception, "Unhandled exception: " + actionExecutedContext.ActionContext.ActionDescriptor.ActionName,
                    new { url = actionExecutedContext.Request.RequestUri.ToString(), tokenInfo = GetTokenInfo(token) });
            }
            else if (aggException != null)
            {
                Logger.Error(aggException.InnerException ?? exception, "Unhandled exception: " + actionExecutedContext.ActionContext.ActionDescriptor.ActionName,
                    new {url = actionExecutedContext.Request.RequestUri.ToString(), tokenInfo = GetTokenInfo(token) });
            }
            else
            {
                Logger.Error(exception, "Unhandled exception: " + actionExecutedContext.ActionContext.ActionDescriptor.ActionName, 
                    new { url = actionExecutedContext.Request.RequestUri.ToString(), tokenInfo = GetTokenInfo(token) });
            }

            actionExecutedContext.Response = new HttpResponseMessage(HttpStatusCode.InternalServerError);
            return Task.FromResult(0);
        }

        private static object GetTokenInfo(AuthenticationToken token)
        {
            return new
            {
                token.IsInvalid,
                token.IsAnonymous,
                token.IsApiRequest,
                token.UserID,
                token.Username,
                token.IssuedTimestamp
            };
        }
    }
}