﻿using System;
using System.Diagnostics;
using System.Net;
using Curse.Friends.Configuration;
using Curse.Friends.MicroService.Exceptions;
using Curse.Logging;
using Newtonsoft.Json;
using System.Web;
using Curse.CloudServices.Authentication.Web;
using Curse.Logging.Uploader;
using Curse.ServiceEncryption;
using System.Web.Configuration;
using System.Web.Http;
using System.Xml.Linq;
using Curse.Friends.MicroService.Areas.HelpPage;
using System.IO;
using System.Web.Mvc;
using System.Web.Routing;
using System.Reflection;
using Curse.Friends.Statistics;
using Curse.Friends.ServiceClients;

namespace Curse.Friends.MicroService
{

    public abstract class MicroServiceApplication : HttpApplication
    {
        protected static DomainWhitelist CorsWhitelist;
        public static int MaxContentLength { get; private set; }
        public static int MaxRequestLength { get; private set; }

        protected virtual bool OverrideHomeController { get { return false; } }

        private int GetMaxRequestLength()
        {
            try
            {
                var config = WebConfigurationManager.OpenWebConfiguration("~");
                var section = config.GetSection("system.web/httpRuntime") as HttpRuntimeSection;
                // maxRequestLength is in KB, convert to bytes for use in BeginRequest.
                return section.MaxRequestLength * 1024;
            }
            catch
            {
                return 4194304; // Default to 4096KB
            }
        }

        private int GetMaxContentLength()
        {
            try
            {                
                var config = WebConfigurationManager.OpenWebConfiguration("~");
                var section = config.GetSection("system.webServer");
                var xml = section.SectionInformation.GetRawXml();
                var doc = XDocument.Parse(xml);
                var element = doc.Root.Element("security").Element("requestFiltering").Element("requestLimits");
                return int.Parse(element.Attribute("maxAllowedContentLength").Value);                
            }
            catch
            {                
                return 10485760; // Default to 10MB (30000000 is the actual default when not specified)
            }
        }

        protected abstract ServiceHostType ServiceType { get; }
        protected abstract string ServiceDescription { get; }
        protected virtual void CustomApplicationStartup() {  }
        protected virtual void CustomApplicationShutdown() { }
        protected abstract ConfigurationServices ConfigurationServices { get; }

        private AssemblyName GetAppInfo()
        {

            try
            {
                var type = GetType();

                while (!type.FullName.StartsWith("Curse"))
                {
                    type = type.BaseType;
                }

                return type.Assembly.GetName();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to get app assembly info");
            }

            return null;
        }

        protected void Application_Start()
        {
            var sw = Stopwatch.StartNew();

            MicroServiceConfiguration.ServiceType = ServiceType;
            MicroServiceConfiguration.ServiceDescription = ServiceDescription;

            EncryptionToken.Initialize(FriendsServiceConfiguration.Instance.EncryptionKey, FriendsServiceConfiguration.Instance.EncryptionIterations);

            MaxRequestLength = GetMaxRequestLength();
            MaxContentLength = GetMaxContentLength();

            // Web API configuration and services
            GlobalConfiguration.Configuration.Formatters.Remove(GlobalConfiguration.Configuration.Formatters.XmlFormatter);

#if CONFIG_DEBUG || CONFIG_STAGING
            var json = GlobalConfiguration.Configuration.Formatters.JsonFormatter;
            json.SerializerSettings.Formatting = Formatting.Indented;
#endif

            Logger.Init(@"C:\Curse\Logs", ServiceType.ToString());

            var appInfo = GetAppInfo();
            
            if (appInfo != null)
            {
                Logger.Info(ServiceType + " version '" + appInfo.Version + "' starting...", new { Version = appInfo.Version.ToString(), appInfo.Name });
            }
            
            Logger.Info("Initializing friends service client...");
            FriendsServiceClients.Initialize(FriendsServiceConfiguration.Instance.EncryptionKey, FriendsServiceConfiguration.Instance.EncryptionIterations, FriendsServiceConfiguration.Instance.CentralServiceApiKey);

            LogUploader.Initialize((int)ServiceType, FriendsServiceConfiguration.Instance.CentralLogServiceUrl, FriendsServiceConfiguration.Instance.LoggingServiceApiKey);
            
            if (!StatsEnabled)
            {
                FriendsStatsManager.Disable();
            }

            Logger.Info("Determining current region...");
            var currentRegion = StorageConfiguration.GetCurrentRegion();

            if (currentRegion == null)
            {
                Logger.Error("Failed to determine current region!");
            }
            else
            {
                Logger.Info("Current region detected: "  + currentRegion.Key);
            }

            try
            {
                FriendsStatsManager.Initialize((int)ServiceType, ServiceType.ToString(), currentRegion != null ? currentRegion.ID : 0, null, HostCounter.All);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to initialize stats manager.");
            }

            FriendsStatsManager.BeginStartup();
            StorageConfiguration.Initialize(ServiceType.ToString(), null, null, ConfigurationServices);
            
            CorsWhitelist = new DomainWhitelist(FriendsServiceConfiguration.Instance.CrossDomainWhitelist);
            
            CustomApplicationStartup();

            SetupGlobalDefaults();

            
            try
            {
                var basePath = HttpContext.Current.Server.MapPath("~");
                var xmlPath = Path.Combine(basePath, "bin", "Docs.xml");
                GlobalConfiguration.Configuration.SetDocumentationProvider(new XmlDocumentationProvider(xmlPath));
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            

            sw.Stop();
            Logger.Info("Service started in " + sw.Elapsed.TotalSeconds.ToString("F1") + " seconds.");            
            FriendsStatsManager.Started();            
        }

        protected virtual bool StatsEnabled
        {
            get { return true; }
        }

        protected virtual string TwitchClientID { get { return null; } }

        protected void SetupGlobalDefaults()
        {
            // Register all areas 
            AreaRegistration.RegisterAllAreas();

            // Ignore AXD from routing
            RouteTable.Routes.IgnoreRoute("{resource}.axd/{*pathInfo}");

            if (!OverrideHomeController)
            {
                RouteTable.Routes.MapRoute(
                    name: "Default",
                    url: "",
                    defaults: new { controller = "Home", action = "Index" },
                    constraints: new { httpMethod = new HttpMethodConstraint("GET", "HEAD") }
                    );
            }

            // Register global error filter
            GlobalFilters.Filters.Add(new HandleErrorAttribute());
            
            // Register WebAPI Routes
            GlobalConfiguration.Configuration.MapHttpAttributeRoutes(new InheritingDirectRouteProvider());

            GlobalConfiguration.Configuration.EnsureInitialized(); 
        }


        private void LogApplicationShutdownReason()
        {
            try
            {
                var runtime = (HttpRuntime)typeof(HttpRuntime).InvokeMember("_theRuntime",
                                                                   BindingFlags.NonPublic
                                                                   | BindingFlags.Static
                                                                   | BindingFlags.GetField,
                                                                   null,
                                                                   null,
                                                                   null);

                if (runtime == null)
                {
                    return;
                }

                var shutDownMessage = (string)runtime.GetType().InvokeMember("_shutDownMessage",
                                                                                 BindingFlags.NonPublic
                                                                                 | BindingFlags.Instance
                                                                                 | BindingFlags.GetField,
                                                                                 null,
                                                                                 runtime,
                                                                                 null);
                shutDownMessage = shutDownMessage.Replace("\n", ", ");

                Logger.Info("Shutdown message: " + shutDownMessage);
            }
            catch (Exception ex)
            {

                Logger.Error(ex, "Unable to log shutdown reason.");
            }
           
        }

        protected void Application_End()
        {
            CustomApplicationShutdown();
            FriendsStatsManager.BeginShutdown();            
            Logger.Info("Service Stopping...");
            LogApplicationShutdownReason();
            StorageConfiguration.Shutdown();
            Logger.Info("Service Stopped!");
            LogUploader.Shutdown();
            FriendsStatsManager.Shutdown();
        }

        protected void Application_Error()
        {
            var error = Server.GetLastError();
            if (error != null && !(error is HttpException) && !(error is OperationCanceledException))
            {
                Logger.Error(error);
            }
        }

        private static readonly LogCategory CorsLogger = new LogCategory("Cors") {ReleaseLevel = LogLevel.Trace, Throttle = TimeSpan.FromMinutes(1) };

        private static void WriteCors(HttpRequest req, HttpResponse res)
        {
            var origin = req.Headers["Origin"];
            if (!string.IsNullOrWhiteSpace(origin))
            {
                if (CorsWhitelist.IsWhiteListed(origin))
                {
                    res.Headers["Access-Control-Allow-Origin"] = origin;
                    res.Headers["Access-Control-Allow-Credentials"] = "true";

                    var requestedMethod = req.Headers["Access-Control-Request-Method"];
                    if (!string.IsNullOrEmpty(requestedMethod))
                    {
                        res.Headers["Access-Control-Allow-Methods"] = "DELETE, GET, HEAD, PATCH, POST, PUT";
                        res.Headers["Access-Control-Max-Age"] = "3600";
                        var requestedHeaders = req.Headers["Access-Control-Request-Headers"];
                        if (!string.IsNullOrEmpty(requestedHeaders))
                        {
                            res.Headers["Access-Control-Allow-Headers"] = requestedHeaders;
                        }
                    }
                    else
                    {
                        res.Headers["Access-Control-Expose-Headers"] = "Cache-Control,Content-Encoding,Content-Length,Content-Type,Date,Expires,Pragma,Server,Vary";
                    }
                }
                else
                {
                    CorsLogger.Trace("Attempt to access API from non-whitelisted domain: " + origin);
                }
            }
        }

        protected void Application_BeginRequest(object sender, EventArgs e)
        {
            var request = HttpContext.Current.Request;
            var response = HttpContext.Current.Response;

            WriteCors(request, response);

            if (Request.ContentLength > MaxRequestLength || Request.ContentLength > MaxContentLength)
            {
                WriteRequestTooBig(request, response);
                response.End();
            }
            
            if (request.HttpMethod == "OPTIONS")
            {
                response.StatusCode = 200;
                response.End();
            }
        }

        protected static void WriteRequestTooBig(HttpRequest request, HttpResponse response)
        {
            response.ContentType = "application/json";
            response.StatusCode = (int)HttpStatusCode.RequestEntityTooLarge;
            response.Write(JsonConvert.SerializeObject(new FileUploadException(request.ContentLength, MaxContentLength).ToResponse()));  
        }

        protected void Application_EndRequest(object sender, EventArgs e)
        {
            var request = HttpContext.Current.Request;
            var response = HttpContext.Current.Response;

            if ((request.HttpMethod == "POST") && (response.StatusCode == 404 && response.SubStatusCode == 13))
            {              
                response.ClearHeaders();
                response.ClearContent();                
                WriteCors(request, response);
                WriteRequestTooBig(request, response);
            }
        }
    }
}
