﻿using System;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Sockets;
using System.Web;
using System.Web.Http;
using Curse.Friends.Data;
using Curse.Friends.MicroService.Exceptions;
using Curse.Friends.MicroService.Filters;
using Curse.CloudServices.Models;
using System.ServiceModel.Channels;
using Curse.Friends.MicroService.Contracts;

namespace Curse.Friends.MicroService
{
    [MicroServiceExceptionFilter]
    [AuthenticationFilter(true, AuthenticationLevel.LoggedIn, false)]
    public abstract class MicroServiceController : ApiController
    {
        protected IHttpActionResult Forbidden()
        {
            return StatusCode(HttpStatusCode.Forbidden);
        }

        protected IHttpActionResult Forbidden(string message)
        {
            var response = Request.CreateResponse(HttpStatusCode.Forbidden, message);
            return ResponseMessage(response);
        }

        protected IHttpActionResult Forbidden<T>(T content)
        {
            var response = Request.CreateResponse(HttpStatusCode.Forbidden, content);
            return ResponseMessage(response);
        }

        protected IHttpActionResult TooManyRequests()
        {
            return StatusCode((HttpStatusCode)429);
        }

        protected IHttpActionResult ServiceUnavailable()
        {
            return StatusCode(HttpStatusCode.ServiceUnavailable);
        }

        protected IHttpActionResult BadRequest(object content)
        {
            var response = Request.CreateResponse(HttpStatusCode.BadRequest, content);
            return ResponseMessage(response);
        }

        protected IHttpActionResult Unauthorized(object content)
        {
            var response = Request.CreateResponse(HttpStatusCode.Unauthorized, content);
            return ResponseMessage(response);
        }

        protected IHttpActionResult NotFound(object content)
        {
            var response = Request.CreateResponse(HttpStatusCode.NotFound, content);
            return ResponseMessage(response);
        }

        protected IHttpActionResult StatusCode(HttpStatusCode statusCode, object content)
        {
            var response = Request.CreateResponse(statusCode, content);
            return ResponseMessage(response);
        }

        protected IHttpActionResult ErrorResponse(HttpStatusCode statusCode, string errorType, string errorMessage=null, object data = null)
        {
            var response = new ErrorResponse(statusCode, errorType, errorMessage, data);
            var message = new HttpResponseMessage(statusCode)
            {
                Content = response.ToHttpContent(Request, Configuration)
            };
            return ResponseMessage(message);
        }

        private AuthenticationToken _token;

        protected AuthenticationToken Token
        {
            get
            {
                if (_token == null)
                {
                    _token = AuthenticationContext.GetTokenFromRequest(Request);
                }
                return _token;
            }
        }

        protected virtual int MaxFileSize
        {
            get { return 10240; }
        }

        protected HttpPostedFile GetPostedFile()
        {
            if (HttpContext.Current.Request.ContentLength > MaxFileSize)
            {
                throw new FileUploadException(HttpContext.Current.Request.ContentLength, MaxFileSize);
            }

            var files = HttpContext.Current.Request.Files;

            if (files.Count != 1)
            {
                throw new FileUploadException(FileUploadFailureReason.Missing);
            }

            var file = files[0];

            if (file.ContentLength > MaxFileSize)
            {
                throw new FileUploadException(file.ContentLength, MaxFileSize);
            }

            return file;
        }

        protected UserRegionInfo GetCurrentUserAndRegion(bool createRegion = false)
        {
            if (_currentUserAndRegion != null)
            {
                return _currentUserAndRegion;
            }

            _currentUserAndRegion =  GetUserAndRegion(Token.UserID, createRegion);
            return _currentUserAndRegion;
        }

        private UserRegionInfo _currentUserAndRegion;

        protected UserRegionInfo GetUserAndRegion(int userID, bool createRegion = false)
        {           
            // See if the user has registered before
            var userRegion = UserRegion.GetLocal(userID);

            // If not, register them to the local storage node
            if (userRegion == null)
            {
                if (!createRegion)
                {
                    throw new UserNotFoundException(userID);
                }

                userRegion = new UserRegion { UserID = userID, RegionID = UserRegion.LocalConfigID };
                userRegion.InsertLocal();
                userRegion = UserRegion.GetLocal(userID);
                if (userRegion == null)
                {
                    throw new InvalidOperationException("Failed to read back UserRegion after inserting it!");
                }
            }

            // From their storage region, get their record and update their status            
            var user = Curse.Friends.Data.User.Get(userRegion.RegionID, userID);

            if (user == null && !createRegion)
            {
                throw new UserNotFoundException(userID);
            }
            
            return new UserRegionInfo(user, userRegion);            
        }


        private static class IpAddressHelper
        {
            private const string HttpContext = "MS_HttpContext";
            private const string RemoteEndpointMessage = "System.ServiceModel.Channels.RemoteEndpointMessageProperty";
            private const string OwinContext = "MS_OwinContext";

            internal static string GetClientIp(HttpRequestMessage request)
            {
                // Web-hosting
                if (request.Properties.ContainsKey(HttpContext))
                {
                    var ctx = (HttpContextWrapper)request.Properties[HttpContext];
                    if (ctx != null)
                    {
                        return ctx.Request.UserHostAddress;
                    }
                }

                // Self-hosting
                if (request.Properties.ContainsKey(RemoteEndpointMessage))
                {
                    var remoteEndpoint = (RemoteEndpointMessageProperty)request.Properties[RemoteEndpointMessage];
                    if (remoteEndpoint != null)
                    {
                        return remoteEndpoint.Address;
                    }
                }

                // Self-hosting using Owin
                if (request.Properties.ContainsKey(OwinContext))
                {
                    dynamic owinContext = request.Properties[OwinContext];
                    if (owinContext != null)
                    {
                        return owinContext.Request.RemoteIpAddress;
                    }
                }

                return null;
            }
        }

        protected IPAddress GetCurrentIpAddress()
        {
            const string cloudFlareIpHeader = "CF-Connecting-IP";
            const string forwardedIpHeader = "X-Forwarded-For";
            string ipAddress = null;

            if (Request.Headers.Contains(cloudFlareIpHeader))
            {
                ipAddress = Request.Headers.GetValues(cloudFlareIpHeader).FirstOrDefault();
            }
            else if (Request.Headers.Contains(forwardedIpHeader))
            {
                ipAddress = Request.Headers.GetValues(forwardedIpHeader).FirstOrDefault();
            }
            else if (Request.Headers.Contains(forwardedIpHeader))
            {
                ipAddress = Request.Headers.GetValues(forwardedIpHeader).FirstOrDefault();
            }
            else
            {
                ipAddress = IpAddressHelper.GetClientIp(Request);
            }

            return GetClientIpAddress(ipAddress);
        }

        private static IPAddress GetClientIpAddress(string ipString, bool silentFailure = true)
        {
            if (ipString.Contains(","))
            {
                var ipArray = ipString.Split(',');
                ipString = ipArray[ipArray.Length - 1].Trim();
            }

            IPAddress ipAddress;
            if (IPAddress.TryParse(ipString, out ipAddress))
            {
                var family = ipAddress.AddressFamily;
                if (family == AddressFamily.InterNetwork || family == AddressFamily.InterNetworkV6)
                {
                    var bytes = ipAddress.GetAddressBytes();
                    if (bytes.Length > 0)
                    {
                        return ipAddress;
                    }
                }
            }
            if (silentFailure)
            {
                return IPAddress.None;
            }
            else
            {
                throw new InvalidOperationException(string.Format("Cannot parse IPAddress '{0}'", ipString));
            }
        }
    }
}
