﻿using System.Collections.Generic;
using System.Diagnostics;
using System.Net;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Statistics;
using Curse.Logging;
using Curse.SocketInterface;
using System;
using System.Collections.Concurrent;
using System.Linq;
using System.Security.Cryptography;
using System.Threading;
using Curse.Extensions;
using Curse.Friends.Configuration;

namespace Curse.Friends.NotificationProcessing
{
    public static class NotificationInstance
    {
        private static readonly ConcurrentDictionary<string, NotificationSession> _sessionsBySessionID = new ConcurrentDictionary<string, NotificationSession>();

        public static int SessionCount
        {
            get { return _sessionsBySessionID.Count; }
        }

        public static NotificationSession GetSession(string sessionID)
        {
            NotificationSession value;
            
            if(_sessionsBySessionID.TryGetValue(sessionID, out value))
            {
                return value;
            }

            return null;
        }

        public static void ClientDisconnected(NotificationSession disconnectedSession)
        {                       
            // Remove this session from our list
            NotificationSession removedSession;
            if (!_sessionsBySessionID.TryRemove(disconnectedSession.SessionID, out removedSession))
            {
                return;
            }

            removedSession.Dispose();                     
            FriendsStatsManager.Current.TrackUserDisconnected(_sessionsBySessionID.Count);
        }

        private static DateTime _lastLoggedSessionReconect = DateTime.UtcNow.AddMinutes(-10);

        public static NotificationSession CreateSession(ISocketInterface socketInterface, User user, UserConnectionStatus desiredStatus, string machineKey, string sessionID, int userRegionID, Version clientVersion, SymmetricAlgorithm encryptionAlgorithm, IPAddress remoteEndPoint, out Dictionary<string, long> timers)
        {            

            // Timers for logging slow notification session joins
            timers = new Dictionary<string, long>();
            var sw = Stopwatch.StartNew();
            
            // Get the endpoint using the user and machine key            
            var endpoint = ClientEndpoint.GetLocal(user.UserID, machineKey);
            timers["GetClientEndpoint"] = sw.ElapsedMilliseconds;
            sw.Restart();

            // No endpoint found
            if(endpoint == null)
            {
                return null;
            }

            // Ensure session ID is valid
            if(!endpoint.SessionID.Equals(sessionID))
            {
                return null;
            }            

            // See if this session is already connected. If so, disconnect it.
            NotificationSession oldSession;

            if(_sessionsBySessionID.TryGetValue(sessionID, out oldSession))
            {
                if (DateTime.UtcNow - _lastLoggedSessionReconect >= TimeSpan.FromMinutes(5))
                {
                    _lastLoggedSessionReconect = DateTime.UtcNow;
                    Logger.Trace("A session has re-connected. Disconnecting old session!", new { endpoint, sessionID, OldSession = oldSession.Timestamp, oldSession.DateUpdatedEndpoint });                    
                }

                oldSession.RaiseDisconnect(SocketDisconnectReason.Dispose);                
            }


            // Create the session object
            var session = new NotificationSession(socketInterface, sessionID, user.UserID, user.Username, machineKey, userRegionID, encryptionAlgorithm, endpoint.Platform, clientVersion);
            
            // Set the endpoint as connected
            ClientEndpoint.SetConnected(ClientEndpoint.LocalConfigID, Environment.MachineName, user.UserID, userRegionID, machineKey, session.ConnectionID, session.Timestamp, clientVersion, remoteEndPoint ?? IPAddress.None);
            timers["ClientEndpointSetConnected"] = sw.ElapsedMilliseconds;
            
            _sessionsBySessionID.TryAdd(sessionID, session);            

            // Track the stats for reporting
            FriendsStatsManager.Current.TrackUserConnected(user.UserID, _sessionsBySessionID.Count);

            return session;
        }
        
        public static void BeginShutdown()
        {
            // Stop processing new requests or accepting new ones

            RequestProcessing.IsStarted = false;

            // Mark all endpoints as disconnected           
            MarkEndpointsDisconnected();

            // Drain the connections over a period of time
            new Thread(DrainConnections) { IsBackground = true }.Start();
        }

        public static void MarkEndpointsDisconnected()
        {            
            var currentSessions = _sessionsBySessionID.Values.ToArray();
            Logger.Info("Marking " + currentSessions.Length + " endpoints as disconnected!");

            foreach (var session in currentSessions)
            {
                try
                {
                    session.MarkEndpointDisconnected();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to mark endpoint disconnected.");
                }
            }

            Logger.Info("Endpoints have been marked as disconnected.");
        }

        private static volatile bool _isDrainingConnections = false;

        public static void DrainConnections()
        {
            if (_isDrainingConnections)
            {
                Logger.Warn("Attempt to drain connections, while an existing drain is in progress.");
                return;
            }

            _isDrainingConnections = true;
            // Disconnect 500 connections every second
            var currentSessions = _sessionsBySessionID.Values.ToArray();

            Logger.Info("Beginning connection daining of " + currentSessions.Length + " endpoints...");

            foreach (var set in currentSessions.InSetsOf(500))
            {
                Logger.Info("Draining next set containing " + set.Count + " endpoints...");
                foreach (var session in set)
                {
                    try
                    {
                        session.RaiseDisconnect();
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, "Failed to drain connection");
                    }                    
                }

                Thread.Sleep(1000);
            }

            _isDrainingConnections = false;
            Logger.Info("All connnections have been drained!");

        }

        public static void DisconnectAll()
        {
            Logger.Info("Disconnecting " + _sessionsBySessionID.Count + " sessions...");

            foreach(var session in _sessionsBySessionID.Values)
            {
                try
                {
                    session.RaiseDisconnect();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to disconnect session during shutdown.");
                }                
            }

            Logger.Info("Done disconnecting sessions!");
        }

    }
}
