﻿using Curse.SocketInterface;
using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Logging;
using System.Security.Cryptography;
using Curse.Aerospike;

namespace Curse.Friends.NotificationProcessing
{
    public class NotificationSession : ISocketSession
    {

        private static readonly LogCategory Logger = new LogCategory("NotificationSession") { ReleaseLevel = LogLevel.Info };
        private static readonly Random _random = new Random();

        public NotificationSession(ISocketInterface serverSocket, string sessionID, int userID, string username, string machineKey, int regionID, SymmetricAlgorithm encryptionAlgorithm, DevicePlatform devicePlatform, Version clientVersion)
        {            
            ServerSocket = serverSocket;
            SessionID = sessionID;
            UserID = userID;
            Username = username;
            MachineKey = machineKey;
            Timestamp = DateTime.UtcNow;
            ConnectionID = Guid.NewGuid().ToString();
            RegionID = regionID;
            EncryptionAlgorithm = encryptionAlgorithm;
            _endpointUpdateFrequency = TimeSpan.FromSeconds(_random.Next(240, 300));
            DevicePlatform = devicePlatform;
            IsLegacyClient = clientVersion.Major < 7;
        }

        public void Dispose()
        {
            if (EncryptionAlgorithm != null)
            {
                try
                {
                    EncryptionAlgorithm.Dispose();
                }
                catch { }
            }
        }

        public bool IsDisconnecting { get; set; }
        public ISocketInterface ServerSocket { get; private set; }
        public string SessionID { get; private set; }
        public string ConnectionID { get; private set; }
        public int RegionID { get; private set; }
        public int UserID { get; private set; }
        public string Username { get; private set; }
        public string MachineKey { get; private set; }
        public DateTime Timestamp { get; private set; }


        public TimeSpan Age
        {
            get { return DateTime.UtcNow.Subtract(Timestamp); }
        }

        /// <summary>
        /// Indicates that the session is at least a minute old.
        /// </summary>
        public bool HasStabilized
        {
            get { return Age >= TimeSpan.FromMinutes(1); }
        }

        public SymmetricAlgorithm EncryptionAlgorithm { get; private set; }
        private readonly TimeSpan _endpointUpdateFrequency;

        public DateTime DateUpdatedEndpoint { get; private set; }
        public DevicePlatform DevicePlatform { get; private set; }

        private DateTime? _lastMessageSent = null;

        public bool IsLegacyClient { get; private set; }

        public bool IsFlooding()
        {
            if (!_lastMessageSent.HasValue)
            {
                _lastMessageSent = DateTime.UtcNow;
                return false;
            }

            if ((DateTime.UtcNow - _lastMessageSent.Value).TotalMilliseconds < 100)
            {
                return true;
            }

            _lastMessageSent = DateTime.UtcNow;
            return false;
        }

        public void RaiseDisconnect(SocketDisconnectReason reason = SocketDisconnectReason.NotAuthenticated)
        {
            Logger.Trace("Disconnecting session", new { SessionID, UserID, Timestamp, MachineKey });
            ServerSocket.RaiseDisconnect(reason);            
        }

        void ISocketSession.Disconnect()
        {
            IsDisconnecting = true;

            try
            {
                NotificationInstance.ClientDisconnected(this);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to disconnect session.");
            }

            // If the server is shutting down, don't bother to resolve anything. Assume the client will reconnect to another node.
            if (!RequestProcessing.IsStarted)
            {
                return;
            }

            try
            {
                string reason;
                ClientEndpoint model;
                if (!ClientEndpoint.SetDisconnected(UserID, RegionID, MachineKey, SessionID, ConnectionID, out model, out reason))
                {
                    Logger.Trace("Endpoint will not be marked as disconnected. " + reason);
                    return;
                }
                UserPresenceResolver.CreateAvailabilityResolver(UserID, RegionID, MachineKey);

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to queue user status resolver.");
            }
        }

        public void MarkEndpointDisconnected()
        {
            try
            {
                ClientEndpoint.UpdateByLocalKey(UpdateMode.Fast, new Dictionary<string, object>() { { "IsConnected", false } }, UserID, MachineKey);            
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to mark endpoint as disconnected");
            }
            
        }

        public ClientEndpoint GetClientEndpoint()
        {
            return ClientEndpoint.GetLocal(UserID, MachineKey);
        }

        public void UpdateEndpoint()
        {
            // Do not do anything until the connection has stabilizied.
            if (!HasStabilized)
            {
                return;
            }

            if (DateTime.UtcNow - DateUpdatedEndpoint < _endpointUpdateFrequency)
            {
                return;
            }

            try
            {
                var endpoint = ClientEndpoint.GetLocal(UserID, MachineKey);

                if (endpoint == null)
                {
                    Logger.Warn("Attempt to send heartbeat without an endpoint.");
                    RaiseDisconnect();
                    return;
                }

                if (!endpoint.IsConnected)
                {
                    Logger.Warn("Attempt to send heartbeat from an endpoint which is disconnected.", new { SessionDate = Timestamp, UserID, endpoint });
                    RaiseDisconnect();
                    return;
                }
                
                var user = User.Get(RegionID, UserID);                

                if (user.ConnectionStatus == UserConnectionStatus.Offline)
                {
                    Logger.Warn("Attempt to send heartbeat from user that is considered offline. Disconnecting user!", new { ConnectionID, SessionID, Timestamp, endpoint, user });
                    RaiseDisconnect();
                    return;
                }

                if (endpoint.ConnectionID != ConnectionID)
                {
                    Logger.Debug("Attempt to send heartbeat from a client with a ConnectionID which differs from the database!", new { ConnectionID, SessionID, Timestamp, endpoint, user });
                    RaiseDisconnect();
                    return;
                }

                // Ensure that the user data is consistent with the endpoint
                var isConsistent = true;
                string consistencyReason = null;

                if (user.ConnectionStatus == UserConnectionStatus.Offline)
                {
                    isConsistent = false;
                    consistencyReason = "User status: " + user.ConnectionStatus;
                }
                else if (!endpoint.IsConnected)
                {
                    isConsistent = false;
                    consistencyReason = "Endpoint not connected";
                }
                else if (endpoint.ServerName != Environment.MachineName)
                {
                    isConsistent = false;
                    consistencyReason = "Endpoint server name: " + endpoint.ServerName;
                }

                if (isConsistent)
                {
                    // Only update client endpoint heartbeat date every 12 hours. This is to avoid XDR lag.
                    if (DateTime.UtcNow - DateUpdatedEndpoint > TimeSpan.FromHours(12))
                    {
                        endpoint.HeartbeatDate = DateTime.UtcNow;
                        endpoint.Update(p => p.HeartbeatDate);
                    }
                }
                else
                {
                    Logger.Warn("Heartbeat received on an endpoint that is inconsistent. Reason: " + consistencyReason, new { ConnectionID, Timestamp, MachineKey, SessionID, user, endpoint });

                    endpoint.IsConnected = true;
                    endpoint.HeartbeatDate = DateTime.UtcNow;
                    endpoint.ServerName = Environment.MachineName;
                    endpoint.Update(p => p.IsConnected, p => p.HeartbeatDate, p => p.ServerName);

                    UserPresenceResolver.CreateAvailabilityResolver(UserID, RegionID, MachineKey);
                    return;
                }

                // Queue off a friend suggestion work, once a day
                if ((DateTime.UtcNow - user.LastFriendsSuggestion).TotalDays >= 1)
                {
                    user.LastFriendsSuggestion = DateTime.UtcNow;
                    user.Update(p => p.LastFriendsSuggestion);

                    new FriendshipSuggestionWorker { UserID = user.UserID }.Enqueue();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                DateUpdatedEndpoint = DateTime.UtcNow;
            }
        }


        public void SendContract(IContract contract)
        {
            var socket = ServerSocket;
            if (socket != null)
            {
                socket.SendContract(contract);
            }
        }

        public void SendString(string serializedMessage)
        {
            var socket = ServerSocket;
            if (socket != null)
            {
                socket.SendString(serializedMessage);
            }
        }
    }
}
