﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.NotificationContracts;
using Curse.Logging;
using System.Diagnostics;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Curse.SocketInterface;

namespace Curse.Friends.NotificationProcessing
{
    public static class QueueProcessors
    {

        public static void StartQueueProcessors()
        {
            var sw = Stopwatch.StartNew();

            // Friend Notifications            
            FriendSuggestionNotifier.StartProcessor(FriendSuggestionNotifier_ProcessMessage);

            // User and Account Notifications
            UserClientSettingsNotifier.StartProcessor(UserClientSettingsNotifier_ProcessMessage);

            // Conversation Notifications            
            ConversationReadNotifier.StartProcessor(ConversationReadNotifier_ProcessMessage);
            FriendMessageNotifier.StartProcessor(FriendMessageNotifier_ProcessMessage);
            ChatMessageResponseNotifier.StartProcessor(ChatMessageResponseNotifier_ProcessMessage);

            // Group Notifications
            GroupPreferenceChangedNotifier.StartProcessor(GroupPreferenceChangedNotifier_ProcessMessage);
            GroupMultiSessionMessageNotifier.StartProcessor(GroupMultiSessionMessageNotifier_ProcessMessage);
            GroupMultiSessionChangeNotifier.StartProcessor(GroupMultiSessionChangeNotifier_ProcessMessage);
            GroupPollMultiSessionChangedNotifier.StartProcessor(GroupPollMultiSessionChangedNotifier_ProcessMessage);
            GroupGiveawayMultiSessionNotifier.StartProcessor(GroupGiveawayMultiSessionNotifier_ProcessMessage);            

            // Call Notifications
            CallNotifier.StartProcessor(CallNotifier_ProcessMessage);
            CallRespondedNotifier.StartProcessor(CallRespondedNotifier_ProcessMessage);

            ClientEndpointResolver.Initialize();
            UserStatusResolver.Initialize();

            FriendshipSuggestionWorker.Initialize();

            ExternalCommunityLinkChangedMultiSessionNotifier.StartProcessor(ExternalCommunityLinkChangedMultiSessionNotifier_ProcessMessage);

            // Simplified
            MultiSessionNotifier.StartProcessor(MultiSessionNotifier_ProcessMessage);
            SingleSessionNotifier.StartProcessor(SingleSessionNotifier_ProcessMessage);

            Logger.Info("Queue processors initialized in " + sw.Elapsed.TotalSeconds.ToString("###,##0.00") + " seconds");
        }


        #region User Notifications


        private static void UserClientSettingsNotifier_ProcessMessage(UserClientSettingsNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);

            if (session == null)
            {
                return;
            }
            session.SendContract(e.Notification);
        }

        #endregion

        #region Friendship Notifications

        private static void FriendSuggestionNotifier_ProcessMessage(FriendSuggestionNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);

            if (session == null)
            {
                return;
            }

            var suggestions = FriendSuggestion.ToValidNotifications(e.ClientUserID, e.SuggestedFriends);
            session.SendContract(new FriendSuggestionNotification { Suggestions = suggestions });
        }

        #endregion

        #region Calls

        private static void CallNotifier_ProcessMessage(CallNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);

            if (session == null)
            {
                return;
            }

            if (session.IsLegacyClient)
            {
                session.SendContract(new GroupVoiceInvitationNotification
                {
                    GroupID = new Guid(e.Notification.ConversationID),
                    InviteUrl = e.Notification.InviteUrl,
                    RequestorID = e.Notification.SenderID,
                    TimeStamp = e.Notification.Timestamp
                });
            }
            else
            {
                session.SendContract(e.Notification);
            }
        }

        private static void CallRespondedNotifier_ProcessMessage(CallRespondedNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);

            if (session == null)
            {
                return;
            }

            session.SendContract(e.Notification);
        }

        #endregion

        #region Conversations

        private static void FriendMessageNotifier_ProcessMessage(FriendMessageNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);

            if (session == null)
            {
                return;
            }

            session.SendContract(e.Notification);
        }

        private static void ConversationReadNotifier_ProcessMessage(ConversationReadNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);

            if (session == null)
            {
                return;
            }

            session.SendContract(e.Notification);
        }

        private static void ChatMessageResponseNotifier_ProcessMessage(ChatMessageResponseNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);

            if (session == null)
            {
                return;
            }

            session.SendContract(e.Notification);

        }

        #endregion

        #region Groups

        private static void GroupPreferenceChangedNotifier_ProcessMessage(GroupPreferenceChangedNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);
            if (session == null)
            {
                return;
            }

            session.SendContract(e.Notification);
        }

        private static void GroupMultiSessionMessageNotifier_ProcessMessage(GroupMultiSessionMessageNotifier e)
        {
            DispatchNotification(e.SessionIDs, session =>
            {
                if (session.IsLegacyClient)
                {
                    session.SendContract(new GroupMessageNotification
                    {
                        ClientMessageID = Guid.Parse(e.Message.ClientID),
                        GroupID = Guid.Parse(e.Message.ConversationID),
                        Timestamp = e.Message.Timestamp.FromEpochMilliconds(),
                        SenderID = e.Message.SenderID,
                        Message = e.Message.Body,
                        ServerMessageID = new Guid(e.Message.ServerID)
                    });
                }
                else
                {
                    session.SendContract(e.Message);
                }

            });
        }

        private static void GroupMultiSessionChangeNotifier_ProcessMessage(GroupMultiSessionChangeNotifier e)
        {
            DispatchNotification(e.SessionIDs, session =>
            {
                if (session.IsLegacyClient)
                {
                    if (e.Notification.Group.GroupType == GroupType.Normal)
                    {
                        // Create the legacy contract
                        var legacyNotification = new LegacyGroupChangeNotification
                        {
                            ChangeType = e.Notification.ChangeType,
                            SenderID = e.Notification.SenderID,

                            Group = new LegacyGroupNotification
                            {
                                GroupID = e.Notification.Group.GroupID,
                                Status = e.Notification.Group.Status,
                                ForcePushToTalk = e.Notification.Group.ForcePushToTalk,
                                GroupTitle = e.Notification.Group.GroupTitle,
                                ParentGroupID = e.Notification.Group.ParentGroupID,
                                GroupType = e.Notification.Group.GroupType,
                                MetaDataOnly = false,
                                RootGroupID = e.Notification.Group.RootGroupID,
                                VoiceSessionCode = e.Notification.Group.VoiceSessionCode
                            },

                            Members = e.Notification.Members.Select(p => new GroupMemberNotification
                            {
                                UserID = p.UserID,
                                RegionID = 1, // Will be replaced
                                Role = p.BestRole == 1 ? LegacyGroupRole.Owner : LegacyGroupRole.Member,
                                CurrentGroupID = Guid.Empty,
                                Username = p.Username
                            }).ToArray()
                        };

                        session.SendContract(legacyNotification);
                    }

                }
                else
                {
                    session.SendContract(e.Notification);
                }

            });
        }

        private static void GroupPollMultiSessionChangedNotifier_ProcessMessage(GroupPollMultiSessionChangedNotifier e)
        {
            DispatchNotification(e.SessionIDs, session =>
            {
                session.SendContract(e.Notification);
            });
        }

        private static void GroupGiveawayMultiSessionNotifier_ProcessMessage(GroupGiveawayMultiSessionNotifier e)
        {
            DispatchNotification(e.SessionIDs, session =>
            {
                session.SendContract(e.Notification);
            });
        }

        private static void ExternalCommunityLinkChangedMultiSessionNotifier_ProcessMessage(ExternalCommunityLinkChangedMultiSessionNotifier e)
        {
            DispatchNotification(e.SessionIDs, session =>
            {
                session.SendContract(e.Notification);
            });
        }

        private static void MultiSessionNotifier_ProcessMessage(MultiSessionNotifier e)
        {
            DispatchNotification(e.SessionIDs, session =>
            {
                session.SendString(e.SerializedMessage);
            });
        }

        private static void SingleSessionNotifier_ProcessMessage(SingleSessionNotifier e)
        {
            var session = GetSession(e.ClientMachineKey, e.ClientUserID, e.ClientSessionID);
            if (session == null)
            {
                return;
            }

            session.SendString(e.SerializedMessage);
        }

        #endregion

        private static void DispatchNotification(HashSet<string> sessionIDs, Action<NotificationSession> action)
        {
            foreach (var sessionID in sessionIDs)
            {
                var session = GetSession(sessionID);
                if (session == null)
                {
                    continue;
                }

                try
                {
                    action(session);
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Failed to dispatch notification");
                }

            }
        }

        private static NotificationSession GetSession(string sessionID)
        {
            return NotificationInstance.GetSession(sessionID);
        }

        private static NotificationSession GetSession(string machineKey, int userID, string sessionID)
        {
            var session = NotificationInstance.GetSession(sessionID);

            if (session != null)
            {
                return session;
            }

            // Service is shutting down, so don't try to resolve the disconnect
            if (!RequestProcessing.IsStarted)
            {
                return null;
            }

            // Session is null. We need to mark this endpoint as offline
            var endpoint = ClientEndpoint.GetLocal(userID, machineKey);

            if (endpoint != null && endpoint.IsConnected && endpoint.ServerName.Equals(Environment.MachineName, StringComparison.InvariantCultureIgnoreCase))
            {
                Logger.Warn("Notification session could not be found for an endpoint. The endpoint will be marked as disconnected.", new { machineKey, userID, sessionID, endpoint });
                UserPresenceResolver.CreateDisconnectResolver(endpoint);
            }

            return null;
        }

    }
}