﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.SocketMessages;
using System.Security.Cryptography;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Models;
using Curse.Friends.Statistics;

namespace Curse.Friends.NotificationProcessing
{
    public static class RequestProcessing
    {
        // This needs to be updated if anything is updated or if an older client version is found to have a security vulnerability
        private static readonly Version MinimumClientVersion = new Version(6, 0, 5000, 0);

        public static bool IsStarted { get; set; }

        public static void Start()
        {
            IsStarted = true;
        }

        public static void Stop()
        {
            IsStarted = false;
        }

        public static void RegisterDispatchers()
        {
            ContractDispatcher.AddContractDispatcher<JoinRequest>(OnJoinRequest);
            ContractDispatcher.AddContractDispatcher<HealthCheckRequest>(OnHealthCheckRequest);
            ContractDispatcher.AddContractDispatcher<ConversationMessageRequest>(OnConversationMessageRequest);
            ContractDispatcher.AddContractDispatcher<Handshake>(OnHandshake);
            ContractDispatcher.AddContractDispatcher<LegacyConversationMarkReadRequest>(OnConversationMarkReadRequest);            
        }
        

        public static void OnHandshake(ISocketInterface socketInterface, Handshake handshake)
        {
            try
            {
                if (!IsStarted)
                {                    
                    return;
                }

                socketInterface.IsHandshaken = true;
                socketInterface.DateLastHandshake = Environment.TickCount;

                var session = (NotificationSession)socketInterface.Session;

                if (session == null)
                {
                    Logger.Warn("Attempt send heartbeat without a session");
                    return;
                }

                if (session.IsDisconnecting)
                {
                    Logger.Warn("Attempt send heartbeat on a session that is disconnecting.");
                    return;
                }

                session.UpdateEndpoint();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                socketInterface.SendContract(handshake);
            }

        }

        public static void OnJoinRequest(ISocketInterface socketInterface, JoinRequest joinRequest)
        {
            var timers = new Dictionary<string, long>();
            Dictionary<string, long> sessionTimers = null;
            var sw = Stopwatch.StartNew();
            var resp = new JoinResponse()
            {
                Status = JoinStatus.FailedUnhandledException,
                ConnectionStatus = UserConnectionStatus.Offline,
            };

            try
            {

                if (!IsStarted)
                {
                    resp.Status = JoinStatus.Timeout;
                    return;
                }

                Version version = null;

                // Check their client version
                if (!Version.TryParse(joinRequest.ClientVersion, out version) || version < MinimumClientVersion)
                {
                    resp.Status = JoinStatus.InvalidClientVersion;
                    return;
                }

                if (joinRequest.UserID <= 0)
                {
                    resp.Status = JoinStatus.InvalidSessionID;
                    return;
                }

                var isw = Stopwatch.StartNew();
                var userRegion = UserRegion.GetLocal(joinRequest.UserID);
                timers["UserRegion"] = isw.ElapsedMilliseconds;
                isw.Restart();

                if (userRegion == null)
                {
                    Logger.Warn("Attempt to join without a UserRegion record.", joinRequest);
                    resp.Status = JoinStatus.InvalidSessionID;
                    return;
                }

                User user = null;
                try
                {
                    user = userRegion.GetUser();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error getting User from the UserRegion", new { userRegion, joinRequest });
                    throw;
                }

                timers["User"] = isw.ElapsedMilliseconds;
                isw.Restart();

                if (user == null)
                {
                    Logger.Warn("Attempt to join without a User record.", joinRequest);
                    resp.Status = JoinStatus.InvalidSessionID;
                    return;
                }

                AesCryptoServiceProvider encryptionAlgorithm = null;
                if (joinRequest.PublicKey != null)
                {
                    encryptionAlgorithm = new AesCryptoServiceProvider();
                    resp.EncryptedSessionKey = CryptoHelper.CreateSecret(encryptionAlgorithm, joinRequest.PublicKey);
                }

                timers["EncryptedSessionKey"] = isw.ElapsedMilliseconds;
                isw.Restart();

                // Inherit the user's last connection status, if it is a valid one
                var newStatus = user.LastConnectionStatus != UserConnectionStatus.Offline && user.LastConnectionStatus != UserConnectionStatus.Idle ? user.LastConnectionStatus : UserConnectionStatus.Online;

                // If the user's connection status is changing, or their capabilities will likely change, resolve their new status and capabilities 
                var shouldResolveStatus = user.ConnectionStatus != newStatus || user.Capabilities != ClientCapability.All;

                socketInterface.Session = NotificationInstance.CreateSession(socketInterface, user, newStatus, joinRequest.MachineKey, joinRequest.SessionID, userRegion.RegionID, version, encryptionAlgorithm, socketInterface.RemoteAddress, out sessionTimers);

                timers["Session"] = isw.ElapsedMilliseconds;
                isw.Restart();
                
                if (socketInterface.Session == null)
                {
                    resp.Status = JoinStatus.InvalidSessionID;
                    return;
                }

                // Only kick off a status resolver if the user's status has changed
                if (shouldResolveStatus)
                {
                    Logger.Debug("Creating status change resolver", new { newStatus, user });
                    UserPresenceResolver.CreateAvailabilityResolver(userRegion.UserID, userRegion.RegionID, joinRequest.MachineKey);
                    timers["CreateStatusChangeResolver"] = isw.ElapsedMilliseconds;
                    isw.Restart();
                }
                else
                {
                    Logger.Debug("Suppressing status change on join", new { newStatus, user });
                }

                resp.ConnectionStatus = newStatus;
                socketInterface.IsAuthenticated = true;
                resp.Status = JoinStatus.Successful;
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                resp.ServerTime = DateTime.UtcNow;
                socketInterface.SendContract(resp);

                sw.Stop();
                if (sw.ElapsedMilliseconds > 1000)
                {
                    Logger.Warn("JoinRequest took " + sw.ElapsedMilliseconds + " ms!", new { timers, sessionTimers });
                }
            }

        }

        public static void OnConversationMarkReadRequest(ISocketInterface socketInterface, LegacyConversationMarkReadRequest request)
        {
            var session = (NotificationSession)socketInterface.Session;

            if (session == null)
            {
                Logger.Warn("Attempt to send without a session");
                return;
            }

            try
            {
                ConversationReadWorker.Create(session.UserID, request.ConversationID, request.Timestamp, string.Empty, string.Empty, false);
            }
            catch (Exception ex)
            {

                Logger.Error(ex, "Failed to mark a conversation read.");
            }
        }

        public static void OnHealthCheckRequest(ISocketInterface socketInterface, HealthCheckRequest request)
        {
            if (request.ApiKey != FriendsServiceConfiguration.Instance.NotificationServiceApiKey)
            {
                Logger.Warn("Attempt to access an API protected endpoint.");
                return;
            }
        }

        [Obsolete]
        public static void OnConversationMessageRequest(ISocketInterface socketInterface, ConversationMessageRequest request)
        {
            var session = (NotificationSession)socketInterface.Session;

            if (session == null)
            {
                Logger.Warn("Attempt to send without a session");
                return;
            }

            var response = new ConversationMessageResponse
            {
                ClientID = request.ClientID,
                ConversationID = request.ConversationID,
                Status = DeliveryStatus.Successful
            };

            try
            {
                // Check if the user is flooding. This only allows one message very 100ms (prevent botting, etc)
                if (session.IsFlooding())
                {
                    response.Status = DeliveryStatus.Throttled;
                    return;
                }

                // Check that the message length is valid
                if (!request.Message.SafeRange(1, ConversationConstants.MaxMessageLength))
                {
                    response.Status = DeliveryStatus.Throttled;
                    return;
                }
                
                // Process any included attachments
                Attachment attachment = null;

                if (request.AttachmentID != Guid.Empty)
                {
                    // First try to get the attachment
                    var regionID = ConfigurationRegion.AllRegionIDs.Contains(request.AttachmentRegionID) ? request.AttachmentRegionID : Attachment.LocalConfigID;
                    attachment = Attachment.Get(regionID, request.AttachmentID);

                    // If the attachment is missing, or was not created by the sender, forbidden!
                    if (attachment == null || attachment.UploaderUserID != session.UserID)
                    {
                        response.Status = DeliveryStatus.Forbidden;
                        return;
                    }
                }
                
                // Get the conversation container (Friendship, Group, etc)
                var conversationContainer = ConversationManager.GetConversationContainer(session.UserID, request.ConversationID, true);
                if (conversationContainer == null)
                {
                    response.Status = DeliveryStatus.UnknownUser;
                    return;
                }
                
                switch (conversationContainer.ConversationType)
                {
                    case ConversationType.Friendship:
                    {
                        // Note: We do not check permissons here. This is left up to the group server
                        PrivateMessageWorker.Create(session.UserID, session.MachineKey, request.ConversationID, request.Message, request.ClientID, attachment);
                        break;
                    }                    
                    case ConversationType.Group:
                    {                       
                        // Note: We do not check permissons here. This is left up to the group server

                        var group = conversationContainer as Group;

                        if (group.Type == GroupType.Large && group.IsRootGroup)
                        {
                            response.Status = DeliveryStatus.Forbidden;
                            return;
                        }

                        FriendsStatsManager.Current.GroupMessagesSent.Track(group.GroupID);
                        FriendsStatsManager.Current.GroupMessagesSentByPlatform.Track((int)session.DevicePlatform);
                        GroupMessageCordinator.Create(group, request, session.UserID, session.GetClientEndpoint(), attachment);

                        break;
                    }
                    case ConversationType.GroupPrivateConversation:
                    {
                        if (!conversationContainer.CanSendMessage(session.UserID))
                        {
                            response.Status = DeliveryStatus.Forbidden;
                            return;
                        }

                        var conversation = conversationContainer as GroupPrivateConversation;
                        if (conversation == null || conversation.Group == null || !conversation.Group.IsRootGroup)
                        {
                            response.Status = DeliveryStatus.Forbidden;
                            return;
                        }

                        FriendsStatsManager.Current.PrivateGroupMessagesSent.Track(request.ConversationID);
                        GroupPrivateMessageCoordinator.Create(conversation.Group, request, session.UserID, conversation.OtherUserID, session.GetClientEndpoint(), attachment);
                        break;
                    }
                }                               
            }
            catch (Exception ex)
            {
                response.Status = DeliveryStatus.Error;
                Logger.Error(ex, "Unhandled exception in OnChatAttachmentRequest");
            }
            finally
            {
                if (response.Status != DeliveryStatus.Successful)
                {
                    socketInterface.SendContract(response);
                }
            }
        }
               
    }
}
