﻿using System;
using System.Linq;
using Curse.SocketServer;
using Curse.SocketInterface;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Data;
using Curse.SocketMessages;
using Curse.Logging;
using Curse.Friends.Enums;
using Curse.Friends.Configuration;
using Curse.Friends.Statistics;
using Curse.Friends.NotificationProcessing;
using Curse.Logging.Uploader;

namespace Curse.Friends.NotificationService
{
	public class NotificationServer : SocketServer.SocketServer
	{
		public static readonly NotificationServer Instance = new NotificationServer(FriendsServiceConfiguration.Instance.NotificationServiceMaxConnections, FriendsServiceConfiguration.Instance.NotificationServicePorts.Length);                
		private NotificationHost _host;

		public NotificationServer(int maxConnections, int numPorts)
			: base(maxConnections, numPorts, false)
		{
		   RequestProcessing.RegisterDispatchers();
		}
		
		public override void Start(System.Net.IPEndPoint[] localEndPoints)
		{
			QueueProcessors.StartQueueProcessors();
			HousekeepEndpoints();
			RequestProcessing.Start();
			base.Start(localEndPoints);
			RegisterHost();
		}

		public void HousekeepEndpoints()
		{
			try
			{
				var endpoints = ClientEndpoint.GetAllLocal(p => p.ServerName, Environment.MachineName).Where(p => p.IsConnected).ToArray();
				Logger.Info("Discovered " + endpoints.Length + " that need to be resolved!");

				if (endpoints.Length > 1000)
				{
					Logger.Warn("More than 1,000 endpoints need to be resolved, so this will be skipped!", endpoints.Length);
					return;
				}

				foreach (var endpoint in endpoints)
				{
					UserStatusResolver.CreateDisconnectResolver(endpoint);
				}
			}
			catch (Exception ex)
			{
				Logger.Error(ex, "Failed to housekeep endpoints!");
			}
		}
	   
		public override void Stop()
		{
			RequestProcessing.Stop();
			State = SocketServerState.Stopping;

			// First mark this host offline, so that the web service stops routing to it
			try
			{
				MarkHostOffline();
			}
			catch (Exception ex)
			{
				Logger.Error(ex, "Failed to mark host as offline.");
			}

			// Stop the TCP service
			base.Stop();
			
			try
			{
				Logger.Info("Shutting down storage...");
				StorageConfiguration.Shutdown();
			}
			catch (Exception ex)
			{
				Logger.Error(ex, "Failed to close all queue connections!");
			}

			try
			{
				Logger.Info("Shutting down stat manager...");
				FriendsStatsManager.Shutdown();
			}
			catch (Exception ex)
			{
				Logger.Error(ex, "Failed to finish stat manager shutdown!");
			}
			
			try
			{
				LogUploader.Shutdown();
			}
			catch (Exception ex)
			{
				Logger.Error(ex, "Failed to finish log upload!");
			}
		}

		private void MarkHostOffline()
		{
			if (_host == null)
			{
				return;
			}

			_host.Status = NotificationHostStatus.Offline;
			_host.Update(p => p.Status);
		}

		private void RegisterHost()
		{
#if CONFIG_LOADTESTING
			const HostEnvironment hostEnvironment = HostEnvironment.Release;
#else
			var hostEnvironment = (HostEnvironment)Enum.Parse(typeof(HostEnvironment), FriendsServiceConfiguration.Mode.ToString(), true);
#endif
			var internalIP = NetworkHelper.GetInternalIpAddress();
			var publicIP = NetworkHelper.GetPublicIpAddress();
			var host = NotificationHost.GetLocal(internalIP);
			if (host == null)
			{
				host = new NotificationHost
				{
					InternalIPAddress = internalIP,
					PublicIPAddress = publicIP,
					DateCreated = DateTime.UtcNow,
					DateOnline = DateTime.UtcNow,
					DateUpdated = DateTime.UtcNow,
					MachineName = Environment.MachineName,
					Status = NotificationHostStatus.Online,
					Environment = hostEnvironment,                     

				};
				host.Version = host.CurrentVersion;
				host.InsertLocal();
			}
			else
			{
				host.DateUpdated = DateTime.UtcNow;
				host.DateOnline = DateTime.UtcNow;
				host.PublicIPAddress = publicIP;
				host.Status = NotificationHostStatus.Online;
				host.Environment = hostEnvironment;
				host.Version = host.CurrentVersion;
				host.Update();
			}

			_host = host;

		}

	   
	}
}
