﻿using System;
using System.Collections.Concurrent;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.ServerHosting;
using Curse.Logging;

namespace Curse.Friends.PollService
{
    public class PollServer : ServerHost<GroupPollHost,GroupPoll>
    {
        private static readonly PollServer _instance = new PollServer();

        public static void StartServer()
        {
            _instance.Start();
        }

        public static void StopServer()
        {
            _instance.Stop();
        }

        private readonly ConcurrentDictionary<string, PollSession> _sessions = new ConcurrentDictionary<string, PollSession>();
 
        private PollServer()
        {
        }

        protected override void CustomStartup()
        {
            GroupPollCoordinator.StartProcessor(GroupPollCoordinator_ProcessMessage);

            AddTimer("Periodic Poll Tasks", TimeSpan.FromSeconds(5), DoPeriodicTasks);
            AddTimer("Immediate Poll Tasks", TimeSpan.FromSeconds(1), DoImmediateTasks);
        }

        private void DoPeriodicTasks()
        {
            foreach (var session in _sessions.Values)
            {
                session.Refresh();

                if (session.IsActive)
                {
                    session.PeriodicNotify();
                }
                else
                {
                    Logger.Warn("Poll has become inactive, but did not get cleared via the coordinator. Unhosting it now.", new { session.GroupID, session.PollID });

                    PollSession removed;
                    if (_sessions.TryRemove(GetKey(session.GroupID, session.PollID), out removed))
                    {
                        removed.Shutdown();
                    }
                }
            }
        }

        private void DoImmediateTasks()
        {
            foreach (var session in _sessions.Values)
            {
                if (session.IsExpired())
                {
                    session.EndExpiredPoll();
                    session.Shutdown();

                    PollSession throwaway;
                    _sessions.TryRemove(GetKey(session.GroupID, session.PollID), out throwaway);
                }
            }
        }

        private PollSession GetOrAddSession(Guid groupID, int pollID, out bool retrievedFromCache)
        {
            var retrieved = true;
            var session = _sessions.GetOrAdd(GetKey(groupID, pollID), k =>
            {
                retrieved = false;
                return new PollSession(groupID, pollID);
            });

            retrievedFromCache = retrieved;
            return session;
        }

        protected override void CustomStop()
        {
            foreach (var session in _sessions.Values)
            {
                session.Shutdown();
            }
        }

        private static string GetKey(Guid groupID, int pollID)
        {
            return groupID.ToString() + pollID;
        }

        private void GroupPollCoordinator_ProcessMessage(GroupPollCoordinator coordinator)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(coordinator.GroupID, coordinator.PollID, out retrievedFromCache);
                switch (coordinator.ChangeType)
                {
                    case GroupPollChangeType.Started:
                        session.Start(coordinator.RequestorID);
                        break;
                    case GroupPollChangeType.Ended:
                        session.End(coordinator.RequestorID);
                        session.Shutdown();
                        _sessions.TryRemove(GetKey(coordinator.GroupID, coordinator.PollID), out session);
                        break;
                    case GroupPollChangeType.Deactivated:
                        session.Deactivate(coordinator.RequestorID);
                        session.Shutdown();
                        _sessions.TryRemove(GetKey(coordinator.GroupID, coordinator.PollID), out session);
                        break;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while processing GroupPollCoordinator", coordinator);
            }
        }
    }
}
