﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Logging;

namespace Curse.Friends.PollService
{
    public class PollSession
    {
        private readonly GroupPoll _poll;
        private readonly Group _group;
        private readonly GroupMember _creator;

        private Dictionary<int, int> _lastVotes = new Dictionary<int, int>(); 

        public Guid GroupID { get { return _poll.GroupID; } }

        public int PollID { get { return _poll.PollID; } }

        public PollSession(Guid groupID, int pollID)
        {
            _poll = GroupPoll.GetLocal(groupID, pollID);
            if (_poll == null)
            {
                throw new DataNotFoundException();
            }

            _group = Group.GetByID(_poll.GroupID);
            if (_group == null)
            {
                throw new DataNotFoundException();
            }

            _creator = _group.GetMember(_poll.CreatorUserID);
            if (_creator == null)
            {
                throw new DataNotFoundException();
            }

            if (_poll.Votes != null)
            {
                _lastVotes = _poll.Votes;
            }
        }

        public void Shutdown()
        {
            _poll.MachineName = string.Empty;
            _poll.Update(p => p.MachineName);
        }

        public void Start(int requestorUserID)
        {
            var requestor = GetRequestor(requestorUserID);

            _poll.Status = GroupPollStatus.Running;
            _poll.StatusChangeDate = DateTime.UtcNow;
            _poll.EndDate = _poll.StatusChangeDate + TimeSpan.FromMinutes(_poll.DurationMinutes);
            _poll.Update(p => p.Status, p => p.StatusChangeDate, p => p.EndDate);

            GroupEventManager.LogCreatePollEvent(requestor, _poll);
            Notify(GroupPollChangeType.Started, requestorUserID);
        }

        public void End(int requestorUserID)
        {
            _poll.Status = GroupPollStatus.Ended;
            _poll.StatusChangeDate = DateTime.UtcNow;
            _poll.EndDate = _poll.StatusChangeDate;
            _poll.Update(p => p.Status, p => p.StatusChangeDate, p => p.EndDate);

            PeriodicNotify();

            var requestor = GetRequestor(requestorUserID);
            GroupEventManager.LogEndPollEvent(requestor, _poll);
            Notify(GroupPollChangeType.Ended, requestor.UserID);
        }

        private GroupMember GetRequestor(int requestorUserID)
        {
            return requestorUserID == _creator.UserID ? _creator : _group.GetMember(requestorUserID) ?? new GroupMember {UserID = requestorUserID, Username = "Unknown"};
        }

        public void Deactivate(int requestorUserID)
        {
            if (_poll.Status == GroupPollStatus.Inactive)
            {
                return;
            }

            if (_poll.Status != GroupPollStatus.Ended)
            {
                End(requestorUserID);
            }

            _poll.Status = GroupPollStatus.Inactive;
            _poll.StatusChangeDate = DateTime.UtcNow;
            _poll.Update(p => p.Status, p => p.StatusChangeDate);

            Notify(GroupPollChangeType.Deactivated, requestorUserID);
        }

        public void EndExpiredPoll()
        {
            End(_creator.UserID);
        }

        public void Refresh()
        {
            _poll.Refresh();
        }

        public bool IsActive
        {
            get
            {
                return _poll.Status != GroupPollStatus.Inactive;
            }
        }

        public void PeriodicNotify()
        {
            if (!IsActive)
            {
                return;
            }

            Dictionary<int, int> options;
            if (_poll.OptionIDs != null)
            {
                options = _poll.GetOptions().ToDictionary(o => o.OptionID, o => o.VoteCount);
            }
            else
            {
                // Legacy
                options = _poll.CalculateVotesLegacy();
                _poll.Votes = options;
                _poll.Update(p => p.Votes);
            }

            var change = false;
            foreach (var opt in options)
            {
                int lastCount;
                if (!_lastVotes.TryGetValue(opt.Key, out lastCount) || lastCount != opt.Value)
                {
                    change = true;
                    break;
                }
            }

            if (change)
            {
                _lastVotes = options;
                Notify(GroupPollChangeType.VotesUpdated);
            }
        }

        public bool IsExpired()
        {
            return _poll.DurationMinutes > 0 && (_poll.EndDate - DateTime.UtcNow <= TimeSpan.Zero);
        }

        private void Notify(GroupPollChangeType changeType, int? requestorUserID=null)
        {
            var notification = new GroupPollChangedNotification
            {
                ChangeType = changeType,
                Poll = _poll.ToNotification(FriendsServiceConfiguration.Instance.PublicPollUrlFormat),
                RequestorUserID = requestorUserID ?? 0
            };

            // Notify the group
            var group = Group.GetLocal(_poll.GroupID);
            if (group != null)
            {
                GroupPollChangedCoordinator.Create(group, notification);
            }
        }
    }
}
