﻿using System;
using System.Collections.Generic;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;

namespace Curse.Friends.PollsWebService.Contracts
{
    public class CreatePollRequest : BaseRequestContract
    {
        /// <summary>
        /// The title of the poll.
        /// </summary>
        public string Title { get; set; }

        /// <summary>
        /// The poll options.
        /// </summary>
        public Dictionary<int, string> Options { get; set; }

        /// <summary>
        /// How to display the poll results.
        /// </summary>
        public GroupPollDisplayType DisplayType { get; set; }

        /// <summary>
        /// Roles allowed to vote.
        /// </summary>
        public HashSet<int> RequiredRoles { get; set; }

        /// <summary>
        /// How long the poll should last.
        /// </summary>
        public int DurationMinutes { get; set; }

        /// <summary>
        /// Allow multiple selected options per vote.
        /// </summary>
        public bool AllowMultiSelect { get; set; }

        /// <summary>
        /// Whether or not people outside of the community can vote in the poll.
        /// </summary>
        public bool IsPublic { get; set; }

        /// <summary>
        /// Whether or not people can change their votes.
        /// </summary>
        public bool AllowRevotes { get; set; }

        public GroupPollDuplicateMode DuplicateMode { get; set; }

        protected override bool DoValidate()
        {
            if (!Title.SafeRange(1,GroupPoll.MaxPollTitleLength))
            {
                _validationMessage = "Poll title is too short or too long";
                return false;
            }

            if (Options == null || Options.Count < 2)
            {
                _validationMessage = "Not enough options specified";
                return false;
            }

            foreach (var option in Options)
            {
                if (!option.Value.SafeRange(1, GroupPollOption.MaxOptionLength))
                {
                    _validationMessage = "One or more options are too short or too long";
                    return false;
                }
            }

            if (IsPublic)
            {
                if (RequiredRoles != null && RequiredRoles.Count > 0)
                {
                    _validationMessage = "Cannot specify required roles on a public poll";
                    return false;
                }

                if (DuplicateMode == GroupPollDuplicateMode.PreventByUserID)
                {
                    _validationMessage = "Cannot prevent duplicates by user ID on a public poll";
                    return false;
                }
            }

            if (DuplicateMode == GroupPollDuplicateMode.AllowDuplicates && AllowRevotes)
            {
                _validationMessage = "Cannot allow revotes when duplicates are enabled";
                return false;
            }

            return true;
        }
    }
}