﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.MicroService.Extensions;
using Curse.Friends.NotificationContracts;
using Curse.Friends.PollsWebService.Contracts;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.PollsWebService.Controllers
{
    [RoutePrefix("polls")]
    public class PollController : MicroServiceController
    {
        /// <summary>
        /// Gets the group's active polls.
        /// </summary>
        /// <param name="groupID">The ID of the group.</param>
        [HttpGet]
        [Route("{groupID}")]
        [ResponseType(typeof (IEnumerable<ActivePollContract>))]
        public IHttpActionResult GetActive(Guid groupID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var polls = group.GetActivePolls();

            var contracts = polls.Select(p =>
            {
                var id = Token.UserID.ToString();
                var participant = p.GetParticipant(id);
                return new ActivePollContract
                {
                    Poll = p.ToNotification(FriendsServiceConfiguration.Instance.PublicPollUrlFormat, true),
                    MyVotes = participant == null ? new HashSet<int>() : participant.Vote
                };
            });
            return Ok(contracts);
        }

        /// <summary>
        /// Start a new poll for the group.
        /// </summary>
        /// <param name="groupID">The ID of the group to run the poll in.</param>
        /// <param name="request">The poll details.</param>
        /// <returns>The ID of the poll created.</returns>
        [HttpPost]
        [Route("{groupID}")]
        [ResponseType(typeof (GroupPollNotification))]
        [SocialBanFilter]
        public IHttpActionResult Create(Guid groupID, [FromBody] CreatePollRequest request)
        {
            request.Validate();

            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var settings = new GroupPollSettings
            {
                GroupID = groupID,
                RegionID = group.RegionID,
                RequiredRoles = request.RequiredRoles??new HashSet<int>(),
                IsPublic = request.IsPublic,
                AllowRevotes = request.AllowRevotes,
                DisplayType = request.DisplayType,
                AllowMultipleSelections = request.AllowMultiSelect,
                DuplicateMode = request.DuplicateMode
            };
            var poll = group.CreatePoll(Token.UserID, request.Title, request.Options, request.DurationMinutes, settings);
            return Ok(poll.ToNotification(FriendsServiceConfiguration.Instance.PublicPollUrlFormat));
        }

        [HttpGet]
        [Route("{groupID}/{pollID}")]
        [ResponseType(typeof (GroupPollNotification))]
        public IHttpActionResult Get(Guid groupID, int pollID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.Access, Token.UserID);

            var poll = group.GetPoll(pollID);
            if (poll == null)
            {
                return NotFound();
            }

            return Ok(poll.ToNotification(FriendsServiceConfiguration.Instance.PublicPollUrlFormat));
        }

        /// <summary>
        /// Gets the most recent settings used in polls for this group.
        /// </summary>
        /// <param name="groupID">The ID of the group.</param>
        [HttpGet]
        [Route("{groupID}/settings")]
        [ResponseType(typeof (GroupPollSettingsNotification))]
        public IHttpActionResult GetLatestSettings(Guid groupID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.ManagePolls, Token.UserID);

            var settings = group.GetPollSettingsOrDefault();
            return Ok(settings.ToNotification());
        }

        /// <summary>
        /// Gets poll history in descending date order.
        /// </summary>
        /// <param name="groupID">The group ID</param>
        /// <param name="pageSize">The optional number of results per page</param>
        /// <param name="pageNumber">The optional page number</param>
        [HttpGet]
        [Route("{groupID}/history")]
        [ResponseType(typeof (IEnumerable<GroupPollResult>))]
        public IHttpActionResult GetPollHistory(Guid groupID, int pageSize = 10, int pageNumber = 1)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.ManagePolls, Token.UserID);

            var events = GroupEventManager.GetByGroupAndType(groupID.ToString(), GroupEventType.PollEnded, pageSize, pageNumber);
            return Ok(events.Select(ToPollResult));
        }

        /// <summary>
        /// Search poll history.
        /// </summary>
        /// <param name="groupID">The group ID</param>
        /// <param name="request">The search request details</param>
        [HttpPost]
        [Route("{groupID}/history/search")]
        [ResponseType(typeof (IEnumerable<GroupPollResult>))]
        public IHttpActionResult SearchPollHistory(Guid groupID, [FromBody] SearchPollHistoryRequest request)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.ManagePolls, Token.UserID);

            var events = GroupEventManager.GetByGroupAndType(groupID.ToString(), GroupEventType.PollEnded, request.PageSize ?? 10, request.PageNumber ?? 1);
            return Ok(events.Select(ToPollResult));
        }

        private static GroupPollResult ToPollResult(GroupEvent e)
        {
            return new GroupPollResult
            {
                GroupID = e.GroupID,
                PollID = e.PollDetails.PollID,
                Title = e.PollDetails.Title,
                TotalVotes = e.PollDetails.TotalVotes,
                RequiredRoles = e.PollDetails.RequiredRoles,
                TotalOptions = e.PollDetails.OptionsCount,
                WinningOptions = e.PollDetails.WinningOptions,
                DurationMinutes = e.PollDetails.DurationMinutes,
                DateEnded = e.Timestamp,
            };
        }

        /// <summary>
        /// Ends a poll.
        /// </summary>
        /// <param name="groupID">The ID of the group running the poll.</param>
        /// <param name="pollID">The ID of the poll within the group.</param>
        [HttpDelete]
        [Route("{groupID}/{pollID}")]
        [ResponseType(typeof (void))]
        public IHttpActionResult End(Guid groupID, int pollID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var poll = group.GetPoll(pollID);
            if (poll == null)
            {
                return NotFound();
            }

            poll.End(Token.UserID);
            return StatusCode(HttpStatusCode.Accepted);
        }

        /// <summary>
        /// Removes a poll from the active polls list.
        /// </summary>
        /// <param name="groupID">The group ID</param>
        /// <param name="pollID">The poll ID</param>
        [HttpDelete]
        [Route("{groupID}/{pollID}/deactivate")]
        [ResponseType(typeof (void))]
        public IHttpActionResult Deactivate(Guid groupID, int pollID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var poll = group.GetPoll(pollID);
            if (poll == null)
            {
                return NotFound();
            }

            poll.Deactivate(Token.UserID);
            return StatusCode(HttpStatusCode.NoContent);
        }

        /// <summary>
        /// Gets the authenticated user's vote for the specified poll.
        /// </summary>
        /// <param name="groupID">The ID of the group.</param>
        /// <param name="pollID">The ID of the poll.</param>
        /// <returns></returns>
        [HttpGet]
        [Route("{groupID}/{pollID}/votes")]
        [ResponseType(typeof (HashSet<int>))]
        public IHttpActionResult GetMyVote(Guid groupID, int pollID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var poll = group.GetPoll(pollID);
            if (poll == null)
            {
                return NotFound();
            }


            var participant = poll.GetParticipant(Token.UserID.ToString());

            return Ok(participant != null ? participant.Vote : new HashSet<int>());
        }

        /// <summary>
        /// Vote in a poll.
        /// </summary>
        /// <param name="groupID">The ID of the group running the poll.</param>
        /// <param name="pollID">The ID of the poll within the group.</param>
        /// <param name="vote">Selections voted for.</param>
        [HttpPost]
        [Route("{groupID}/{pollID}/votes")]
        [ResponseType(typeof (ActivePollContract))]
        [SocialBanFilter]
        public IHttpActionResult Vote(Guid groupID, int pollID, [FromBody] HashSet<int> vote)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var poll = group.GetPoll(pollID, true);
            if (poll == null)
            {
                return NotFound();
            }

            if (string.IsNullOrEmpty(poll.Code))
            {
                var member = group.CheckPermission(GroupPermissions.Access, Token.UserID);
                if (poll.RequiredRoles != null && poll.RequiredRoles.Count > 0 && !poll.RequiredRoles.Contains(group.DefaultRoleID) && !member.Roles.Overlaps(poll.RequiredRoles))
                {
                    return Forbidden("Requestor does not meet requirements");
                }
            }

            var requestorID = Token.UserID.ToString();
            poll.Vote(requestorID, vote);

            var response = Request.CreateResponse(HttpStatusCode.OK, new ActivePollContract
            {
                Poll = poll.ToNotification(FriendsServiceConfiguration.Instance.PublicPollUrlFormat, true),
                MyVotes = vote
            });

            if (poll.DuplicateMode == GroupPollDuplicateMode.PreventByCookie && Request.GetCookie(_identifierCookieName)==null)
            {
                AddVoterCookie(response, requestorID);
            }

            return ResponseMessage(response);
        }

        #region Public Poll Endpoints


        const string _identifierCookieName = "voteid";

        [HttpGet]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        [ResponseType(typeof (PublicActivePollContract))]
        [Route("public/{pollCode}")]
        public IHttpActionResult GetPublicPoll(string pollCode)
        {
            var poll = GroupPoll.GetByPollCode(pollCode);
            if (poll == null)
            {
                return NotFound();
            }

            var group = Group.GetByID(poll.GroupID);
            if (group == null)
            {
                return NotFound();
            }

            var activePoll = new PublicActivePollContract
            {
                Group = group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, true),
                Poll = poll.ToNotification(FriendsServiceConfiguration.Instance.PublicPollUrlFormat),
            };

            if (poll.DuplicateMode != GroupPollDuplicateMode.AllowDuplicates)
            {
                var voter = GetVoterID(poll);
                var me = poll.GetParticipant(voter);
                if (me != null)
                {
                    activePoll.MyVotes = me.Vote;
                }
            }

            return Ok(activePoll);
        }

        [HttpPost]
        [ResponseType(typeof (PublicActivePollContract))]
        [Route("public/{pollCode}/votes")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult VotePublicPoll(string pollCode, [FromBody] HashSet<int> votes)
        {
            var poll = GroupPoll.GetByPollCode(pollCode);
            if (poll == null)
            {
                return NotFound();
            }

            var group = Group.GetByID(poll.GroupID);
            if (group == null)
            {
                return NotFound();
            }

            if (poll.DuplicateMode == GroupPollDuplicateMode.Legacy && Token.IsAnonymous)
            {
                return Forbidden();
            }

            var requestorID = GetVoterID(poll);

            poll.Vote(requestorID, votes);

            var response = Request.CreateResponse(HttpStatusCode.OK, new PublicActivePollContract
            {
                Group = group.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, true),
                Poll = poll.ToNotification(FriendsServiceConfiguration.Instance.PublicPollUrlFormat, true),
                MyVotes = votes
            });

            if (poll.DuplicateMode == GroupPollDuplicateMode.PreventByCookie && Request.GetCookie(_identifierCookieName) == null && Token.IsAnonymous)
            {
                AddVoterCookie(response, requestorID);
            }

            return ResponseMessage(response);
        }

        #endregion

        private string GetVoterID(GroupPoll poll)
        {
            if (!Token.IsAnonymous)
            {
                return Token.UserID.ToString();
            }
            switch (poll.DuplicateMode)
            {
                case GroupPollDuplicateMode.Legacy:
                    return Token.UserID.ToString();
                case GroupPollDuplicateMode.PreventByIP:
                    return GetCurrentIpAddress().ToString();
                case GroupPollDuplicateMode.PreventByCookie:
                    return Request.GetCookie(_identifierCookieName)??Guid.NewGuid().ToString();
                case GroupPollDuplicateMode.PreventByUserID:
                    return Token.UserID.ToString();
                case GroupPollDuplicateMode.AllowDuplicates:
                default:
                    return null;
            }
        }

        private void AddVoterCookie(HttpResponseMessage message, string id)
        {
            // add a new cookie
            message.Headers.AddCookies(new[]
            {
                new CookieHeaderValue(_identifierCookieName, id)
                {
                    HttpOnly = true,
                    Domain = Request.RequestUri.Host,
                    Path = Request.RequestUri.AbsolutePath.Replace("/votes",string.Empty)
                }
            });
        }
    }
}
