﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text.RegularExpressions;
using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.CloudSearch;
using Curse.CloudServices.Jobs;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Queues;
using Curse.Friends.Data.Search;
using Curse.Friends.Data.Search.FriendSearch;
using Curse.Friends.UserEvents;
using Curse.VaultSharp;
using Nest;

namespace Curse.Friends.PreDeployTools
{
    class Program
    {
        private static ConfigurationMode mode;

        private static void Main(string[] args)
        {
            string selectedMode;
            if(!PromptUser(Enum.GetNames(typeof(ConfigurationMode)), out selectedMode, "Cancel"))
            {
                Console.WriteLine("Canceled. Press Enter to exit");
                Console.ReadLine();
                return;
            }
            mode = (ConfigurationMode) Enum.Parse(typeof(ConfigurationMode), selectedMode, true);
            Console.WriteLine("Using configuration: " + mode);

            Console.WriteLine("Press the Y key to confirm...");
            if (Console.ReadKey(true).Key != ConsoleKey.Y)
            {
                Console.WriteLine("Canceled. Press Enter to exit");
                Console.ReadLine();
                return;
            }

            var secretsHelper = SecretsHelper.GetSecretsHelper();
            switch (mode)
            {
                case ConfigurationMode.Release:
                case ConfigurationMode.Beta:
                    secretsHelper.ReplaceSecrets(AppDomain.CurrentDomain.BaseDirectory, "release");
                    //VaultSharpHelper.ReplaceSecrets(AppDomain.CurrentDomain.BaseDirectory, "release");
                    break;
                case ConfigurationMode.Staging:
                    secretsHelper.ReplaceSecrets(AppDomain.CurrentDomain.BaseDirectory, "staging");
                    //VaultSharpHelper.ReplaceSecrets(AppDomain.CurrentDomain.BaseDirectory, "staging");
                    break;
            }


            Console.Write("Initializing storage configuration...");
            StorageConfiguration.Initialize("PreDeployTools", ConfigurationRegion.USEast.Key, mode);
            AerospikeConfiguration.CreateIndexes = true;
            QueueConfiguration.FastCreateQueues = false;

            Console.Write("Done!");

            var options = new Dictionary<string, Action>
            {
                {"validate-aerospike", ValidateAerospike},
                {"reindex-aerospike", CreateAerospikeIndexes},
                {"retemplate-elastic", RetemplateElastic},
                {"rabbit", ConfigureRabbit},
                {"run-test-code", RunTestCode}
            };
            string selectedOption;
            while (PromptUser(options.Keys, out selectedOption, "exit"))
            {
                options[selectedOption]();
            }
            StorageConfiguration.Shutdown();
        }

        static void ConfigureRabbit()
        {
            var options = new Dictionary<string, Action>
            {
                {"Validate", ValidateRabbit},
                {"Create-Queues", CreateQueues}
            };

            string selected;
            while (PromptUser(options.Keys.ToArray(), out selected))
            {
                options[selected]();
            }
        }

        static bool InheritsFromQueue(Type type)
        {
            var baseTypes = new[]
            {
                typeof (BaseCloudQueueRoutedMessage<>),
                typeof (BaseCloudQueueShoveledMessage<>),
                typeof (BaseCloudQueueWorkerMessage<>)
            };

            var baseType = type.BaseType;

            if (baseType == null || baseType == typeof(object))
            {
                return false;
            }

            if (baseType.IsGenericType)
            {
                var genericDefinition = baseType.GetGenericTypeDefinition();
                if (baseTypes.Any(t => genericDefinition == t))
                {
                    return true;
                }
            }

            return InheritsFromQueue(baseType);
        }

        static Type[] GetAllQueueTypes()
        {
            var allTypes = typeof(SingleSessionNotifier).Assembly.GetTypes().Concat(typeof(UserEventWorker).Assembly.GetTypes());
            return allTypes.Where(t => t.IsClass && !t.IsAbstract && InheritsFromQueue(t)).OrderBy(t => t.Name).ToArray();
        }

        static void ValidateRabbit()
        {
            // Get all queues
            var types = GetAllQueueTypes();

            var explicitConfig = QueueConfiguration.ExplicitQueueConfig.ToDictionary(kvp => kvp.Key, kvp => new { Types = new HashSet<string>(kvp.Value.Types), kvp.Value.IsLegacy });
            var badQueuesDetected = false;
            foreach (var type in types)
            {
                var config = explicitConfig.Select(kvp => new { kvp.Key, kvp.Value }).FirstOrDefault(c => c.Value.Types.Contains(type.Name));
                if (config == null)
                {
                    PrintMessageWithColor(ConsoleColor.Red, $"{type.Name} has no explicit route!");
                    badQueuesDetected = true;
                }
                else if (config.Value.IsLegacy)
                {
                    PrintMessageWithColor(ConsoleColor.Yellow, $"{type.Name} is in a legacy cluster!");
                    badQueuesDetected = true;
                }
            }

            var typeLookup = explicitConfig.SelectMany(c => c.Value.Types.Select(t => new { c.Key, Value = t })).ToArray();
            foreach (var explicitType in typeLookup)
            {
                if (types.All(t => t.Name != explicitType.Value))
                {
                    PrintMessageWithColor(ConsoleColor.White, $"{explicitType.Value} is defined in route {explicitType.Key} but does not exist as a type");
                }
            }

            foreach(var type in types)
            {
                var matchingConfigs = typeLookup.Where(t => t.Value == type.Name).ToArray();
                if(matchingConfigs.Length > 1)
                {
                    PrintMessageWithColor(ConsoleColor.Yellow, $"{type.Name} is defined in multiple routes: {string.Join(",", matchingConfigs.Select(c => c.Key).ToArray())}");
                }
            }

            if (!badQueuesDetected)
            {
                Console.WriteLine("All queues are explicitly routed in non-legacy clusters");
            }
            Console.ReadLine();
        }

        static void PrintMessageWithColor(ConsoleColor color, string message)
        {
            var original = Console.ForegroundColor;
            Console.ForegroundColor = color;
            Console.WriteLine(message);
            Console.ForegroundColor = original;
        }

        static bool PromptUser(IEnumerable<string> options, out string selectedOption, string exitLoopOptionText = "Back")
        {
            var invariantOptions = options.ToDictionary(o => o.ToLowerInvariant(), o => o);
            var invariantExit = exitLoopOptionText.ToLowerInvariant();

            while (true)
            {
                Console.Clear();

                foreach (var option in options)
                {
                    Console.WriteLine("- {0}", option);
                }
                Console.WriteLine("- {0}", exitLoopOptionText);

                var choice = Console.ReadLine().ToLowerInvariant();

                if (choice == invariantExit)
                {
                    selectedOption = exitLoopOptionText;
                    return false;
                }

                if (invariantOptions.TryGetValue(choice, out selectedOption))
                {
                    return true;
                }

                Console.WriteLine("Unknown option: {0}. Press Enter to continue", choice);
                Console.ReadLine();
            }
        }

        static void RunTestCode()
        {
            var total = 0;
            var favoritesOnly = 0;

            User.BatchOperateLocal(1000, p =>
            {
                total = total + p.Count();
                favoritesOnly = favoritesOnly + p.Count(u => (int)u.FriendMessagePushPreference == 1);
            });

            Console.WriteLine("Total: " + total + ", Favs only: " + favoritesOnly);
        }

        static void CreateQueues()
        {
            Console.WriteLine("Do you want to replace remote queues if configuration doesn't match? (y|n)");
            QueueConfiguration.ReplaceRemoteQueues = Console.ReadKey(true).Key == ConsoleKey.Y;

            Console.WriteLine("Enter an optional type route ID: ");
            var routeVal = Console.ReadLine();
            var routeFilter = 0;
            if (!string.IsNullOrWhiteSpace(routeVal))
            {
                if (!int.TryParse(routeVal, out routeFilter))
                {
                    Console.WriteLine("Invalid entry: " + routeVal);
                    return;
                }
            }

            Console.WriteLine("Enter an optional queue type filter: ");
            var queueNameFilter = Console.ReadLine();


            var allQueueTypes = GetAllQueueTypes();

            if (!string.IsNullOrEmpty(queueNameFilter))
            {
                allQueueTypes = allQueueTypes.Where(p => p.Name.ToLowerInvariant().Contains(queueNameFilter.ToLowerInvariant())).ToArray();
            }

            Console.WriteLine("The following types have no explicit route mapping:");
            foreach (var type in allQueueTypes.Where(p => !QueueConfiguration.ExplcitQueueRoutes.Contains(p.Name)))
            {
                Console.WriteLine(type.Name);
            }


            Console.WriteLine("The following types have explicit route mappings:");
            foreach (var type in allQueueTypes.Where(p => QueueConfiguration.ExplcitQueueRoutes.Contains(p.Name)))
            {
                Console.WriteLine(type.Name);
            }

            Console.WriteLine("Do you wish to proceed?");

            if (Console.ReadKey(true).Key != ConsoleKey.Y)
            {
                return;
            }

            QueueConfiguration.CreateQueues = true;

            // Group configs by their type route ID (ex: 1, 2, 3, etc);
            var configsByRoute = QueueConfiguration.Configurations.GroupBy(c => c.TypeRouteID).ToArray();

            // Distinct regions
            var regions = QueueConfiguration.Configurations.GroupBy(c => c.RegionKey).Select(r => r.Key).Distinct().ToArray();
            
            // Iterate over each region
            foreach (var region in regions)
            {
                // TODO: allow user to enter a region. 
                Console.WriteLine("Processing region: " + region);

                foreach (var routeGroup in configsByRoute)
                {
                    var routeID = routeGroup.Key;

                    if (routeFilter != 0 && routeID != routeFilter)
                    {
                        Console.WriteLine("Skipping route: " + routeID);
                        continue;
                        
                    }

                    Console.WriteLine("Processing route: " + routeID);

                    var localConfig = routeGroup.FirstOrDefault(t => t.RegionKey == region);
                    if (localConfig == null)
                    {
                        Console.WriteLine("No local config found for route: {0}", routeID);
                        continue;
                    }
                    var remoteConfigs = routeGroup.Where(t => t.RegionKey != region).ToArray();

                    // TODO: Check remote configs for similarity
                    var connectionManager = new RabbitConnectionManager(localConfig, remoteConfigs);
                    
                    foreach (var type in allQueueTypes)
                    {
                        var typeName = type.Name;

                        // Only create the queue for this config if it is mapped here, or this is the default
                        if (localConfig.Types.Contains(typeName))
                        {
                            Console.Write("Creating queue '" + typeName + "' on '" + localConfig.Addresses.First() + "' in route '" + routeID + "' ... ");
                            var rabbitQueueType = typeof(RabbitQueue<>).MakeGenericType(type);
                            Activator.CreateInstance(rabbitQueueType, GetArgumentsForMessageType(type, connectionManager));
                            Console.WriteLine("Done!");
                        }
                    }
                }                
            }
        }

        static object[] GetArgumentsForMessageType(Type type, RabbitConnectionManager connectionManager)
        {
            bool isShoveled;
            bool isMachineRouted;
            bool isPersistent = false;
            bool isAcknowledged = false;
            int? retryLimit = null;
            int? messageTtlMilliseconds = null;
            int? maxQueueLength = null;
            int? maxParallelProcessing = null;
            int? prefetchMultiplier = null;
            int? maxQueueBytes = null;

            var queueAttribute = type.GetCustomAttribute<CloudQueueAttribute>();
            if (queueAttribute != null)
            {
                isPersistent = queueAttribute.IsPersistent;
            }

            var workerAttribute = type.GetCustomAttribute<CloudWorkerQueueAttribute>();
            if (workerAttribute != null)
            {
                maxQueueLength = workerAttribute.MaxQueueLength;
                maxQueueBytes = workerAttribute.MaxQueueBytes;
            }

            if (IsWorkerMessage(type))
            {
                isShoveled = false;
                isMachineRouted = false;

            }
            else if (IsRoutedMessage(type))
            {
                isShoveled = true;
                isMachineRouted = true;
            }
            else if (IsShoveledMessage(type))
            {
                isShoveled = true;
                isMachineRouted = false;
            }
            else
            {
                throw new InvalidOperationException("Queue message was not a worker, routed, or shoveled message!");
            }

            return new object[]
            {
                connectionManager,
                CloudQueueMode.Consumer,
                type.Name,
                isShoveled,
                isMachineRouted,
                isPersistent,
                isAcknowledged,
                retryLimit,
                messageTtlMilliseconds,
                maxQueueLength,
                maxParallelProcessing,
                prefetchMultiplier,
                maxQueueBytes,
            };
        }

        static bool IsSubclassOfGeneric(Type type, params Type[] expectedTypes)
        {
            var t = type;
            while (t != null)
            {
                if (t.IsGenericType)
                {
                    var genTypeDef = t.GetGenericTypeDefinition();
                    if (expectedTypes.Any(e => genTypeDef == e))
                    {
                        return true;
                    }
                }

                t = t.BaseType;
            }
            return false;
        }

        static bool IsCloudQueueMessage(Type type)
        {
            return IsSubclassOfGeneric(type, typeof(BaseCloudQueueRoutedMessage<>), typeof(BaseCloudQueueShoveledMessage<>), typeof(BaseCloudQueueWorkerMessage<>));
        }

        static bool IsRoutedMessage(Type type)
        {
            return IsSubclassOfGeneric(type, typeof(BaseCloudQueueRoutedMessage<>));
        }

        static bool IsShoveledMessage(Type type)
        {
            return IsSubclassOfGeneric(type, typeof(BaseCloudQueueShoveledMessage<>));
        }

        static bool IsWorkerMessage(Type type)
        {
            return IsSubclassOfGeneric(type, typeof(BaseCloudQueueWorkerMessage<>));
        }

        static void ValidateAerospike()
        {
            // Get all tables
            var allTableTypes = typeof(ClientEndpoint).Assembly.GetTypes().Where(p => p.GetCustomAttribute<TableDefinitionAttribute>() != null)
                .Select(t => new { Type = t, TableDefinition = t.GetCustomAttribute<TableDefinitionAttribute>() }).ToArray();

            // Go over each region, and build indexes
            var configurationsByRegion = AerospikeConfiguration.Configurations.GroupBy(c => c.RegionKey).ToArray();
            foreach (var region in ConfigurationRegion.AllRegions.Where(region => configurationsByRegion.All(g => g.Key != region.Key)))
            {
                Console.WriteLine("No configs found for region {0}.", region);
            }

            // Ensure all types are represented in each region
            foreach (var regionGroup in configurationsByRegion)
            {
                Console.WriteLine("----------------------------------------------");
                Console.WriteLine("Validating Region {0}", regionGroup.Key);
                var sets = regionGroup.SelectMany(r => r.Sets).ToArray();

                var setGrouping = sets.GroupBy(s => s);
                foreach (var setGroup in setGrouping)
                {
                    if (setGroup.Count() > 1)
                    {
                        Console.WriteLine("Set {0} specified more than once in region {1}", setGroup.Key, regionGroup.Key);
                    }
                }

                foreach (var table in allTableTypes)
                {
                    if (table.Type.GetCustomAttribute<ObsoleteAttribute>() != null)
                    {
                        Console.WriteLine("Skipping obsolete table: " + table.Type.Name);
                        continue;
                    }

                    if (sets.All(s => s != table.TableDefinition.TableName))
                    {
                        Console.WriteLine("No set specified for table {0} in region {1}", table.TableDefinition.TableName, regionGroup.Key);
                    }
                }

                foreach (var set in sets)
                {
                    if (allTableTypes.All(t => t.TableDefinition.TableName != set))
                    {
                        Console.WriteLine("No type found for set {0} in region {1}", set, regionGroup.Key);
                    }
                }

                foreach (var configuration in regionGroup)
                {
                    foreach (var address in configuration.Addresses)
                    {
                        if (mode == ConfigurationMode.Release)
                        {
                            if (!Regex.IsMatch(address, @"(?:nosql|aerospike)(?:\d+)(?:a|b|c)-(?:iad|dub|sin)\.curse\.(?:io|us)"))
                            {
                                Console.WriteLine("Address {0} in region {1} does not match expected format.", address, regionGroup.Key);
                            }
                            if ((regionGroup.Key == "US-East" && !address.Contains("iad"))
                                || (regionGroup.Key == "EU-West" && !address.Contains("dub"))
                                || (regionGroup.Key == "AP-Southeast" && !address.Contains("sin")))
                            {
                                Console.WriteLine("Address {0} in region {1} does not have the right region suffix");
                            }
                        }
                    }
                }
                Console.WriteLine("----------------------------------------------");
            }
        }

        private static void CreateAerospikeIndexes()
        {
            // Go over each region, and build indexes
            var configurationsByRegion = AerospikeConfiguration.Configurations.GroupBy(c => c.RegionKey).ToArray();
            foreach (var region in ConfigurationRegion.AllRegions.Where(region => configurationsByRegion.All(g => g.Key != region.Key)))
            {
                Console.WriteLine("No configs found for region {0}. This region will be skipped.", region);
            }


            var allAssemblyTypes = typeof(JobInfo).Assembly.GetTypes().Concat(typeof(ClientEndpoint).Assembly.GetTypes());

            // Get all tables
            var allTableTypes = allAssemblyTypes.Where(p => p.GetCustomAttribute<TableDefinitionAttribute>() != null)
                .Select(t => new TableMap { Type = t, TableDefinition = t.GetCustomAttribute<TableDefinitionAttribute>() }).ToArray();


            foreach (var regionGroup in configurationsByRegion)
            {
                Console.WriteLine("Processing " + regionGroup.Key + "...");

                foreach (var config in regionGroup)
                {
                    foreach (var set in config.Sets.OrderBy(p => p.ToLowerInvariant()))
                    {
                        Console.WriteLine("Processing set: " + set);

                        if (!ProcessTable(allTableTypes.FirstOrDefault(p => p.TableDefinition.TableName.Equals(set)), set, config))
                        {
                            return;
                        }
                    }
                }
            }

            Console.WriteLine("Done!");
        }

        private class TableMap
        {
            public Type Type { get; set; }
            public TableDefinitionAttribute TableDefinition { get; set; }
        }

        static bool ProcessTable(TableMap tableMap, string set, AerospikeConfiguration config)
        {

            if (tableMap == null)
            {
                Console.WriteLine("No type found for set {0}", set);
                Console.WriteLine("Do you want to continue anyway? (Y,N)");
                if (Console.ReadKey().Key != ConsoleKey.Y)
                {
                    Console.WriteLine();
                    return false;
                }
                Console.WriteLine();
                return true;
            }

            if (tableMap.Type.GetCustomAttribute<ObsoleteAttribute>() != null)
            {
                Console.WriteLine("Skipping obsolete table '" + tableMap.Type.Name + "'");
                return true;
            }

            var indexedColumns = tableMap.Type.GetProperties().Where(p => p.GetCustomAttribute<ColumnAttribute>() != null);
            foreach (var column in indexedColumns)
            {
                var attr = column.GetCustomAttribute<ColumnAttribute>();
                if (!attr.IsIndexed)
                {
                    continue;
                }
                Console.WriteLine("Creating index: " + tableMap.TableDefinition.TableName + "." + column.Name + " on node: " + config.Addresses.FirstOrDefault());

                var method = tableMap.Type.BaseType.GetMethod("CreateIndex", BindingFlags.Static | BindingFlags.NonPublic);
                method.Invoke(null, BindingFlags.Static, null, new object[] { config, attr.Name, column.PropertyType }, null);
            }
            return true;
        }

        static void RetemplateElastic()
        {
            var options = new Dictionary<string, Action>
            {
                {"conversations", () => RetemplateSearchModel2(new ConversationManager())},
                {"groupevents", () => RetemplateSearchModel2(new GroupEventManager())},
                {"groups", () => RetemplateSearchModel2(new GroupSearchManager())},
                {"groupmembers", () => RetemplateSearchModel2(new GroupMemberManager())},
                {"groupbannedusers", () => RetemplateSearchModel2(new GroupBannedUserSearchManager())},
                {"externalmembers", () => RetemplateSearchModel2(new ExternalCommunityMemberManager())},
                {"characters", () => RetemplateSearchModel2(new CharacterFriendSearchManager())},
                {"usernames", () => RetemplateSearchModel2(new UsernameFriendSearchManager())},
                {"emails", () => RetemplateSearchModel2(new EmailFriendSearchManager())},
                {"platforms", () => RetemplateSearchModel2(new PlatformFriendSearchManager())},
                {"externalcommunities", () => RetemplateSearchModel2(new ExternalCommunitySearchManager())},
            };

            while (true)
            {
                Console.Clear();
                Console.WriteLine("Set Elastic Templates");
                foreach (var option in options)
                {
                    Console.WriteLine("- {0}", option.Key);
                }

                Console.WriteLine("- back");
                var choice = Console.ReadLine();

                if (choice.ToLower() == "back")
                {
                    break;
                }

                Action action;
                if (options.TryGetValue(choice, out action))
                {
                    action();
                }
                else
                {
                    Console.WriteLine("Unknown command: {0}", choice);
                }

                Console.WriteLine("Press enter to continue");
                Console.ReadLine();
            }
        }

        static void RetemplateSearchModel<T>() where T : class, new()
        {
            var modelAttribute = typeof(T).GetCustomAttribute<CloudSearchModelAttribute>();
            if (modelAttribute != null && !modelAttribute.UseDefaultIndex)
            {
                foreach (var regionConfigs in SearchConfiguration.Configurations.GroupBy(c => c.RegionKey))
                {
                    var templateName = modelAttribute.IndexTypeName + "-template";
                    var wildcardPath = modelAttribute.IndexTypeName + "-*";

                    Console.WriteLine("Setting up template {0} for {1} in region {2}", templateName, wildcardPath, regionConfigs.Key);

                    var connectionSettings = CloudSearchManager<T>.GetDefaultConnectionSettings(regionConfigs.ToArray());
                    var client = new ElasticClient(connectionSettings);

                    var existsResponse = client.TemplateExists(templateName);
                    if (!existsResponse.ConnectionStatus.Success)
                    {
                        Console.WriteLine("Connection to elastic failed");
                        Console.WriteLine(existsResponse.ServerError.Error);
                        continue;
                    }

                    if (existsResponse.Exists)
                    {
                        Console.Write("Template {0} already exists, do you want to replace it? (Y,N): ", templateName);
                        if (Console.ReadKey(false).Key != ConsoleKey.Y)
                        {
                            Console.WriteLine();
                            Console.WriteLine("Skipping");
                            return;
                        }
                        Console.WriteLine();
                    }

                    var putResponse = client.PutTemplate("", t => t
                        .Name(templateName)
                        .Template(wildcardPath)
                        .AddMapping<T>(d => d
                            .MapFromAttributes()
                            .SourceField(sf => sf.Enabled(true))
                            .AllField(af => af.Enabled(false))
                        ));

                    Console.WriteLine("Do you want to delete the index?");
                    if (Console.ReadKey(true).Key == ConsoleKey.Y)
                    {
                        Console.WriteLine("What index name?");
                        var indexName = Console.ReadLine();
                        Console.WriteLine("Are you sure you want to delete index: '" + indexName + "'?");
                        if (Console.ReadKey(true).Key == ConsoleKey.Y)
                        {
                            client.DeleteIndex(indexName);
                        }
                    }

                    if (!putResponse.ConnectionStatus.Success)
                    {
                        Console.WriteLine("Connection to elastic failed");
                        Console.WriteLine(putResponse.ServerError.Error);
                    }
                    else
                    {
                        Console.WriteLine("Replaced template {0}", templateName);
                    }
                }
            }
        }

        static void RetemplateSearchModel2<T>(CloudSearchManager<T> manager) where T : class, new()
        {
            var modelAttribute = typeof(T).GetCustomAttribute<CloudSearchModelAttribute>();
            if (modelAttribute != null && !modelAttribute.UseDefaultIndex)
            {
                foreach (var regionConfigs in SearchConfiguration.Configurations.GroupBy(c => c.RegionKey))
                {
                    var templateName = CloudSearchManager<T>.TemplateName;
                    Console.WriteLine("Setting up template {0} for {1} in region {2}", templateName, modelAttribute.IndexTypeName, regionConfigs.Key);

                    var connectionSettings = CloudSearchManager<T>.GetDefaultConnectionSettings(regionConfigs.ToArray());

                    var client = new ElasticClient(connectionSettings);
                    var existsResponse = client.TemplateExists(templateName);
                    if (!existsResponse.ConnectionStatus.Success)
                    {
                        Console.WriteLine("Connection to elastic failed");
                        Console.WriteLine(existsResponse.ServerError.Error);
                        continue;
                    }

                    if (existsResponse.Exists)
                    {
                        Console.Write("Template {0} already exists, do you want to replace it? (Y,N): ", templateName);
                        if (Console.ReadKey(false).Key != ConsoleKey.Y)
                        {
                            Console.WriteLine();
                            Console.WriteLine("Skipping");
                            return;
                        }
                        Console.WriteLine();
                    }

                    if (!manager.SetupTemplate(client))
                    {
                        Console.WriteLine("An error occurred while setting up the template");
                    }
                    else
                    {
                        Console.WriteLine("Replaced template {0}", templateName);
                    }

                    Console.WriteLine("Do you want to delete the index?");
                    if (Console.ReadKey(true).Key == ConsoleKey.Y)
                    {
                        Console.WriteLine("What index name?");
                        var indexName = Console.ReadLine();
                        Console.WriteLine("Are you sure you want to delete index: '" + indexName + "'?");
                        if (Console.ReadKey(true).Key == ConsoleKey.Y)
                        {
                            client.DeleteIndex(indexName);
                        }
                    }

                }
            }
        }

    }
}
