﻿using System;
using System.Collections.Concurrent;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using Curse.Friends.ReportingWebService.Configuration;
using Curse.Friends.S3;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Friends.ReportingWebService
{
    public class EventFilter
    {
        private readonly Regex[] _includeFilters;
        private readonly Regex[] _excludeFilters;
        private readonly ConcurrentDictionary<string, bool> _includeCache;

        public EventFilter(string[] includeFilters, string[] excludeFilters)
        {
            _includeCache = new ConcurrentDictionary<string, bool>();

            if (includeFilters != null)
            {
                _includeFilters = includeFilters.Select(f => new Regex(f, RegexOptions.Compiled)).ToArray();
            }

            if (excludeFilters != null)
            {
                _excludeFilters = excludeFilters.Select(f => new Regex(f, RegexOptions.Compiled)).ToArray();
            }
        }

        public bool ShouldInclude(EventReport report)
        {
            bool include;
            if (_includeCache.TryGetValue(report.EventKey, out include))
            {
                return include;
            }

            if (_includeFilters != null && _includeFilters.Length > 0)
            {
                if (!_includeFilters.Any(f => f.IsMatch(report.EventKey)))
                {
                    _includeCache.TryAdd(report.EventKey, false);
                    return false;
                }
            }

            if (_excludeFilters != null && _excludeFilters.Length > 0)
            {
                if (_excludeFilters.Any(f => f.IsMatch(report.EventKey)))
                {
                    _includeCache.TryAdd(report.EventKey, false);
                    return false;
                }
            }

            _includeCache.TryAdd(report.EventKey, true);
            return true;
        }
    }

    public class EventDump
    {
        #region S3

        private ConcurrentBag<EventReport> _currentReports = new ConcurrentBag<EventReport>();
        private ConcurrentBag<EventReport> _previousReports;

        private readonly string _name;
        private readonly LogCategory Logger;
        private readonly EventFilter _filter;


        public EventDump(string name, EventFilter filter)
        {
            _name = name;
            Logger = new LogCategory(name + " S3");
            _filter = filter;
        }

        public void AddEvent(EventReport report)
        {
            if (_filter.ShouldInclude(report))
            {
                _currentReports.Add(report);
            }
        }

        public void DumpToS3(DateTime timestamp)
        {
            try
            {
                _previousReports = _currentReports;
                _currentReports = new ConcurrentBag<EventReport>();
                ReportToS3(_previousReports.ToArray(), timestamp);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Error dumping event reports to S3");
            }
        }

        private void ReportToS3(EventReport[] report, DateTime timestamp)
        {
            if (report.Length == 0)
            {
                Logger.Trace("No events to dump to S3");
                return;
            }

            Logger.Trace(string.Format("Dumping {0} event reports to S3", report.Length));
            var serialized = JsonConvert.SerializeObject(report);
            using (var ms = new MemoryStream(Encoding.UTF8.GetBytes(serialized)))
            {
                var filename = string.Format("{0}_{1}_{2}.json", timestamp.ToString("yyyyMMddTHHmmssfffffffZ"), Environment.MachineName, _name);
                var fileKey = string.Format("{0:D4}/{1:D2}/{2:D2}/{3}", timestamp.Year, timestamp.Month, timestamp.Day, filename);
                var metadata = new FileMetadata(0, filename, (int)ms.Length, DateTime.UtcNow, "application/json");
                S3Manager.SaveFile(fileKey, ReportingWebServiceConfiguration.Current.EventBucket, metadata, ms);
            }
        }

        #endregion

    }
}