﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.ReportingWebService.Configuration;
using Curse.Friends.ReportingWebService.Contracts;
using Curse.Logging;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Curse.Friends.ReportingWebService
{
    public class EventManager
    {
        private static readonly BlockingCollection<Tuple<int, IPAddress, EventReportContract>> _reportQueue =
            new BlockingCollection<Tuple<int, IPAddress, EventReportContract>>(new ConcurrentQueue<Tuple<int, IPAddress, EventReportContract>>());
        private static readonly CancellationTokenSource _cts = new CancellationTokenSource();
        private static readonly LogCategory ThrottledLogger = new LogCategory("ReportingEventManager") { ReleaseLevel = LogLevel.Debug, Throttle = TimeSpan.FromMinutes(5) };

        private static EventDump[] _eventDumps;

        public static void Start()
        {
            _eventDumps = ReportingWebServiceConfiguration.Current.EventDumpConfigs.Select(c => new EventDump(c.Name, new EventFilter(c.IncludeFilters, c.ExcludeFilters))).ToArray();

            new Thread(ConsumeQueue) { IsBackground = true }.Start();
            new Thread(DumpToS3) { IsBackground = true }.Start();
        }

        public static void Stop()
        {
            _cts.Cancel();
            DoDumpToS3();
        }

        public static void ReportEvent(int userID, IPAddress ipAddress, EventReportContract eventInfo)
        {
            _reportQueue.Add(Tuple.Create(userID, ipAddress, eventInfo));
        }

        private static void ConsumeQueue()
        {
            while (!_cts.IsCancellationRequested)
            {
                try
                {
                    var eventInfo = _reportQueue.Take(_cts.Token);

                    // Run async instead of on a single thread
                    Task.Factory.StartNew(() => ProcessReport(eventInfo.Item1, eventInfo.Item2, eventInfo.Item3))
                        .ContinueWith(error =>
                        {
                            Logger.Error(error.Exception, "Error processing report from Task.");
                            ReportingStats.FailedProcessReporting();
                        }, TaskContinuationOptions.OnlyOnFaulted);
                }
                catch (OperationCanceledException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error running the event report consume queue");
                }
            }
        }

        private static void ProcessReport(int userID, IPAddress ipAddress, EventReportContract eventInfo)
        {
            var eventReport = new EventReport
            {
                EventKey = eventInfo.EventKey,
                AdditionalData = eventInfo.AdditionalData,
                Branch = eventInfo.Branch,
                ClientTime = eventInfo.ClientTime,
                ClientBuild = eventInfo.ClientBuild,
                ClientVersion = eventInfo.ClientVersion,
                Device = eventInfo.Device,
                DeviceID = eventInfo.DeviceID,
                DeviceModel = eventInfo.DeviceModel,
                DeviceSoftware = eventInfo.DeviceSoftware,
                EventCategory = string.IsNullOrWhiteSpace(eventInfo.EventCategory) ? "none" : eventInfo.EventCategory.ToLowerInvariant(),
                EventName = string.IsNullOrWhiteSpace(eventInfo.EventName) ? "none" : eventInfo.EventName.ToLowerInvariant(),
                Locale = eventInfo.Locale,
                IPAddress = ipAddress.ToString(),

                Timestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                UserID = userID,
                EventID = Guid.NewGuid(),
            };

            if (userID > 0)
            {
                var userStatistics = UserStatistics.GetByUserOrDefault(userID);
                if (!string.IsNullOrEmpty(userStatistics.Username))
                {
                    eventReport.Username = userStatistics.Username;
                }

                if (userStatistics.HasTwitchID)
                {
                    var linkedTwitchAccount = ExternalAccount.GetLocal(userStatistics.TwitchID, AccountType.Twitch);
                    long twitchUserID;
                    if (linkedTwitchAccount != null && long.TryParse(linkedTwitchAccount.ExternalID, out twitchUserID))
                    {
                        eventReport.TwitchUserID = twitchUserID;
                        eventReport.TwitchUsername = linkedTwitchAccount.ExternalUsername;
                    }
                }
            }

            foreach (var dump in _eventDumps)
            {
                dump.AddEvent(eventReport);
            }

            try
            {
                ReportToTwitch(eventReport);
            }
            catch (Exception ex)
            {

                Logger.Debug(ex, "Error reporting event to Twitch", eventReport);
            }
        }

        #region S3

        private static DateTime _lastDumpToS3;

        private static void DumpToS3()
        {
            while (!_cts.IsCancellationRequested)
            {
                try
                {
                    var now = DateTime.UtcNow;
                    if (now - _lastDumpToS3 > TimeSpan.FromSeconds(60))
                    {
                        _lastDumpToS3 = DateTime.UtcNow;
                        DoDumpToS3(now);
                    }
                    _cts.Token.WaitHandle.WaitOne(1000);
                }
                catch (OperationCanceledException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Error dumping event reports to S3");
                }
            }
        }

        private static void DoDumpToS3(DateTime? timestamp = null)
        {
            var now = timestamp ?? DateTime.UtcNow;
            Parallel.ForEach(_eventDumps, dump =>
            {
                dump.DumpToS3(now);
            });
        }

        #endregion

        #region Twitch

        private static readonly HttpClient _client = new HttpClient
        {
            DefaultRequestHeaders =
            {
                Referrer = new Uri("https://www.curse.com"),
            }
        };

        private static readonly System.Text.RegularExpressions.Regex _regex = new System.Text.RegularExpressions.Regex(@"\[(\d+)\]", System.Text.RegularExpressions.RegexOptions.Compiled);

        private static void FlattenProperties(JToken node, ref Dictionary<string, object> propertiesDictionary, bool arraysInPath = false)
        {
            switch (node.Type)
            {
                case JTokenType.Property:
                case JTokenType.Object:
                    foreach (var child in node.Children())
                    {
                        FlattenProperties(child, ref propertiesDictionary, arraysInPath);
                    }
                    break;
                case JTokenType.Array:
                    foreach (var child in node.Children())
                    {
                        FlattenProperties(child, ref propertiesDictionary, true);
                    }
                    break;
                default:
                    var path = arraysInPath ? _regex.Replace(node.Path, ".$1") : node.Path;
                    propertiesDictionary[path] = node;
                    break;
            }
        }

        private static void ReportToTwitch(EventReport report)
        {
            // Build out properties to send
            var propertiesDictionary = new Dictionary<string, object>();

            // Do additional properties first so they can't overwrite required fields
            if (report.AdditionalData != null)
            {
                FlattenProperties(JObject.FromObject(report.AdditionalData), ref propertiesDictionary);
            }

            propertiesDictionary["curse_user_id"] = report.UserID;
            propertiesDictionary["timestamp"] = report.ClientTime;
            propertiesDictionary["platform"] = report.Device;
            propertiesDictionary["device_id"] = report.DeviceID;
            propertiesDictionary["device_model"] = report.DeviceModel;
            propertiesDictionary["device_software"] = report.DeviceSoftware;
            propertiesDictionary["app_version"] = report.ClientVersion;
            propertiesDictionary["app_build"] = report.ClientBuild;
            propertiesDictionary["client_ip"] = report.IPAddress;
            propertiesDictionary["event_id"] = report.EventID;
            propertiesDictionary["branch"] = report.Branch;
            propertiesDictionary["locale"] = report.Locale;

            // Additional Info for Twitch
            if (report.Username != null)
            {
                propertiesDictionary["curse_login"] = report.Username;
            }

            if (report.TwitchUserID > 0)
            {
                propertiesDictionary["login"] = report.TwitchUsername;
                propertiesDictionary["user_id"] = report.TwitchUserID;
            }

            var eventObject = new
            {
                @event = report.EventKey,
                properties = propertiesDictionary
            };
            var serialized = JsonConvert.SerializeObject(eventObject);
            var base64 = Convert.ToBase64String(Encoding.UTF8.GetBytes(serialized));
            var body = new FormUrlEncodedContent(new Dictionary<string, string> { { "data", base64 } });
            ReportingStats.SpadeSubmissionCount();
            _client.PostAsync("https://spade.twitch.tv", body)
                .ContinueWith(task =>
                {
                    var result = task.Result;
                    if (result != null && !result.IsSuccessStatusCode)
                    {
                        ThrottledLogger.Warn("Bad response from spade: " + result.StatusCode,
                          new
                          {
                              statusCode = result.StatusCode,
                              responseBody = result.Content.ReadAsStringAsync().Result,
                              sendEvent = serialized
                          });
                        ReportingStats.FailedSpadeSubmission();
                    }
                });
        }

        #endregion
    }
}