﻿using Curse.Friends.ReportingWebService.Configuration;
using Curse.Friends.Statistics;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Web;

namespace Curse.Friends.ReportingWebService
{
    public static class ReportingStats
    {
        private static readonly LogCategory Logger = new LogCategory("ReportingStats") { AlphaLevel = LogLevel.Info };
        private static Timer timer = new Timer(LogData, null, Timeout.Infinite, Timeout.Infinite);
        private static TimeSpan interval = TimeSpan.FromMinutes(ReportingWebServiceConfiguration.Current.StatsIntervalMinutes);

        private static ReportingStatCounter _spadeSubmissionCount = new ReportingStatCounter();
        private static ReportingStatCounter _spadeFailureCount = new ReportingStatCounter();

        private static ReportingStatCounter _reportRequestsCount = new ReportingStatCounter();
        private static ReportingStatCounter _reportsCount = new ReportingStatCounter();

        private static ReportingStatCounter _failedReportProcessing = new ReportingStatCounter();


        public static void Start()
        {            
            timer.Change(interval, interval);
        }

        private static void LogData(object state)
        {
            try
            {
                var spadeSubmissionSnapshot = _spadeSubmissionCount.GetSnapshot();
                var spadeFailureSnapshot = _spadeFailureCount.GetSnapshot();
                var reportRequestsSnapshot = _reportRequestsCount.GetSnapshot();
                var reportsSnapshot = _reportsCount.GetSnapshot();
                var failedReportProcessing = _failedReportProcessing.GetSnapshot();

                var totalAvgBatch = SafeDivide(reportsSnapshot.TotalCount, reportRequestsSnapshot.TotalCount);
                var snapshotAvgBatch = SafeDivide(reportsSnapshot.SnapshotCount, reportRequestsSnapshot.SnapshotCount);

                var totalSpadeFail = SafeDivide(spadeFailureSnapshot.TotalCount, spadeSubmissionSnapshot.TotalCount);
                var snapshotSpadeFail = SafeDivide(spadeFailureSnapshot.SnapshotCount, spadeSubmissionSnapshot.SnapshotCount);

                Logger.Info("Reporting service stats",
                    new
                    {
                        Aggregates = new
                        {
                            TotalAverageBatchSize = totalAvgBatch,
                            SnapshotAverageBatchSize = snapshotAvgBatch,
                            TotalSpadeFailuresPercent = totalSpadeFail,
                            SnapshotSpadeFailuresPercent = snapshotSpadeFail,
                        },

                        ReportRequests = reportRequestsSnapshot,
                        Reports = reportsSnapshot,
                        ProcessReportFail = failedReportProcessing,
                        SpadeSubmissions = spadeSubmissionSnapshot,
                        SpadeFailures = spadeFailureSnapshot,                        
                    });
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to log reporting stats");
            }
        }

        private static double SafeDivide(long numerator, long denominator)
        {
            if (denominator == 0)
            {
                return 0;
            }

            return numerator / (double)denominator;
        }

        public static void BatchCount()
        {
            _reportRequestsCount.Track();
        }

        public static void BatchItemCount()
        {
            _reportsCount.Track();
        }

        public static void SpadeSubmissionCount()
        {
            _spadeSubmissionCount.Track();
        }

        public static void FailedSpadeSubmission()
        {
            _spadeFailureCount.Track();
        }

        public static void FailedProcessReporting()
        {
            _failedReportProcessing.Track();
        }

        internal static void Stop()
        {
            timer.Change(Timeout.Infinite, Timeout.Infinite);
            timer.Dispose();
        }
    }
}