﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceProcess;
using Curse.CloudQueue;
using Curse.Friends.Configuration;
using Curse.Friends.ServiceClients;
using Curse.Logging;
using Curse.Logging.Uploader;
using Curse.Friends.Statistics;
using Curse.Friends.Statistics.Models;

namespace Curse.Friends.ServerHosting
{
    public abstract class CurseWindowsService : ServiceBase
    {
        protected abstract ServiceHostType HostType { get; }

        protected abstract ConfigurationServices ConfigurationServices { get; }

        public void OnDebugStart()
        {
            OnStart(null);
        }

        public void OnDebugStop()
        {
            OnStop();
        }

        protected abstract void CustomStartup();
        protected abstract void CustomStop();

        IEnumerable<FriendsQueueStats> GetQueueStats()
        {
            return CloudQueueInstanceManager.GetQueueStats().Select(p => new FriendsQueueStats
            {
                QueueName = p.QueueName,
                TotalMessagesProcessed = p.TotalMessagesProcessed,
                CurrentlyProcessing = p.CurrentlyProcessing,
                AverageProcessingTimeMs = (int)p.AverageProcessTime.TotalMilliseconds,
                BestProcessingTimeMs = (int)p.BestProcessTime.TotalMilliseconds,
                WorstProcessingTimeMs = (int)p.WorstProcessTime.TotalMilliseconds,
                AverageQueuedTimeMs = (int)p.AverageQueuedTime.TotalMilliseconds,
                BestQueuedTimeMs = (int)p.BestQueuedTime.TotalMilliseconds,
                WorstQueuedTimeMs = (int)p.WorstQueuedTime.TotalMilliseconds
            }).ToArray();
        }

        protected override void OnStart(string[] args)
        {
            AppDomain.CurrentDomain.UnhandledException += CurrentDomain_UnhandledException;

            try
            {
                Logger.Init(@"C:\Curse\Logs", HostType.ToString());
                LogUploader.Initialize((int) HostType, FriendsServiceConfiguration.Instance.CentralLogServiceUrl, FriendsServiceConfiguration.Instance.LoggingServiceApiKey);

                Logger.Info("Determining current region...");
                var currentRegion = StorageConfiguration.GetCurrentRegion();

                if (currentRegion == null)
                {
                    Logger.Error("Failed to determine current region!");
                }
                else
                {
                    Logger.Info("Current region detected: " + currentRegion.Key);
                }
                
                Logger.Info("Initializing service client...");                
                FriendsServiceClients.Initialize(FriendsServiceConfiguration.Instance.EncryptionKey, FriendsServiceConfiguration.Instance.EncryptionIterations, FriendsServiceConfiguration.Instance.CentralServiceApiKey);

                Logger.Info("Initializing stats manager...");
                FriendsStatsManager.Initialize((int)HostType, HostType.ToString(), currentRegion != null ? currentRegion.ID : 1, GetQueueStats);

                FriendsStatsManager.BeginStartup();
                StorageConfiguration.Initialize(HostType.ToString(), null, null, ConfigurationServices);
                Logger.Info(HostType + " is starting.");
                CustomStartup();
                Logger.Info(HostType + " started!");
                FriendsStatsManager.Started();
                
            }
            catch (Exception ex)
            {
                Logger.Error(ex, HostType + " failed to start due to an unhandled exception.");
                throw;
            }
        }

        protected override void OnStop()
        {
            Logger.Info(HostType + " is stopping.");
            try
            {
                RequestAdditionalTime((int) TimeSpan.FromSeconds(30).TotalMilliseconds);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, HostType + " failed to request additional time to stop.");
            }

            FriendsStatsManager.BeginShutdown();

            try
            {
                CustomStop();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, HostType + " did not stop cleanly due to an unhandled exception.");
            }

            try
            {
                StorageConfiguration.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while shutting down storage configuration.");

            }

            try
            {
                FriendsStatsManager.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while shutting down stats manager.");
            }
            
            Logger.Info(HostType + " stopped!");
            LogUploader.Shutdown();
        }

        void CurrentDomain_UnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            var exception = e.ExceptionObject as Exception;
            Logger.Error(exception, HostType + " has crashed!");
        }
    }
}
