﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Timers;
using Curse.Aerospike;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Logging;
using System.Threading;

namespace Curse.Friends.ServerHosting
{
    public abstract class ServerHost<THost, THostable> where THost : BaseTable<THost>, IServiceHost, new() where THostable : BaseTable<THostable>, IHostable, new()
    {
        private readonly LogCategory Logger = new LogCategory("ServiceHost");
        private readonly LogCategory DiagLogger = new LogCategory("ServiceHost") { Throttle = TimeSpan.FromSeconds(10) };

        protected abstract void CustomStartup();

        protected virtual void EnsureHosted(THostable hostable)
        {

        }

        protected virtual void EnsureUnhosted(THostable hostable)
        {

        }

        protected virtual void AfterHostRegistered()
        {

        }

        protected abstract void CustomStop();

        protected void EnsureAllHosted()
        {

            var allHostables = BaseTable<THostable>.GetAllLocal(s => s.RegionID, ExternalCommunity.LocalConfigID);

            // Ensure that all hostables that think they are hosted by this node actually are
            var localHostables = allHostables.Where(p => !string.IsNullOrWhiteSpace(p.MachineName) && p.MachineName.Equals(Environment.MachineName)).ToArray();

            foreach (var hostable in localHostables)
            {
                try
                {
                    EnsureHosted(hostable);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to host.", new { hostable.DisplayName });
                }
            }

            // Ensure that all hostables that should not be hosted by this machine are actually not
            var remoteHostables = allHostables.Where(p => !string.IsNullOrWhiteSpace(p.MachineName) && !p.MachineName.Equals(Environment.MachineName)).ToArray();

            foreach (var hostable in remoteHostables)
            {
                try
                {
                    EnsureUnhosted(hostable);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to host.", new { hostable.DisplayName });
                }
            }


            var allOrphanedHostables = allHostables.Where(m => string.IsNullOrWhiteSpace(m.MachineName)).ToArray();

            if (!allOrphanedHostables.Any())
            {
                return;
            }

            foreach (var orphaned in allOrphanedHostables)
            {
                ServiceHostManager<THost, THostable>.EnsureServiceHost(orphaned);
            }

        }



        protected THost _host;
        private readonly List<Thread> _timers = new List<Thread>();
        private readonly CancellationTokenSource _cts = new CancellationTokenSource();
        private readonly LogCategory TimerLogger = new LogCategory("Timers") { Throttle = TimeSpan.FromMinutes(5), ReleaseLevel = LogLevel.Debug };

        public void Start()
        {
            CustomStartup();

            AddTimer("UpdateHostActivity", TimeSpan.FromSeconds(15), UpdateHostActivity);
            AddTimer("FixOrphans", TimeSpan.FromSeconds(30), FixOrphans);

            RegisterHost();

            AfterHostRegistered();
        }

        protected void AddTimer(string name, TimeSpan interval, Action elapsedAction, bool completeBeforeContinue = true)
        {
            var thread = new Thread(() => TimerThread(name, interval, elapsedAction, completeBeforeContinue));
            _timers.Add(thread);
            thread.Start();
        }

        private void TimerThread(string name, TimeSpan interval, Action elapsedAction, bool completeBeforeContinue)
        {
            var lastRun = DateTime.UtcNow;
            var intervalMillis = (int) interval.TotalMilliseconds;
            while (!_cts.IsCancellationRequested)
            {
                var timeToWait = intervalMillis;
                try
                {
                    TimerLogger.Debug("Running timer: " + name);
                    elapsedAction();
                }
                catch (Exception ex)
                {
                    TimerLogger.Error(ex, "Failed to run timer: " + name);
                }
                finally
                {
                    if (!completeBeforeContinue)
                    {
                        timeToWait = Math.Max(0, intervalMillis - (int)(DateTime.UtcNow - lastRun).TotalMilliseconds);
                        lastRun = DateTime.UtcNow;
                    }
                }

                try
                { 
                    _cts.Token.WaitHandle.WaitOne(timeToWait);
                }
                catch (OperationCanceledException)
                {
                    break;
                }
            }
        }


        protected virtual void RegisterHost()
        {
            var hostEnvironment = (HostEnvironment)Enum.Parse(typeof(HostEnvironment), FriendsServiceConfiguration.Mode.ToString(), true);
            var internalIP = NetworkHelper.GetInternalIpAddress();

            var host = BaseTable<THost>.GetLocal(internalIP);
            if (host == null)
            {
                host = new THost
                {
                    InternalIPAddress = internalIP,
                    DateCreated = DateTime.UtcNow,
                    DateOnline = DateTime.UtcNow,
                    DateUpdated = DateTime.UtcNow,
                    MachineName = Environment.MachineName,
                    Status = ServiceHostStatus.Online,
                    Environment = hostEnvironment,
                };
                host.Version = host.CurrentVersion;
                host.InsertLocal();
                Logger.Info("Registered new service host", new { host.MachineName, host.DateOnline, host.DateUpdated, host.Status });
            }
            else
            {
                host.DateUpdated = DateTime.UtcNow;
                host.DateOnline = DateTime.UtcNow;
                host.Status = ServiceHostStatus.Online;
                host.Environment = hostEnvironment;
                host.Version = host.CurrentVersion;
                host.MachineName = Environment.MachineName;
                host.Update();
                Logger.Info("Updated existing service host", new { host.MachineName, host.DateOnline, host.DateUpdated, host.Status });
            }

            _host = host;            
        }

        protected virtual void FixOrphans()
        {
            var staleHosts = BaseTable<THost>.GetAllLocal(p => p.IndexMode, IndexMode.Default)
                .Where(p => (DateTime.UtcNow - p.DateUpdated > TimeSpan.FromMinutes(1))
                            && !p.IsCleaning
                            && !p.MachineName.Equals(Environment.MachineName)).ToArray();

            try
            {
                var onlineStaleHosts = staleHosts.Where(p => p.Status == ServiceHostStatus.Online).ToArray();

                // Quickly mark all of that as unhealthy
                foreach (var serviceHost in onlineStaleHosts)
                {
                    Logger.Warn("Detected unhealthy host: " + serviceHost.MachineName);

                    serviceHost.Status = ServiceHostStatus.Unhealthy;
                    serviceHost.DateUnhealthy = DateTime.UtcNow;
                    serviceHost.IsCleaning = true;
                    serviceHost.Update(p => p.Status, p => p.DateUnhealthy, p => p.IsCleaning);
                }

                // Get all streams hosted by these hosts, and fail them over
                foreach (var serviceHost in staleHosts)
                {
                    var hostables = BaseTable<THostable>.GetAllLocal(p => p.MachineName, serviceHost.MachineName);
                    if (hostables.Any())
                    {
                        Logger.Info(string.Format("Unhosting {0} hostables from unhealthy host '{1}'", hostables.Length, serviceHost.MachineName));

                        foreach (var hostable in hostables)
                        {
                            hostable.MachineName = string.Empty;
                            hostable.Update(p => p.MachineName);
                            DiagLogger.Info("Detected orphaned hostable on " + serviceHost.MachineName);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                var cleaningHosts = staleHosts.Where(p => p.IsCleaning).ToArray();

                foreach (var serviceHost in cleaningHosts)
                {
                    serviceHost.IsCleaning = false;
                    serviceHost.Update(p => p.IsCleaning);
                }
            }
        }

        private void UpdateHostActivity()
        {
            try
            {                
                if (_host == null)
                {
                    Logger.Warn("Unable to update host activity. Host is null.");
                    return;
                }

                if (_host.Status == ServiceHostStatus.Offline)
                {
                    Logger.Warn("Unable to update host activity. Host thinks it is offline.");
                    return;
                }

                _host.DateUpdated = DateTime.UtcNow;
                _host.Status = ServiceHostStatus.Online;
                _host.Update(p => p.Status, p => p.DateUpdated);

                Logger.Trace("Updated host activity", _host);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while updating Host Activity.");
            }
        }

        private void MarkHostOffline()
        {
            if (_host == null)
            {
                return;
            }

            _host.Status = ServiceHostStatus.Offline;
            _host.Update(p => p.Status);
        }

        public void Stop()
        {
            try
            {
                _cts.Cancel();
                foreach (var timer in _timers.ToArray())
                {
                    timer.Join();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while stopping timers.");
            }

            try
            {
                MarkHostOffline();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while marking host offline.");
            }

            try
            {
                CustomStop();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception during custom stop code.");
            }
        }

    }
}
