﻿using System;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.CloudServices.Models;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.SessionsWebService.Configuration;
using Curse.Friends.SessionsWebService.Contracts;
using Curse.Logging;

namespace Curse.Friends.SessionsWebService.Controllers
{
    [RoutePrefix("sessions")]
    public class SessionController : MicroServiceController
    {
        private static readonly LogCategory Logger = new LogCategory("SessionController") { BetaLevel = LogLevel.Trace };

        private ClientEndpoint CreateNewEndpoint(CreateSessionRequest request, bool issueNewMachineKey)
        {
            var endpoint = new ClientEndpoint
            {
                UserID = Token.UserID,
                MachineKey = issueNewMachineKey ? Guid.NewGuid().ToString() : request.MachineKey.ToString(),
                SessionDate = DateTime.UtcNow,
                IsConnected = false,
                RegionID = ClientEndpoint.LocalConfigID,
                SessionID = Guid.NewGuid().ToString(),
                Platform = request.Platform,
                DeviceID = request.DeviceID,
                PushKitToken = request.PushKitToken,
                IPAddress = GetCurrentIpAddress().ToString()
            };

            endpoint.InsertLocal();

            return endpoint;
        }

        private ClientEndpoint FindSuitableAlternate(CreateSessionRequest request)
        {
            
            var alternateEndpoint = ClientEndpoint.GetAllLocal(p => p.UserID, Token.UserID)
                                                   .FirstOrDefault(p => p.Platform == request.Platform
                                                                    && !p.IsDeleted
                                                                    && !p.IsConnected);


            if (alternateEndpoint != null)
            {
                Logger.Trace("FindSuitableAlternate: Found an alternate endpoint.", new { Token.UserID, alternateEndpoint.MachineKey });
                return alternateEndpoint;
            }

            Logger.Trace("FindSuitableAlternate: Creating a new client endpoint.", new { Token.UserID });
            return CreateNewEndpoint(request, true);
        }

        /// <summary>
        /// Creates a new session given a platform, machine key and optional device and push kit token (for mobile clients).
        /// Writes a session cookie for access to all micro-services on the domain.
        /// </summary>
        [HttpPost]
        [Route("")]
        [ResponseType(typeof(CreateSessionResponse))]
        public IHttpActionResult Create(CreateSessionRequest request)
        {
            request.Validate();
                        
            var userAndRegion = RegisterUser(Token.UserID, Token.Username);

            // Register the client endpoint
            var endpoint = ClientEndpoint.GetLocal(Token.UserID, request.MachineKey.ToString());

            if (endpoint == null) // Create a new endpoint and session
            {
                endpoint = CreateNewEndpoint(request, false);
            }
            else if (endpoint.IsConnected) // Assign a new endpoint, or use one of the specified alternate keys 
            {
                endpoint = FindSuitableAlternate(request);
            }
            else // Endpoint already exists, so just update it
            {
                if (DateTime.UtcNow - endpoint.SessionDate > TimeSpan.FromDays(1)) // Issue a new session ID
                {
                    endpoint.UpdateSession(Guid.NewGuid().ToString());
                }

                var clientIP = GetCurrentIpAddress().ToString();
                if (endpoint.IPAddress != clientIP)
                {
                    endpoint.IPAddress = clientIP;
                    endpoint.Update(e => e.IPAddress);
                }

                // If the endpoint was previously pruned, restore it
                if (endpoint.IsDeleted)
                {
                    endpoint.IsDeleted = false;
                    endpoint.Update(p => p.IsDeleted);
                }

                // Ensure that this endpoint is indexed
                endpoint.ValidateMap();
            }

            if (endpoint.Platform != request.Platform)
            {
                ClientEndpoint.UpdatePlatform(request.Platform, Token.UserID, request.MachineKey.ToString());
            }

            if (endpoint.DeviceID != request.DeviceID)
            {
                Logger.Trace("Updating endpoint device ID", endpoint);
                ClientEndpoint.UpdateDeviceID(request.DeviceID, Token.UserID, request.MachineKey.ToString());
            }

            if (endpoint.PushKitToken != request.PushKitToken)
            {
                Logger.Trace("Updating endpoint Voice Token", new { endpoint, request.PushKitToken });
                ClientEndpoint.UpdatePushKitToken(request.PushKitToken, Token.UserID, request.MachineKey.ToString());
            }

            DevicePlatform[] platforms;

            // Get the user's platforms
            try
            {
                platforms = ClientEndpoint.GetAllPlatforms(Token.UserID);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to retrieve user's device platforms, after creating a session.");
                platforms = new DevicePlatform[0];
            }

            var externalAccount = userAndRegion.User.IsMerged ? ExternalAccount.GetByTwitchUserID(userAndRegion.User.TwitchID) : null;
            var sessionResponse = new CreateSessionResponse
            {
                MachineKey = endpoint.MachineKey,
                SessionID = endpoint.SessionID,
                User = userAndRegion.User.ToNotification(externalAccount),
                Platforms = platforms,                
                NotificationServiceUrl = SessionsWebServiceConfiguration.Current.NotificationServiceUrl,
                TwitchID = externalAccount?.ExternalID,
                TwitchAuthToken = externalAccount?.AuthToken
            };

            // Write this value out as a cookie
            var cookie = GetSessionCooke();
            var response = Request.CreateResponse(HttpStatusCode.OK, sessionResponse);
            response.Headers.AddCookies(new[] { cookie });
            return ResponseMessage(response);
        }

        /// <summary>
        /// Removes the session cookie.
        /// </summary>
        [HttpDelete]
        [Route("")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult Logout()
        {
            // Write this value out as a cookie
            var cookie = new CookieHeaderValue(AuthenticationToken.HttpCookieName, string.Empty)
            {
                Expires = DateTimeOffset.Now.AddYears(-1),
                HttpOnly = true
            };

            var response = Request.CreateResponse(HttpStatusCode.OK);
            response.Headers.AddCookies(new[] { cookie });
            return ResponseMessage(response);
        }

        /// <summary>
        /// Updates an existing session's device tokens (for mobile endpoints only)
        /// </summary>
        [HttpPost]
        [Route("{machineKey}/tokens")]
        public IHttpActionResult Update(Guid machineKey, UpdateDeviceTokensRequest request)
        {
            if (!ClientEndpoint.UpdateDeviceTokens(request.DeviceID, request.PushKitID, Token.UserID, machineKey))
            {
                return NotFound();
            }

            return Ok();
        }

        /// <summary>
        /// Deletes an existing session's device tokens, given a machine key and device ID (for mobile endpoints only)
        /// </summary>
        [HttpDelete]
        [Route("{machineKey}/tokens/{deviceID}")]
        public IHttpActionResult Delete(Guid machineKey, string deviceID)
        {

            var endpoint = ClientEndpoint.GetLocal(Token.UserID, machineKey.ToString());

            if (endpoint == null)
            {
                return NotFound();
            }

            if (endpoint.DeviceID != deviceID)
            {
                return NotFound();
            }

            // Update any endpoints with this device ID, for this user
            var allDeviceEndpoints = ClientEndpoint.GetAllLocal(p => p.UserID, Token.UserID).Where(p => p.DeviceID == deviceID);

            Logger.Trace("Unregistering endpoint", new { machineKey, deviceID });

            foreach (var deviceEndpoint in allDeviceEndpoints)
            {
                deviceEndpoint.DeviceID = null;
                deviceEndpoint.PushKitToken = null;
                deviceEndpoint.Update(p => p.DeviceID, p => p.PushKitToken);
            }

            return Ok();
        }

        private UserRegionInfo RegisterUser(int userID, string username)
        {
            // From their storage region, get their record and update their status            
            var userAndRegion = GetUserAndRegion(userID, true);
            var reindexSearch = false;

            if (userAndRegion.User == null) // If it doesn't exist, insert or replace
            {
                userAndRegion.User = new User
                {
                    UserID = userID,
                    Username = username,
                    IsTempAccount = TempAccount.IsTempAccount(userID),
                    GroupMessagePushPreference = PushNotificationPreference.Favorites,
                    FriendMessagePushPreference = PushNotificationPreference.All,
                    FriendRequestPushEnabled = true
                };
                
                userAndRegion.User.Insert(userAndRegion.Region.RegionID);
                userAndRegion.User = Curse.Friends.Data.User.Get(userAndRegion.Region.RegionID, userID);

                if (userAndRegion.User == null)
                {
                    throw new InvalidOperationException("Failed to retrieve user after inserting it.");
                }

                reindexSearch = true;
            }
            
            // Upsert their friend hints
            if (reindexSearch || FriendsServiceConfiguration.Instance.AlwaysIndexUserSearch)
            {
                try
                {
                    if (!string.IsNullOrEmpty(username))
                    {
                        new FriendHint() { UserID = userID, Type = FriendHintType.Username, SearchTerm = username, AvatarUrl = userAndRegion.User.AvatarUrl, Verification = FriendHintVerification.Verified }.InsertLocal();
                    }                   

                    // Queue up a job to reindex the user
                    FriendHintSearchIndexer.CreateForUser(userAndRegion.User.UserID);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to create friend hint search indexer for registering user.");
                }
            }

            return userAndRegion;
        }

        private CookieHeaderValue GetSessionCooke()
        {
            var cookie = new CookieHeaderValue(AuthenticationToken.HttpCookieName, Token.Token)
            {
                Domain = "." + FriendsServiceConfiguration.Instance.AuthTokenCookieDomain,
                Expires = DateTimeOffset.Now.AddMonths(1),
                HttpOnly = true, 
                Secure = true
            };

            return cookie;
        }

        [HttpPost]
        [Route("{machineKey}/group")]
        public IHttpActionResult UpdateCurrentGroup(Guid machineKey, UpdateCurrentGroupRequest request)
        {
            var user = GetCurrentUserAndRegion();
            var ep = ClientEndpoint.GetLocal(user.Region.UserID, machineKey);
            if (ep == null || !ep.IsConnected)
            {
                return NotFound();
            }

            ep.UpdateCurrentGroup(user.Region, request.GroupID);

            return StatusCode(HttpStatusCode.NoContent);
        }
    }
}