﻿using System.Net;
using Curse.Friends.Statistics.Models;
using Curse.Logging;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Curse.Friends.ServiceClients;
using HostStatus = Curse.ServiceClients.Contracts.HostStatus;
using StatsPayload = Curse.ServiceClients.Contracts.StatsPayload;
using HostStatusContract = Curse.ServiceClients.Contracts.HostStatusContract;

namespace Curse.Friends.Statistics
{
    public static class FriendsStatsManager
    {
        private static readonly LogCategory Logger = new LogCategory("FriendsStatsManager");
        private static readonly List<FriendsStats> _historicalStats = new List<FriendsStats>();
        private static readonly List<FriendsRegionStats> _regionalStats = new List<FriendsRegionStats>();
        private static FriendsStats _currentStats;
        private static FriendsStats _previousStats;
        private static DateTime _currentInterval;
        private static readonly object _syncRoot = new object();
        private static bool _isInitialized = false;
        private static int _hostTypeID;
        private static string _hostTypeName;
        private static bool _isDisabled = false;
        private static Func<IEnumerable<FriendsQueueStats>> GetQueueStats;

        public static FriendsStats Current
        {
            get
            {
                return _currentStats ?? new FriendsStats(GetQuarter());
            }
        }
        
        public static void Disable()
        {
            Logger.Info("Stats manager has been disabled.");
            _isDisabled = true;
        }

        public static int StatsRegionID { get; set; }

        public static void Initialize(int hostTypeID, string hostTypeName, int hostRegionID, Func<IEnumerable<FriendsQueueStats>> getQueueStats = null, HostCounter counters = HostCounter.Default)
        {
            if (_isDisabled)
            {
                return;
            }

            lock (_syncRoot)
            {
                if (_isInitialized)
                {
                    return;
                }

                if (FriendsServiceClients.Instance.Stats == null)
                {
                    Logger.Warn("Stats will not be uploaded! The stats service client failed to be created.");
                    _isDisabled = true;
                    return;
                }
                                                                 
                StatsRegionID = hostRegionID;
                GetQueueStats = getQueueStats;
                
                Logger.Info("Stats have been initialized. They will be uploaded to: " + FriendsServiceClients.Instance.Stats.Url, new { hostTypeID, hostTypeName, hostRegionID, queueStats = getQueueStats != null, counters });
                

                ServicePointManager.Expect100Continue = false;                
                _hostTypeID = hostTypeID;
                _hostTypeName = hostTypeName;
                _currentInterval = GetQuarter();
                _currentStats = new FriendsStats(_currentInterval);
                Task.Factory.StartNew(ProcessStats, TaskCreationOptions.LongRunning);
                Task.Factory.StartNew(PublishStats, TaskCreationOptions.LongRunning);
                HostPerformanceMonitor.Initialize(counters);
                _isInitialized = true;
            }
        }

        static void ProcessStats()
        {
            while (true)
            {
                try
                {
                    Thread.Sleep(1000);
                    var newInterval = GetQuarter();
                    if (newInterval.Equals(_currentInterval))
                    {
                        continue;
                    }

                    lock (_syncRoot)
                    {
                        _previousStats = _currentStats;
                        _historicalStats.Add(_previousStats);
                        _currentInterval = newInterval;
                        _currentStats = new FriendsStats(newInterval, _previousStats);
                    }
                }
                catch (ThreadAbortException)
                {

                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to process stats!");
                }
            }
        }

        private static volatile bool _isShuttingDown = false;

        static void PublishStats()
        {
            while (!_isShuttingDown)
            {
                try
                {
                    Thread.Sleep(1000);

                    if (_isShuttingDown)
                    {
                        return;
                    }

                    // Update queue stats
                    if (GetQueueStats != null)
                    {
                        Current.QueueStats = GetQueueStats();
                    }
                    else
                    {
                        Current.QueueStats = new FriendsQueueStats[0];
                    }
                    

                    try
                    {
                        PublishRealtimeStats();
                    }
                    catch (Exception ex)
                    {
                        if (ex is AggregateException)
                        {
                            ex = (ex as AggregateException).Flatten();
                        }
                        Logger.Error(ex, "Failed to publish realtime stats");
                    }

                    try
                    {
                        PublishPeriodicStats();
                    }
                    catch (Exception ex)
                    {
                        if (ex is AggregateException)
                        {
                            ex = (ex as AggregateException).Flatten();
                        }
                        Logger.Error(ex, "Failed to publish periodic stats");
                    }

                    try
                    {
                        PublishRegionStats();
                    }
                    catch (Exception ex)
                    {
                        if (ex is AggregateException)
                        {
                            ex = (ex as AggregateException).Flatten();
                        }
                        Logger.Error(ex, "Failed to publish region stats");
                    }

                }
                catch (ThreadAbortException)
                {

                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to publish stats!");
                }
            }
        }

        public static void BeginShutdown()
        {            
            UpdateHostStatus(HostStatus.ShuttingDown);
        }

        public static void BeginStartup()
        {
            UpdateHostStatus(HostStatus.Starting);
        }

        public static void Started()
        {
            UpdateHostStatus(HostStatus.Started);
        }

        public static void Shutdown()
        {
            _isShuttingDown = true;                       

            try
            {
                ClearRealtimeStats();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to clear realtime stats!");
            }

            try
            {
                var currentStats = _currentStats;
                if (currentStats == null)
                {
                    return;
                }

                lock (_syncRoot)
                {
                    _historicalStats.Add(currentStats);
                }

                PublishPeriodicStats();
                PublishRegionStats();
                UpdateHostStatus(HostStatus.Offline);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to complete shutdown.");
            }
        }
        
        private static void UpdateHostStatus(HostStatus status)
        {
            if (!_isInitialized)
            {
                return;
            }

            try
            {
                var resp = FriendsServiceClients.Instance.Stats.Status(new HostStatusContract
                {
                    HostTypeID = _hostTypeID,
                    RegionID = StatsRegionID,
                    HostTypeName = _hostTypeName,
                    HostStatus = status,
                    HostName = Environment.MachineName
                });

                if (!resp.Success)
                {
                    Logger.Warn("Failed to update host status: " + resp.StatusCode);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update host status");
            }           

        }
               
        private static void PublishPeriodicStats()
        {         
            FriendsStats[] pendingStats = null;
            lock (_syncRoot)
            {
                if (_historicalStats.Any())
                {
                    pendingStats = _historicalStats.ToArray();
                    _historicalStats.Clear();
                }
            }

            if (pendingStats == null || pendingStats.Length == 0)
            {
                return;
            }

            foreach (var stats in pendingStats)
            {
                try
                {
                    var periodicStats = CreatePeriodicStats(stats);
                    var serialized = JsonConvert.SerializeObject(periodicStats);
                    var resp = FriendsServiceClients.Instance.Stats.Periodic(new StatsPayload { HostName = Environment.MachineName, HostTypeID = _hostTypeID, HostTypeName = _hostTypeName, StatsJson = serialized, RegionID = StatsRegionID });
                    if (!resp.Success)
                    {
                        Logger.Warn("Failed to publish periodic stats. They will be re-added to the pending list.", resp);
                        lock (_syncRoot)
                        {
                            _historicalStats.Add(stats);
                        }
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "General failure publishing periodic stats!");
                }
            }

            Logger.Trace("Publishing periodic stats.");

        }

        private static void TrySubmitRealtimeStats(string json, int attempt = 1, int maxAttempts = 3)
        {
            try
            {
                var req = new StatsPayload
                {
                    HostName = Environment.MachineName,
                    HostTypeID = _hostTypeID,
                    HostTypeName = _hostTypeName,
                    StatsJson = json,
                    RegionID = StatsRegionID
                };

                var resp = FriendsServiceClients.Instance.Stats.Realtime(req);

                if (!resp.Success)
                {
                    var currentAttempt = attempt;
                    if (attempt++ < maxAttempts)
                    {
                        // Only log these issues after the first attempt
                        if (attempt > 1)
                        {
                            Logger.Warn("Stats submission failed on attempt " + currentAttempt + ". Retrying...", new {Request = req, Response = resp});
                        }

                        Thread.Sleep(250);
                        TrySubmitRealtimeStats(json, attempt);
                    }
                    else
                    {
                        Logger.Error("Stats submission failed after " + attempt + " attempts", resp);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to submit stats due to a general exception");
            }
        }

        private static FriendsHostStats CreatePeriodicStats(FriendsStats stats)
        {
            var periodicStats = new FriendsHostStats()
            {
                StartDate = stats.StartDate,
                FriendshipsConfirmed = stats.FriendRequestsConfirmed.Counter,
                FriendshipsDeclined = stats.FriendRequestsDeclined.Counter,
                FriendshipsRemoved = stats.FriendRequestsRemoved.Counter,
                FriendshipsRequested = stats.FriendRequestsSent.Counter,
                IdentitiesCreated = stats.IdentitiesCreated.Counter,
                NewMessages = stats.MessagesSent.LifetimeCounter,
                NewMessagesByPlatform = stats.MessagesSentByPlatform.Values,
                PeakConnections = stats.PeakConnections.Amount,
                GameFriendSuggestionsCreated = stats.GameSuggestions.Counter,
                MutalFriendSuggestionsCreated = stats.MutualFriendSuggestions.Counter,
                PlatformFriendSuggestionsCreated = stats.PlatformSuggestions.Counter,
                SuggestionsDeclined = stats.SuggestionsDeclined.Counter,
                TotalProfilesViewed = stats.ProfilesViewed.Counter,
                UniqueProfilesViewed = stats.ProfilesViewed.UniqueCounter,
                SearchesPerformed = stats.SearchesPerformed.Counter,
                GameNotifications = stats.GameNotifications.Values,
                ConcurrentConnections = stats.CurrentConnections.Amount,
                MemoryUsageKilobytes = HostPerformanceMonitor.MemoryUsageKilobytes,
                ProcessorUtilization = HostPerformanceMonitor.ProcessorUtilization,
                ReceivedBitsPerSecond = HostPerformanceMonitor.ReceivedBitsPerSecond,
                RequestsPerSecond = HostPerformanceMonitor.RequestsPerSecond,
                SentBitsPerSecond = HostPerformanceMonitor.SentBitsPerSecond,
                SuggestionsAccepted = stats.SuggestionsAccepted.Counter,
                TotalProfileUpdates = stats.ProfilesUpdated.Counter,
                UniqueProfileUpdates = stats.ProfilesUpdated.UniqueCounter,
                MessagesRecipients = stats.MessagesSent.UniqueCounter,
                IdentitiesDeclined = stats.IdentitiesDeclined.Counter,
                PlatformIdentitiesDeclinedByType = stats.PlatformIdentitiesDeclinedByType.Values,
                PlatformIdentitiesCreatedByType = stats.PlatformIdentitiesCreatedByType.Values,
                CurrentGames = stats.CurrentGames.Values,
                VoiceInvitationsSent = stats.VoiceInvitationsSent.Counter,
                VoiceInvitationsDeclined = stats.VoiceInvitationsDeclined.Counter,
                FriendSyncsByResult = stats.FriendSyncsByResult.Values,

                // Group Stats
                GroupVoiceInvitationsSent = stats.GroupVoiceInvitationsSent.Counter,
                GroupVoiceInvitationsDeclined = stats.GroupVoiceInvitationsDeclined.Counter,
                NewGroupMessages = stats.GroupMessagesSent.LifetimeCounter,
                GroupsCreatedByType = stats.GroupsCreatedByType.Values,
                NewGroupMessagesByPlatform = stats.GroupMessagesSentByPlatform.Values,

                // Queue Stats
                QueueStats = stats.QueueStats,

                // Image Service Stats
                ImagesUploaded = stats.ImagesUploaded.Counter,

            };

            return periodicStats;
        }

        private static void PublishRegionStats()
        {
            FriendsRegionStats[] pendingStats = null;
            lock (_syncRoot)
            {
                if (_regionalStats.Any())
                {
                    pendingStats = _regionalStats.ToArray();
                    _regionalStats.Clear();
                }
            }

            if (pendingStats == null || pendingStats.Length == 0)
            {
                return;
            }

            foreach (var stats in pendingStats)
            {
                try
                {
                    Logger.Info("Publishing regional stats for region: " + stats.RegionID);

                    var serialized = JsonConvert.SerializeObject(stats);
                    var resp = FriendsServiceClients.Instance.Stats.Regional(new StatsPayload { HostName = Environment.MachineName, HostTypeID = _hostTypeID, HostTypeName = _hostTypeName, RegionID = stats.RegionID, StatsJson = serialized });

                    if (!resp.Success)
                    {
                        Logger.Error("Failed to submit regional stats! They will be re-added to the pending list.");

                        lock (_syncRoot)
                        {
                            _regionalStats.Add(stats);
                        }
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to submit regional stats due to a general exception");
                }

            }

            Logger.Debug("Publishing region stats.");

        }

        /// <summary>
        /// Publish the realtime stats to the central service (CPU, Memory, Connections, etc).
        /// </summary>
        private static void ClearRealtimeStats()
        {
            var serialized = JsonConvert.SerializeObject(new FriendsHostStats());
            TrySubmitRealtimeStats(serialized);
        }

        /// <summary>
        /// Publish the realtime stats to the central service (CPU, Memory, Connections, etc).
        /// </summary>
        private static void PublishRealtimeStats()
        {
            FriendsStats currentStats;
            FriendsStats previousStats;

            // Get a snapshot of the current and previous stats
            lock (_syncRoot)
            {
                currentStats = Current;
                previousStats = _previousStats;
            }

            // Get the amount of mins away from the end of this interval, to determine waiting:
            var elapsed = DateTime.UtcNow - GetQuarter();

            var weightedStats = CreatePeriodicStats(currentStats);

            if (weightedStats.NewMessages > 0)
            {
                weightedStats.MessagesPerMinute = currentStats.MessagesSent.MinutelyCounter / currentStats.MessagesSent.ElapsedMinutes;
                weightedStats.MessagesPerSecond = currentStats.MessagesSent.SecondlyCounter / currentStats.MessagesSent.ElapsedSeconds;
            }

            if (weightedStats.NewGroupMessages > 0)
            {
                weightedStats.GroupMessagesPerMinute = currentStats.GroupMessagesSent.MinutelyCounter / currentStats.GroupMessagesSent.ElapsedMinutes;
                weightedStats.GroupMessagesPerSecond = currentStats.GroupMessagesSent.SecondlyCounter / currentStats.GroupMessagesSent.ElapsedSeconds;
            }

            if (weightedStats.NewPrivateGroupMessages > 0)
            {
                weightedStats.PrivateGroupMessagesPerMinute = currentStats.PrivateGroupMessagesSent.MinutelyCounter / currentStats.PrivateGroupMessagesSent.ElapsedMinutes;
                weightedStats.PrivateGroupMessagesPerSecond = currentStats.PrivateGroupMessagesSent.SecondlyCounter / currentStats.PrivateGroupMessagesSent.ElapsedSeconds;
            }


            if (previousStats != null)
            {
                // Determine how much to wait the prior period
                var priorWeight = 1 - (elapsed.TotalMinutes / 15);
                var previousRealtimeStats = CreatePeriodicStats(previousStats);
                weightedStats.ApplyWeightedStats(previousRealtimeStats, priorWeight);
            }

            var serialized = JsonConvert.SerializeObject(weightedStats);
            TrySubmitRealtimeStats(serialized);
        }

        public static void AddRegionStats(FriendsRegionStats stats)
        {
            if (!_isInitialized)
            {
                return;
            }

            lock (_syncRoot)
            {
                _regionalStats.Add(stats);
            }
        }
      
        public static DateTime GetQuarter()
        {
            return new DateTime(DateTime.UtcNow.Year, DateTime.UtcNow.Month, DateTime.UtcNow.Day, DateTime.UtcNow.Hour, GetMinute(), 0, DateTimeKind.Utc);
        }

        private static int GetMinute()
        {
            if (DateTime.UtcNow.Minute < 15)
            {
                return 0;
            }

            if (DateTime.UtcNow.Minute < 30)
            {
                return 15;
            }

            if (DateTime.UtcNow.Minute < 45)
            {
                return 30;
            }

            return 45;
        }

    }

    
}
