﻿using Curse.Logging;
using System;
using System.Diagnostics;
using System.Linq;
using System.Net.NetworkInformation;
using System.Threading;

namespace Curse.Friends.Statistics
{
    [Flags]
    public enum HostCounter
    {        
        Network = 1,
        Processor = 2,
        Requests = 4,
        Default = Network | Processor,
        All = Network | Processor | Requests
    }

    public static class HostPerformanceMonitor
    {
        private static readonly object SyncRoot = new object();
        private static bool _hasInitialized = false;
        private static string _interfaceName = null;
        private static Process _currentProcess;
        private static PerformanceCounter _outgoingNetworkCounter;
        private static PerformanceCounter _incomingNetworkCounter;
        private static PerformanceCounter _processorCounter;
        private static PerformanceCounter _requestsCounter;

        public static void Initialize(HostCounter counters = HostCounter.Default)
        {
            lock (SyncRoot)
            {
                if (_hasInitialized)
                {
                    return;
                }

                _currentProcess = Process.GetCurrentProcess();

                if (counters.HasFlag(HostCounter.Network))
                {
                    var networkInterfaces = NetworkInterface.GetAllNetworkInterfaces();
                    var primaryNetworkInterface =
                        networkInterfaces.FirstOrDefault(
                            p =>
                                p.OperationalStatus == OperationalStatus.Up &&
                                (p.NetworkInterfaceType == NetworkInterfaceType.Ethernet ||
                                 p.NetworkInterfaceType == NetworkInterfaceType.Wireless80211));

                    if (primaryNetworkInterface == null)
                    {
                        Logger.Info(
                            "Unable to determine the primary network interface! Bandwidth monitoring will not occur!");
                        return;
                    }

                    _interfaceName = primaryNetworkInterface.Description.Replace("#", "_")
                        .Replace("(", "[")
                        .Replace(")", "]");
                    _outgoingNetworkCounter = new PerformanceCounter("Network Interface", "Bytes Sent/sec",
                        _interfaceName);
                    _incomingNetworkCounter = new PerformanceCounter("Network Interface", "Bytes Received/sec",
                        _interfaceName);
                }

                if (counters.HasFlag(HostCounter.Processor))
                {
                    _processorCounter = new PerformanceCounter("Processor Information", "% Processor Time", "_Total");
                }

                if (counters.HasFlag(HostCounter.Requests))
                {
                    _requestsCounter = new PerformanceCounter("W3SVC_W3WP", "Requests / Sec", "_Total");
                }

                new Thread(PollThread) { IsBackground = true }.Start();

                _hasInitialized = true;
            }

        }


        public static int ReceivedBitsPerSecond
        {
            get;
            private set;
        }

        public static int SentBitsPerSecond
        {
            get;
            private set;
        }

        public static double ProcessorUtilization
        {
            get;
            private set;
        }

        public static int MemoryUsageKilobytes
        {
            get;
            private set;
        }

        public static double RequestsPerSecond
        {
            get;
            private set;
        }

        private static void PollThread()
        {
            var hasSucceeded = false;
            while (true)
            {
                try
                {
                    ReceivedBitsPerSecond = GetIncomingNetworkUtilization();
                    SentBitsPerSecond = GetOutgoingNetworkUtilization();
                    ProcessorUtilization = GetProcessorUtilization();
                    MemoryUsageKilobytes = GetMemoryUsageKilobytes();
                    RequestsPerSecond = GetRequestsPerSecond();

                    if (!hasSucceeded)
                    {
                        hasSucceeded = true;
                        Logger.Info("Completed first host performance profile.");
                    }

                }
                catch (Exception ex)
                {
                                        
                    if (!hasSucceeded)
                    {
                        var networkInterfaces = NetworkInterface.GetAllNetworkInterfaces().Select(p => new { p.Name, p.Description, p.NetworkInterfaceType}).ToArray();
                        Logger.Error(ex, "A performance counter has failed against " + _interfaceName, networkInterfaces);

                        return;
                    }
                    
                    Logger.Warn(ex, "Unable to get network utilization for interface: " + _interfaceName);
                    
                }

                Thread.Sleep(1000);
            }
        }


        private static int GetMemoryUsageKilobytes()
        {

            return (int)(_currentProcess.PrivateMemorySize64 / 1024);
        }

        private static int GetIncomingNetworkUtilization()
        {
            if (_incomingNetworkCounter == null)
            {
                return 0;
            }


            _incomingNetworkCounter.NextValue();
            Thread.Sleep(1000);
            var bytesPerSecond = _incomingNetworkCounter.NextValue();
            double utilization = 8 * bytesPerSecond;
            return (int)utilization;
        }

        private static int GetOutgoingNetworkUtilization()
        {
            if (_outgoingNetworkCounter == null)
            {
                return 0;
            }

            _outgoingNetworkCounter.NextValue();
            Thread.Sleep(1000);
            var bytesPerSecond = _outgoingNetworkCounter.NextValue();
            double utilization = 8 * bytesPerSecond;
            return (int)utilization;
        }

        private static float GetProcessorUtilization()
        {
            if (_processorCounter == null)
            {
                return 0;
            }

            _processorCounter.NextValue();
            Thread.Sleep(1000);
            var utilization = _processorCounter.NextValue();
            return utilization;
        }

        private static float GetRequestsPerSecond()
        {

            if (_requestsCounter == null)
            {
                return 0;
            }

            _requestsCounter.NextValue();
            Thread.Sleep(1000);
            var val = _requestsCounter.NextValue();
            return val;
        }
    }
}
